import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link, useLocation } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Button,
  Tooltip,
  Chip
} from '@mui/material';
import StorageIcon from '@mui/icons-material/Storage';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import DnsIcon from '@mui/icons-material/Dns';
import VisibilityIcon from '@mui/icons-material/Visibility';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography)(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const VectorDBCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const VectorDBListContainer = styled(List)(({ theme }) => ({
  padding: 0,
}));

const VectorDBItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  }
}));

const VectorDBLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  display: 'flex',
  alignItems: 'center',
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const VectorDBIcon = styled(StorageIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const HostChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.05)',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(2),
  fontSize: '0.75rem',
  height: 24,
}));

const ViewButton = styled(Button)(({ theme }) => ({
  minWidth: 'unset',
  padding: theme.spacing(0.5, 1),
  marginLeft: theme.spacing(1),
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const VectorDBList = ({ isEmbedded = false }) => {
    const [vectorDBs, setVectorDBs] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const location = useLocation();

    useEffect(() => {
        const fetchVectorDBs = async () => {
            try {
                const response = await axios.get(`${BASE_URL}/vectordb/`);
                if (response.status === 200) {
                    setVectorDBs(response.data);
                } else {
                    throw new Error('Failed to fetch VectorDBs');
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchVectorDBs();
    }, []);

    return (
        <ContentContainer>
            {!isEmbedded && location.pathname !== "/vectordbs/home" && (
                <BackButton 
                    component={Link} 
                    to="/vectordbs/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to VectorDB Home
                </BackButton>
            )}
            
            <SectionTitle variant="h4" embedded={isEmbedded}>
                VectorDB List
                <Tooltip title="List of vector databases in your system">
                    <InfoIcon />
                </Tooltip>
            </SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error.message} />}
            
            {!loading && !error && (
                <VectorDBCard>
                    {vectorDBs.length === 0 ? (
                        <EmptyState>
                            <Typography variant="body1">No Vector Databases available.</Typography>
                        </EmptyState>
                    ) : (
                        <VectorDBListContainer>
                            {vectorDBs.map(vectorDB => (
                                <VectorDBItem key={vectorDB.id}>
                                    <ListItemIcon style={{ minWidth: 'auto', marginRight: '8px' }}>
                                        <VectorDBIcon />
                                    </ListItemIcon>
                                    <ListItemText 
                                        primary={
                                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                                <VectorDBLink to={`/vectordbs/${vectorDB.id}`}>
                                                    {vectorDB.name || `VectorDB ${vectorDB.id}`}
                                                </VectorDBLink>
                                                {vectorDB.host && (
                                                    <HostChip 
                                                        icon={<DnsIcon fontSize="small" />}
                                                        label={`${vectorDB.host}:${vectorDB.port || 'N/A'}`}
                                                        size="small"
                                                    />
                                                )}
                                                <Tooltip title="View Details">
                                                    <ViewButton
                                                        component={Link} 
                                                        to={`/vectordbs/${vectorDB.id}`}
                                                        size="small"
                                                        startIcon={<VisibilityIcon fontSize="small" />}
                                                    >
                                                        View
                                                    </ViewButton>
                                                </Tooltip>
                                            </Box>
                                        }
                                        secondary={vectorDB.description && (
                                            <Typography variant="body2" color="text.secondary" sx={{ mt: 0.5, fontSize: '0.85rem' }}>
                                                {vectorDB.description}
                                            </Typography>
                                        )}
                                    />
                                </VectorDBItem>
                            ))}
                        </VectorDBListContainer>
                    )}
                </VectorDBCard>
            )}
        </ContentContainer>
    );
};

export default VectorDBList;
