import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Button,
  Grid,
  Divider,
  Chip,
  Tooltip,
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import StorageIcon from '@mui/icons-material/Storage';
import DnsIcon from '@mui/icons-material/Dns';
import CalendarTodayIcon from '@mui/icons-material/CalendarToday';
import DescriptionIcon from '@mui/icons-material/Description';
import AccessTimeIcon from '@mui/icons-material/AccessTime';
import HelpOutlineIcon from '@mui/icons-material/HelpOutline';
import OpenInNewIcon from '@mui/icons-material/OpenInNew';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const OpenAPIButton = styled(Button)(({ theme }) => ({
  marginLeft: theme.spacing(1),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: '#fff',
  backgroundColor: theme.palette.primary.main,
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
  },
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  },
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionLabel = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
  marginBottom: theme.spacing(1),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
  },
}));

const StyledList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const DetailItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(1, 0),
  '& svg': {
    color: theme.palette.primary.light,
  },
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  minWidth: '150px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
}));

const MetadataCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 8,
  padding: theme.spacing(2),
  height: '100%',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  transition: 'all 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
    boxShadow: '0 6px 16px rgba(0, 0, 0, 0.2)',
  },
}));

const MetadataLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontSize: '0.875rem',
  marginBottom: theme.spacing(0.5),
}));

const MetadataValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '1.25rem',
}));

const ButtonContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
}));

const EmptyState = styled(Typography)(({ theme }) => ({
  fontStyle: 'italic',
  color: theme.palette.text.secondary,
}));

const formatDate = dateString => {
  if (!dateString) return 'N/A';

  const date = new Date(dateString);
  if (isNaN(date.getTime())) return dateString;

  return new Intl.DateTimeFormat('en-US', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
  }).format(date);
};

const VectorDBDetails = () => {
  const [vectorDB, setVectorDB] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const { vectordb_id } = useParams();

  useEffect(() => {
    const fetchVectorDBDetails = async () => {
      try {
        const response = await axios.get(`${BASE_URL}/vectordb/${vectordb_id}`);
        setVectorDB(response.data);
      } catch (err) {
        setError(err);
      } finally {
        setLoading(false);
      }
    };

    fetchVectorDBDetails();
  }, [vectordb_id]);

  return (
    <ContentContainer>
      <ButtonContainer>
        <BackButton
          component={Link}
          to="/vectordbs/home"
          startIcon={<ArrowBackIcon />}
          size="small"
        >
          Back to VectorDB Home
        </BackButton>

        <BackButton component={Link} to="/vectordbs" startIcon={<ArrowBackIcon />} size="small">
          Back to VectorDB List
        </BackButton>

        {vectorDB && vectorDB.host && !!vectorDB.port && (
          <OpenAPIButton
            startIcon={<OpenInNewIcon />}
            size="small"
            component="a"
            href={`http://${vectorDB.host}:${vectorDB.port}/docs`}
            target="_blank"
            rel="noopener noreferrer"
          >
            Open API
          </OpenAPIButton>
        )}
      </ButtonContainer>

      <SectionTitle variant="h4">VectorDB Details</SectionTitle>

      {loading && <Spinner />}
      {error && <ErrorComponent message={error?.message || 'An error occurred'} />}

      {!loading && !error && !vectorDB && <EmptyState>VectorDB not found</EmptyState>}

      {!loading && !error && vectorDB && (
        <>
          <DetailCard>
            <DetailSubtitle variant="h5">
              <StorageIcon />
              {vectorDB.name || `VectorDB ${vectordb_id}`}
            </DetailSubtitle>

            <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />

            <Grid container spacing={3} sx={{ mb: 3 }}>
              <Grid item xs={12} sm={6} md={3}>
                <MetadataCard>
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <DnsIcon sx={{ mr: 1, color: 'primary.main' }} />
                    <MetadataLabel>Host</MetadataLabel>
                  </Box>
                  <MetadataValue>{vectorDB.host || 'N/A'}</MetadataValue>
                </MetadataCard>
              </Grid>

              <Grid item xs={12} sm={6} md={3}>
                <MetadataCard>
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <DnsIcon sx={{ mr: 1, color: 'primary.main' }} />
                    <MetadataLabel>Port</MetadataLabel>
                  </Box>
                  <MetadataValue>{vectorDB.port || 'N/A'}</MetadataValue>
                </MetadataCard>
              </Grid>

              <Grid item xs={12} sm={6} md={3}>
                <MetadataCard>
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <StorageIcon sx={{ mr: 1, color: 'primary.main' }} />
                    <MetadataLabel>Type</MetadataLabel>
                    <Tooltip title="The type of vector database (e.g., FAISS, Chroma, Pinecone)">
                      <HelpOutlineIcon
                        sx={{ ml: 0.5, fontSize: '16px', color: 'text.secondary' }}
                      />
                    </Tooltip>
                  </Box>
                  <MetadataValue>{vectorDB.type || 'Standard'}</MetadataValue>
                </MetadataCard>
              </Grid>

              <Grid item xs={12} sm={6} md={3}>
                <MetadataCard>
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <CalendarTodayIcon sx={{ mr: 1, color: 'primary.main' }} />
                    <MetadataLabel>Created</MetadataLabel>
                  </Box>
                  <MetadataValue sx={{ fontSize: '1rem' }}>
                    {vectorDB.created_at ? formatDate(vectorDB.created_at).split(',')[0] : 'N/A'}
                  </MetadataValue>
                </MetadataCard>
              </Grid>
            </Grid>

            {vectorDB.description && (
              <InfoSection>
                <SectionLabel>
                  <DescriptionIcon />
                  Description
                </SectionLabel>
                <Box
                  sx={{ px: 2, py: 1.5, backgroundColor: 'rgba(0, 0, 0, 0.1)', borderRadius: 1 }}
                >
                  <Typography variant="body1" color="text.primary">
                    {vectorDB.description}
                  </Typography>
                </Box>
              </InfoSection>
            )}

            <InfoSection>
              <SectionLabel>
                <AccessTimeIcon />
                Details
              </SectionLabel>
              <StyledList>
                {vectorDB.created_at && (
                  <DetailItem>
                    <ListItemIcon>
                      <CalendarTodayIcon />
                    </ListItemIcon>
                    <DetailLabel>Created At:</DetailLabel>
                    <DetailValue>{formatDate(vectorDB.created_at)}</DetailValue>
                  </DetailItem>
                )}

                {/* Add more details here as they become available in the API */}
              </StyledList>
            </InfoSection>
          </DetailCard>
        </>
      )}
    </ContentContainer>
  );
};

export default VectorDBDetails;
