import React, { useState } from 'react';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import { 
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Button,
  Grid,
  Typography,
  IconButton,
  Box,
  Divider
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import SaveIcon from '@mui/icons-material/Save';
import StorageIcon from '@mui/icons-material/Storage';
import { BASE_URL } from '../../const';

const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    backgroundColor: theme.palette.background.paper,
    borderRadius: '12px',
    boxShadow: '0 8px 32px rgba(0, 0, 0, 0.3)',
    border: '1px solid rgba(255, 255, 255, 0.08)',
    padding: theme.spacing(2),
    minWidth: '400px',
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between',
  padding: theme.spacing(2, 3),
  color: theme.palette.primary.main,
  fontWeight: 600,
  '& svg': {
    marginRight: theme.spacing(1),
  },
}));

const CloseButton = styled(IconButton)(({ theme }) => ({
  color: theme.palette.text.secondary,
  '&:hover': {
    color: theme.palette.error.main,
    backgroundColor: 'rgba(255, 0, 0, 0.1)',
  },
}));

const StyledTextField = styled(TextField)(({ theme }) => ({
  margin: theme.spacing(1, 0),
  '& .MuiOutlinedInput-root': {
    '& fieldset': {
      borderColor: 'rgba(255, 255, 255, 0.15)',
    },
    '&:hover fieldset': {
      borderColor: 'rgba(255, 255, 255, 0.25)',
    },
    '&.Mui-focused fieldset': {
      borderColor: `${theme.palette.primary.main} !important`,
      borderWidth: '1px !important',
    },
    // Fix for autofill blue background
    '& input:-webkit-autofill, & input:-webkit-autofill:hover, & input:-webkit-autofill:focus, & input:-webkit-autofill:active': {
      WebkitBoxShadow: '0 0 0 30px #161b22 inset !important', // Match your background color
      WebkitTextFillColor: '#ffffff !important', // Match your text color
      caretColor: '#ffffff !important',
      borderRadius: 'inherit',
    },
  },
  '& .MuiInputLabel-root': {
    color: theme.palette.text.secondary,
  },
  '& .MuiInputBase-input': {
    color: theme.palette.text.primary,
    position: 'relative',
    '&:focus': {
      boxShadow: 'none',
    },
  },
  // Ensure there's only one border
  '& .MuiOutlinedInput-notchedOutline': {
    borderWidth: '1px !important',
  },
}));

const ActionButton = styled(Button)(({ theme, cancel }) => ({
  textTransform: 'none',
  padding: theme.spacing(1, 3),
  fontWeight: 500,
  color: cancel ? theme.palette.text.primary : '#fff',
  backgroundColor: cancel ? 'rgba(255, 255, 255, 0.05)' : theme.palette.primary.main,
  border: cancel ? '1px solid rgba(255, 255, 255, 0.15)' : 'none',
  '&:hover': {
    backgroundColor: cancel ? 'rgba(255, 255, 255, 0.1)' : theme.palette.primary.dark,
  },
}));

const FormLabel = styled(Typography)(({ theme }) => ({
  fontWeight: 500,
  fontSize: '0.875rem',
  color: theme.palette.text.secondary,
  marginBottom: theme.spacing(0.5),
}));

const AddVectorDBModal = ({ closeModal, showModal, addVectorDB }) => {
    const [vectorDBData, setVectorDBData] = useState({
        name: '',
        host: '',
        port: '',
        description: '',
        type: 'Standard'
    });

    const [errors, setErrors] = useState({});

    const validateForm = () => {
        const newErrors = {};
        
        if (!vectorDBData.name.trim()) {
            newErrors.name = 'Name is required';
        }
        
        if (!vectorDBData.host.trim()) {
            newErrors.host = 'Host is required';
        }
        
        if (!vectorDBData.port) {
            newErrors.port = 'Port is required';
        } else if (isNaN(vectorDBData.port) || parseInt(vectorDBData.port) <= 0) {
            newErrors.port = 'Port must be a positive number';
        }
        
        setErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleAddVectorDB = async () => {
        if (!validateForm()) return;
        
        try {
            const response = await axios.post(`${BASE_URL}/vectordb/vectordb/`, vectorDBData);
            if (response.status === 200) {
                if (typeof addVectorDB === 'function') {
                    addVectorDB(response.data);
                }
                closeModal();
            } else {
                // Handle non-successful response
                setErrors({ submit: 'Failed to add VectorDB' });
            }
        } catch (err) {
            console.error('Add VectorDB error:', err);
            setErrors({ submit: err.message || 'An error occurred' });
        }
    };

    const handleInputChange = (event) => {
        const { name, value } = event.target;
        setVectorDBData({
            ...vectorDBData,
            [name]: value
        });
        
        // Clear error for this field when user types
        if (errors[name]) {
            setErrors({
                ...errors,
                [name]: undefined
            });
        }
    };

    return (
        <StyledDialog open={showModal} onClose={closeModal} maxWidth="sm" fullWidth>
            <StyledDialogTitle>
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                    <StorageIcon />
                    Add New Vector Database
                </Box>
                <CloseButton onClick={closeModal}>
                    <CloseIcon />
                </CloseButton>
            </StyledDialogTitle>
            
            <Divider />
            
            <DialogContent>
                <Grid container spacing={2}>
                    <Grid item xs={12}>
                        <FormLabel>Name</FormLabel>
                        <StyledTextField
                            fullWidth
                            placeholder="Enter VectorDB name"
                            name="name"
                            value={vectorDBData.name}
                            onChange={handleInputChange}
                            error={!!errors.name}
                            helperText={errors.name}
                            variant="outlined"
                        />
                    </Grid>
                    
                    <Grid item xs={12} sm={8}>
                        <FormLabel>Host</FormLabel>
                        <StyledTextField
                            fullWidth
                            placeholder="localhost or IP address"
                            name="host"
                            value={vectorDBData.host}
                            onChange={handleInputChange}
                            error={!!errors.host}
                            helperText={errors.host}
                            variant="outlined"
                        />
                    </Grid>
                    
                    <Grid item xs={12} sm={4}>
                        <FormLabel>Port</FormLabel>
                        <StyledTextField
                            fullWidth
                            placeholder="8000"
                            name="port"
                            value={vectorDBData.port}
                            onChange={handleInputChange}
                            error={!!errors.port}
                            helperText={errors.port}
                            variant="outlined"
                            type="number"
                        />
                    </Grid>
                    
                    <Grid item xs={12}>
                        <FormLabel>Type</FormLabel>
                        <StyledTextField
                            fullWidth
                            placeholder="Vector database type (e.g., FAISS, Chroma)"
                            name="type"
                            value={vectorDBData.type}
                            onChange={handleInputChange}
                            variant="outlined"
                        />
                    </Grid>
                    
                    <Grid item xs={12}>
                        <FormLabel>Description</FormLabel>
                        <StyledTextField
                            fullWidth
                            placeholder="Brief description of this vector database"
                            name="description"
                            value={vectorDBData.description}
                            onChange={handleInputChange}
                            variant="outlined"
                            multiline
                            rows={3}
                        />
                    </Grid>
                </Grid>
                
                {errors.submit && (
                    <Typography color="error" variant="body2" sx={{ mt: 2 }}>
                        {errors.submit}
                    </Typography>
                )}
            </DialogContent>
            
            <DialogActions sx={{ p: 2 }}>
                <ActionButton cancel onClick={closeModal}>
                    Cancel
                </ActionButton>
                <ActionButton 
                    onClick={handleAddVectorDB}
                    startIcon={<SaveIcon />}
                >
                    Create VectorDB
                </ActionButton>
            </DialogActions>
        </StyledDialog>
    );
};

export default AddVectorDBModal;
