import React from 'react';
import { render, screen, fireEvent, waitFor } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import ToolDeploymentDetail from '../ToolDeploymentDetail';
import theme from '../../../Theme';

// Mock the api module completely
jest.mock('../../../utils/api', () => ({
  __esModule: true,
  default: {
    get: jest.fn(),
  },
}));

// Import api after the mock
const api = require('../../../utils/api').default;

describe('ToolDeploymentDetail', () => {
  const mockOnClose = jest.fn();
  const mockOnDeploymentStopped = jest.fn();

  const mockDeploymentData = {
    name: 'mcp-test-tool',
    url: 'http://localhost:3001',
    status: 'DEPLOYED',
    created_at: new Date().toISOString(),
    deployed_at: new Date().toISOString(),
    env_vars: {
      API_KEY: 'test-key',
    },
  };

  beforeEach(() => {
    jest.clearAllMocks();
    api.get.mockResolvedValue({ data: mockDeploymentData });
  });

  const renderComponent = (props = {}) => {
    const defaultProps = {
      deploymentId: 'test-deployment-id',
      isOpen: true,
      onClose: mockOnClose,
      onDeploymentStopped: mockOnDeploymentStopped,
      ...props,
    };

    return render(
      <ThemeProvider theme={theme}>
        <ToolDeploymentDetail {...defaultProps} />
      </ThemeProvider>
    );
  };

  it('renders the dialog when open', async () => {
    renderComponent();

    await waitFor(() => {
      expect(screen.getByText('Tool Server Details')).toBeInTheDocument();
    });
  });

  it('does not render when isOpen is false', () => {
    renderComponent({ isOpen: false });

    expect(screen.queryByText('Tool Server Details')).not.toBeInTheDocument();
  });

  describe('Close button functionality', () => {
    it('renders close button with proper attributes', async () => {
      renderComponent();

      await waitFor(() => {
        const closeButtons = screen.getAllByRole('button');
        const closeButton = closeButtons.find(button =>
          button.querySelector('[data-testid="CloseIcon"]')
        );
        expect(closeButton).toBeInTheDocument();
        expect(closeButton).toHaveClass('MuiIconButton-sizeSmall');
      });
    });

    it('calls onClose when close button is clicked', async () => {
      renderComponent();

      await waitFor(() => {
        const closeButtons = screen.getAllByRole('button');
        const closeButton = closeButtons.find(button =>
          button.querySelector('[data-testid="CloseIcon"]')
        );
        fireEvent.click(closeButton);
        expect(mockOnClose).toHaveBeenCalledTimes(1);
      });
    });

    it('close button is positioned correctly in the dialog title', async () => {
      renderComponent();

      await waitFor(() => {
        const dialogTitle = screen.getByText('Tool Server Details').closest('h2');
        // Check that the parent has display flex
        const computedStyle = window.getComputedStyle(dialogTitle);
        expect(computedStyle.display).toBe('flex');
      });
    });

    it('close button has correct size attribute', async () => {
      renderComponent();

      await waitFor(() => {
        const closeButtons = screen.getAllByRole('button');
        const closeButton = closeButtons.find(button =>
          button.querySelector('[data-testid="CloseIcon"]')
        );
        expect(closeButton.className).toMatch(/MuiIconButton-sizeSmall/);
      });
    });
  });

  it('shows loading spinner while fetching data', async () => {
    let resolvePromise;
    const promise = new Promise(resolve => {
      resolvePromise = resolve;
    });
    api.get.mockReturnValueOnce(promise);

    renderComponent();

    expect(screen.getByRole('progressbar')).toBeInTheDocument();

    // Clean up by resolving the promise
    resolvePromise({ data: mockDeploymentData });
    await waitFor(() => {
      expect(screen.queryByRole('progressbar')).not.toBeInTheDocument();
    });
  });

  it('displays deployment details when data is loaded', async () => {
    renderComponent();

    await waitFor(() => {
      expect(screen.getByText('mcp-test-tool')).toBeInTheDocument();
      // URL is in an input field, so we need to check for the input value
      const urlInput = screen.getByDisplayValue('http://localhost:3001');
      expect(urlInput).toBeInTheDocument();
      expect(screen.getByText('DEPLOYED')).toBeInTheDocument();
    });
  });

  it('handles error state gracefully', async () => {
    const errorMessage = 'Failed to load deployment';
    api.get.mockRejectedValueOnce(new Error(errorMessage));

    renderComponent();

    await waitFor(() => {
      // ErrorComponent displays "Error: " + message
      expect(screen.getByText(`Error: ${errorMessage}`)).toBeInTheDocument();
    });
  });
});
