import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Grid,
  Tooltip,
  Alert,
  Chip,
  Switch,
  FormControlLabel,
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import LanguageIcon from '@mui/icons-material/Language';
import api from '../../utils/api';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import ToolTemplateCard from './ToolTemplateCard';
import ToolDeployModal from './ToolDeployModal';

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 600,
  marginTop: theme.spacing(4),
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const FilterSection = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
  flexWrap: 'wrap',
}));

const CategoryChip = styled(Chip)(({ theme }) => ({
  cursor: 'pointer',
  transition: 'all 0.2s',
  '&:hover': {
    backgroundColor: theme.palette.primary.main,
    color: theme.palette.primary.contrastText,
  },
  '&.MuiChip-filled': {
    backgroundColor: theme.palette.primary.main,
    color: theme.palette.primary.contrastText,
  },
}));

const InternetToggle = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  marginLeft: 'auto',
  gap: theme.spacing(1),
}));

// Category definitions
const CATEGORIES = {
  internet_data: 'Internet Data',
  model_tools: 'Model Tools',
  communication: 'Communication',
  data_transformation: 'Data Transformation',
  personal_productivity: 'Personal Productivity',
};

const ToolTemplateList = ({ onDeploymentChange, refreshTemplates }) => {
  const [templates, setTemplates] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedTemplate, setSelectedTemplate] = useState(null);
  const [showDeployModal, setShowDeployModal] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [includeInternetTools, setIncludeInternetTools] = useState(true);

  useEffect(() => {
    fetchTemplates();
  }, [refreshTemplates]);

  const fetchTemplates = async () => {
    setLoading(true);
    try {
      const response = await api.get('/tool/templates');
      setTemplates(response.data);
    } catch (err) {
      console.error('Error fetching tool templates:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  const handleDeployClick = template => {
    setSelectedTemplate(template);
    setShowDeployModal(true);
  };

  const handleDeploySuccess = () => {
    setShowDeployModal(false);
    setSelectedTemplate(null);
    if (onDeploymentChange) {
      onDeploymentChange();
    }
  };

  // Get available categories based on templates
  const getAvailableCategories = () => {
    const categories = new Set();
    templates.forEach(template => {
      if (template.category) {
        categories.add(template.category);
      }
    });
    return Array.from(categories);
  };

  // Filter templates based on category and internet access
  const getFilteredTemplates = () => {
    return templates.filter(template => {
      // Category filter
      if (selectedCategory !== 'all' && template.category !== selectedCategory) {
        return false;
      }

      // Internet access filter
      if (!includeInternetTools && template.capabilities?.includes('internet_access')) {
        return false;
      }

      return true;
    });
  };

  const availableCategories = getAvailableCategories();
  const filteredTemplates = getFilteredTemplates();

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error?.message || 'Failed to load tool templates'} />;

  return (
    <Box sx={{ mb: 6 }} data-tour="tool-categories">
      <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }} data-tour="tool-integration">
        <SubTitle variant="h5">
          Available Tool Servers
          <Tooltip title="Pre-configured tool servers ready to deploy">
            <InfoIcon />
          </Tooltip>
        </SubTitle>
      </Box>

      <FilterSection data-tour="tool-filters">
        <CategoryChip
          label="All"
          variant={selectedCategory === 'all' ? 'filled' : 'outlined'}
          onClick={() => setSelectedCategory('all')}
        />
        {availableCategories.map(category => (
          <CategoryChip
            key={category}
            label={CATEGORIES[category] || category}
            variant={selectedCategory === category ? 'filled' : 'outlined'}
            onClick={() => setSelectedCategory(category)}
          />
        ))}

        <InternetToggle>
          <FormControlLabel
            control={
              <Switch
                checked={includeInternetTools}
                onChange={e => setIncludeInternetTools(e.target.checked)}
                color="primary"
              />
            }
            label={
              <Box sx={{ display: 'flex', alignItems: 'center' }}>
                <LanguageIcon sx={{ mr: 0.5, fontSize: 20 }} />
                Include tools that access the internet
              </Box>
            }
          />
          <Tooltip title="Non-internet tools may access the internet incidentally, but should function offline and do not use internet as part of their primary function.">
            <InfoIcon sx={{ fontSize: 20 }} />
          </Tooltip>
        </InternetToggle>
      </FilterSection>

      {filteredTemplates.length === 0 ? (
        <Alert severity="info" sx={{ maxWidth: 600 }}>
          {templates.length === 0
            ? 'No tool templates available yet. Use the "Deploy Custom Tool Server" button to deploy your own!'
            : 'No tools match the current filters. Try adjusting your filter settings.'}
        </Alert>
      ) : (
        <Grid container spacing={3}>
          {filteredTemplates.map((template, index) => (
            <Grid item xs={12} sm={6} md={4} lg={3} key={template.name || index}>
              <ToolTemplateCard
                template={template}
                onDeploy={() => handleDeployClick(template)}
                isFirstCard={index === 0}
              />
            </Grid>
          ))}
        </Grid>
      )}

      {showDeployModal && selectedTemplate && (
        <ToolDeployModal
          template={selectedTemplate}
          open={showDeployModal}
          onClose={() => setShowDeployModal(false)}
          onSuccess={handleDeploySuccess}
        />
      )}
    </Box>
  );
};

export default ToolTemplateList;
