import React, { useState } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import { 
  Card, 
  CardContent, 
  Typography, 
  Button, 
  Box, 
  Chip, 
  Tooltip, 
  IconButton,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Badge
} from '@mui/material';
import SearchIcon from '@mui/icons-material/Search';
import SmartToyIcon from '@mui/icons-material/SmartToy';
import DataObjectIcon from '@mui/icons-material/DataObject';
import WebIcon from '@mui/icons-material/Web';
import FolderIcon from '@mui/icons-material/Folder';
import CodeIcon from '@mui/icons-material/Code';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import PlayArrowIcon from '@mui/icons-material/PlayArrow';
import MoreVertIcon from '@mui/icons-material/MoreVert';
import UpdateIcon from '@mui/icons-material/Update';
import DeleteIcon from '@mui/icons-material/Delete';

// Icon mapping for different Tool types
const iconMap = {
  'mcp-websearch': SearchIcon,
  'mcp-math-tools': DataObjectIcon,
  'mcp-web-scraper': WebIcon,
  'mcp-file-manager': FolderIcon,
  'mcp-database-query': DataObjectIcon,
  'mcp-code-analyzer': CodeIcon,
  'mcp-kamiwaza-model': SmartToyIcon,
};

// Styled components
const StyledCard = styled(Card)(({ theme }) => ({
  height: '100%',
  display: 'flex',
  flexDirection: 'column',
  transition: 'all 0.3s ease',
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  borderRadius: 12,
  overflow: 'hidden',
  '&:hover': {
    transform: 'translateY(-4px)',
    boxShadow: '0 8px 24px rgba(0, 192, 127, 0.15)',
    borderColor: 'rgba(0, 192, 127, 0.3)',
  },
}));

const IconWrapper = styled(Box)(({ theme }) => ({
  width: 60,
  height: 60,
  borderRadius: 12,
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.1), rgba(0, 192, 160, 0.1))',
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  marginBottom: theme.spacing(2),
}));

const TemplateName = styled(Typography)(({ theme }) => ({
  fontWeight: 600,
  fontSize: '1.1rem',
  marginBottom: theme.spacing(1),
  color: theme.palette.primary.main,
}));

const TemplateDescription = styled(Typography)(({ theme }) => ({
  fontSize: '0.9rem',
  color: theme.palette.text.secondary,
  marginBottom: theme.spacing(2),
  minHeight: '2.7em',
  lineHeight: 1.5,
}));

const RequiredVarsBox = styled(Box)(({ theme }) => ({
  marginTop: 'auto',
  marginBottom: theme.spacing(2),
  paddingTop: theme.spacing(2),
}));

const RequiredVarChip = styled(Chip)(({ theme }) => ({
  fontSize: '0.75rem',
  height: 24,
  marginRight: theme.spacing(0.5),
  marginBottom: theme.spacing(0.5),
  backgroundColor: 'rgba(255, 152, 0, 0.1)',
  color: '#ff9800',
  borderColor: '#ff9800',
}));

const DeployButton = styled(Button)(({ theme }) => ({
  width: '100%',
  marginTop: 'auto',
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  fontWeight: 600,
  padding: theme.spacing(1.2),
  borderRadius: 8,
  transition: 'all 0.2s',
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
    transform: 'translateY(-1px)',
    boxShadow: '0 4px 8px rgba(0, 192, 127, 0.3)',
  },
}));

const VersionChip = styled(Chip)(({ theme }) => ({
  height: 20,
  fontSize: '0.7rem',
  backgroundColor: 'rgba(0, 192, 127, 0.15)',
  color: theme.palette.primary.main,
  fontWeight: 500,
}));

const ToolTemplateCard = ({ 
  template, 
  onDeploy, 
  onDelete, 
  onUpdate,
  hasUpdate = false,
  deployments = [],
  isFirstCard = false 
}) => {
  const [menuAnchor, setMenuAnchor] = useState(null);
  const IconComponent = iconMap[template.name] || SmartToyIcon;

  // Format template name for display
  const displayName =
    template.display_name ||
    template.name
      .replace('mcp-', '')
      .replace('tool-', '')
      .split('-')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1))
      .join(' ');

  return (
    <StyledCard data-tour={isFirstCard ? 'tool-card' : undefined}>
      <CardContent sx={{ display: 'flex', flexDirection: 'column', height: '100%', p: 3 }}>
        <IconWrapper>
          <IconComponent sx={{ fontSize: 32, color: '#00c07f' }} />
        </IconWrapper>

        <Box display="flex" alignItems="center" gap={1} mb={1}>
          <TemplateName variant="h6" sx={{ mb: 0 }}>{displayName}</TemplateName>
          <VersionChip label={`v${template.version || '1.0.0'}`} size="small" />
        </Box>

        <TemplateDescription>
          {template.description || 'Deploy this tool server to extend your AI capabilities'}
        </TemplateDescription>

        {deployments.length > 0 && (
          <Box display="flex" alignItems="center" gap={1} mb={2}>
            <PlayArrowIcon color="success" fontSize="small" />
            <Typography variant="caption" color="success.main">
              {deployments.length} active deployment{deployments.length > 1 ? 's' : ''}
            </Typography>
          </Box>
        )}

        {template.required_env_vars && template.required_env_vars.length > 0 && (
          <RequiredVarsBox>
            <Typography variant="caption" color="text.secondary" sx={{ display: 'block', mb: 1 }}>
              Required:
            </Typography>
            {template.required_env_vars.map(varName => (
              <Tooltip key={varName} title={`Environment variable: ${varName}`}>
                <RequiredVarChip label={varName} size="small" variant="outlined" />
              </Tooltip>
            ))}
          </RequiredVarsBox>
        )}

        <Box display="flex" gap={1} mt="auto">
          <DeployButton
            variant="contained"
            onClick={onDeploy}
            startIcon={<RocketLaunchIcon />}
            data-tour={isFirstCard ? 'tool-deploy' : undefined}
            sx={{ flex: 1 }}
          >
            Deploy
          </DeployButton>

          {(onUpdate || onDelete) && (
            <Tooltip title="More actions">
              <IconButton
                onClick={(e) => setMenuAnchor(e.currentTarget)}
                size="small"
                sx={{ 
                  border: '1px solid rgba(255, 255, 255, 0.23)',
                  color: 'rgba(255, 255, 255, 0.7)',
                  '&:hover': {
                    backgroundColor: 'rgba(255, 255, 255, 0.08)',
                  }
                }}
              >
                {hasUpdate ? (
                  <Badge badgeContent="!" color="warning">
                    <MoreVertIcon fontSize="small" />
                  </Badge>
                ) : (
                  <MoreVertIcon fontSize="small" />
                )}
              </IconButton>
            </Tooltip>
          )}
        </Box>

        {/* Actions Menu */}
        <Menu
          anchorEl={menuAnchor}
          open={Boolean(menuAnchor)}
          onClose={() => setMenuAnchor(null)}
          transformOrigin={{ horizontal: 'right', vertical: 'top' }}
          anchorOrigin={{ horizontal: 'right', vertical: 'bottom' }}
        >
          {onUpdate && (
            <MenuItem onClick={() => { onUpdate(); setMenuAnchor(null); }}>
              <ListItemIcon>
                {hasUpdate ? (
                  <Badge badgeContent="!" color="warning">
                    <UpdateIcon fontSize="small" />
                  </Badge>
                ) : (
                  <UpdateIcon fontSize="small" />
                )}
              </ListItemIcon>
              <ListItemText>
                {hasUpdate ? 'Update Available' : 'Check for Updates'}
              </ListItemText>
            </MenuItem>
          )}
          {onDelete && (
            <MenuItem 
              onClick={() => { onDelete(); setMenuAnchor(null); }}
              sx={{ color: 'error.main' }}
            >
              <ListItemIcon>
                <DeleteIcon fontSize="small" color="error" />
              </ListItemIcon>
              <ListItemText>Delete Template</ListItemText>
            </MenuItem>
          )}
        </Menu>
      </CardContent>
    </StyledCard>
  );
};

ToolTemplateCard.propTypes = {
  template: PropTypes.shape({
    id: PropTypes.string.isRequired,
    name: PropTypes.string.isRequired,
    display_name: PropTypes.string,
    description: PropTypes.string,
    required_env_vars: PropTypes.array,
    version: PropTypes.string,
  }).isRequired,
  onDeploy: PropTypes.func.isRequired,
  onDelete: PropTypes.func,
  onUpdate: PropTypes.func,
  hasUpdate: PropTypes.bool,
  deployments: PropTypes.array,
  isFirstCard: PropTypes.bool,
};

ToolTemplateCard.defaultProps = {
  isFirstCard: false,
  hasUpdate: false,
  deployments: [],
  onDelete: null,
  onUpdate: null,
};

export default ToolTemplateCard;
