import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import { Typography, Box, Container, Button, CircularProgress, Alert } from '@mui/material';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import DownloadIcon from '@mui/icons-material/Download';
import ToolTemplateList from './ToolTemplateList';
import ToolDeploymentList from './ToolDeploymentList';
import ToolDeployModal from './ToolDeployModal';
import api from '../../utils/api';
import { useTour } from '../../contexts/TourContext';

// Styled components
const PageContainer = styled(Container)(({ theme }) => ({
  paddingTop: theme.spacing(2),
  paddingBottom: theme.spacing(4),
  maxWidth: '1400px',
}));

const PageTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '2.5rem',
  fontWeight: 700,
  marginBottom: theme.spacing(1),
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  backgroundClip: 'text',
  textFillColor: 'transparent',
  WebkitBackgroundClip: 'text',
  WebkitTextFillColor: 'transparent',
}));

const PageSubtitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontSize: '1.1rem',
  marginBottom: theme.spacing(4),
}));

const DeployCustomButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  fontWeight: 600,
  padding: theme.spacing(1.5, 3),
  borderRadius: 8,
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
  },
}));

const ImportButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(2),
  marginRight: theme.spacing(2),
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  fontWeight: 600,
  padding: theme.spacing(1.5, 3),
  borderRadius: 8,
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
  },
  '&:disabled': {
    background: 'linear-gradient(90deg, #ccc, #ddd)',
    color: 'rgba(255, 255, 255, 0.7)',
  },
}));

const ToolShed = () => {
  const { startTour } = useTour();
  const [refreshDeployments, setRefreshDeployments] = useState(false);
  const [refreshTemplates, setRefreshTemplates] = useState(false);
  const [showCustomDeployModal, setShowCustomDeployModal] = useState(false);
  const [gardenStatus, setGardenStatus] = useState(null);
  const [importing, setImporting] = useState(false);
  const [importResult, setImportResult] = useState(null);

  useEffect(() => {
    checkGardenStatus();
    syncRemoteTools(); // Add remote sync on page load, like news
    
    // Start toolshed tour after a delay
    const timer = setTimeout(() => {
      startTour('toolshed');
    }, 1500);

    return () => clearTimeout(timer);
  }, [startTour]);

  const checkGardenStatus = async () => {
    try {
      const response = await api.get('/tool/garden/status');
      setGardenStatus(response.data);
    } catch (error) {
      console.error('Error checking tool garden status:', error);
      // Set a default status if the endpoint fails
      setGardenStatus({
        tool_servers_available: false,
        total_tool_servers: 0,
        imported_tool_servers: 0,
        missing_tool_servers: [],
      });
    }
  };

  const syncRemoteTools = async () => {
    try {
      // Call remote sync endpoint (has built-in 4-hour cache like news)
      const response = await api.post('/tool/remote/sync');
      
      if (response.data.success && response.data.imported_count > 0) {
        console.log(`Auto-synced ${response.data.imported_count} remote tools`);
        
        // Refresh garden status and templates after successful sync
        await checkGardenStatus();
        setRefreshTemplates(!refreshTemplates);
        handleDeploymentChange();
      }
    } catch (error) {
      // Silently fail like news - don't show user errors for background sync
      console.warn('Background remote tool sync failed:', error);
    }
  };

  const handleImportToolServers = async () => {
    setImporting(true);
    setImportResult(null);

    try {
      const response = await api.post('/tool/garden/import');
      setImportResult(response.data);

      if (response.data.success) {
        // Refresh garden status and deployments after successful import
        await checkGardenStatus();
        setRefreshTemplates(!refreshTemplates);
        handleDeploymentChange();
      }
    } catch (error) {
      console.error('Error importing tool servers:', error);
      setImportResult({
        success: false,
        errors: [error.response?.data?.detail || 'Failed to import tool servers'],
      });
    } finally {
      setImporting(false);
    }
  };

  const handleDeploymentChange = () => {
    // Toggle to trigger refresh in ToolDeploymentList
    setRefreshDeployments(!refreshDeployments);
  };

  // Determine if we should show the import button
  const shouldShowImportButton =
    gardenStatus &&
    gardenStatus.tool_servers_available &&
    gardenStatus.missing_tool_servers?.length > 0;

  return (
    <PageContainer data-tour="toolshed-intro">
      <Box sx={{ mb: 4 }}>
        <PageTitle variant="h1">ToolShed</PageTitle>
        <PageSubtitle variant="body1">Deploy Tool Servers</PageSubtitle>
      </Box>

      {/* Garden Import Section */}
      {shouldShowImportButton && (
        <Box sx={{ mb: 3 }}>
          <ImportButton
            startIcon={
              importing ? <CircularProgress size={20} color="inherit" /> : <DownloadIcon />
            }
            onClick={handleImportToolServers}
            disabled={importing}
          >
            {importing
              ? 'Importing...'
              : `Import ${gardenStatus.missing_tool_servers.length} Tool ${gardenStatus.missing_tool_servers.length === 1 ? 'Server' : 'Servers'}`}
          </ImportButton>

          {gardenStatus.missing_tool_servers?.length > 0 && (
            <Typography variant="caption" sx={{ display: 'block', color: 'text.secondary', mt: 1 }}>
              Available servers: {gardenStatus.missing_tool_servers.join(', ')}
            </Typography>
          )}
        </Box>
      )}

      {/* Import Result Alert */}
      {importResult && (
        <Alert
          severity={importResult.success ? 'success' : 'error'}
          sx={{ mb: 3 }}
          onClose={() => setImportResult(null)}
        >
          {importResult.success
            ? `Successfully imported ${importResult.imported_count} tool servers!`
            : `Import failed: ${importResult.errors?.join(', ') || 'Unknown error'}`}
        </Alert>
      )}

      {/* Deploy Custom Tool Server Button */}
      <DeployCustomButton
        startIcon={<RocketLaunchIcon />}
        onClick={() => setShowCustomDeployModal(true)}
      >
        Deploy Custom Tool Server
      </DeployCustomButton>

      {/* Available Tool Templates */}
      <ToolTemplateList
        onDeploymentChange={handleDeploymentChange}
        refreshTemplates={refreshTemplates}
      />

      {/* Deployed Tool Servers */}
      <ToolDeploymentList refreshDeployments={refreshDeployments} />

      {/* Custom Deploy Modal */}
      {showCustomDeployModal && (
        <ToolDeployModal
          open={showCustomDeployModal}
          onClose={() => setShowCustomDeployModal(false)}
          onSuccess={handleDeploymentChange}
          isCustom={true}
        />
      )}
    </PageContainer>
  );
};

export default ToolShed;
