import React, { useState, useEffect, useCallback } from 'react';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Tooltip,
  Chip,
  CircularProgress,
  IconButton,
  Snackbar,
  Alert
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import ContentCopyIcon from '@mui/icons-material/ContentCopy';
import StopIcon from '@mui/icons-material/Stop';
import VisibilityIcon from '@mui/icons-material/Visibility';
import api from '../../utils/api';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import ToolDeploymentDetail from './ToolDeploymentDetail';
import ConfirmModal from '../common/ConfirmModal';

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 600,
  marginTop: theme.spacing(4),
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 2),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  '&.empty-row': {
    height: 100,
    '& td': {
      textAlign: 'center',
      color: theme.palette.text.secondary,
    }
  }
}));

const ActionButton = styled(Button)(({ theme, color = 'primary' }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.8, 2),
  borderRadius: theme.shape.borderRadius,
  transition: 'all 0.2s',
  backgroundColor: color === 'primary' 
    ? theme.palette.primary.main 
    : theme.palette.error.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: color === 'primary' 
      ? theme.palette.primary.dark
      : theme.palette.error.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
  '&:disabled': {
    backgroundColor: theme.palette.action.disabledBackground,
  },
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 500,
  ...(status === 'DEPLOYED' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'STOPPED' && {
    backgroundColor: 'rgba(158, 158, 158, 0.1)',
    color: '#9e9e9e',
    borderColor: '#9e9e9e',
  }),
  ...(status === 'FAILED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    color: '#f44336',
    borderColor: '#f44336',
  }),
  ...(status === 'DEPLOYING' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    color: '#ff9800',
    borderColor: '#ff9800',
  }),
}));

const URLCell = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
  maxWidth: 400,
}));

const URLText = styled(Typography)(({ theme }) => ({
  fontFamily: 'monospace',
  fontSize: '0.85rem',
  color: theme.palette.primary.main,
  overflow: 'hidden',
  textOverflow: 'ellipsis',
  whiteSpace: 'nowrap',
  flex: 1,
}));

const ToolDeploymentList = ({ refreshDeployments }) => {
  const [deployments, setDeployments] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedDeploymentId, setSelectedDeploymentId] = useState(null);
  const [confirmStop, setConfirmStop] = useState(null);
  const [stoppingDeploymentId, setStoppingDeploymentId] = useState(null);
  const [snackbar, setSnackbar] = useState({ open: false, message: '' });

  const fetchDeployments = useCallback(async () => {
    setLoading(true);
    try {
      const response = await api.get('/tool/deployments');
      setDeployments(response.data);
    } catch (err) {
      console.error('Error fetching Tool deployments:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchDeployments();
  }, [fetchDeployments, refreshDeployments]);

  const handleCopyURL = (url) => {
    navigator.clipboard.writeText(url);
    setSnackbar({ open: true, message: 'URL copied to clipboard!' });
  };

  const handleOpenDetails = (deploymentId) => {
    setSelectedDeploymentId(deploymentId);
  };

  const handleStopDeployment = async (deploymentId) => {
    setStoppingDeploymentId(deploymentId);
    try {
      await api.delete(`/tool/deployment/${deploymentId}`);
      fetchDeployments();
    } catch (err) {
      console.error('Error stopping Tool deployment:', err);
      setError(err);
    } finally {
      setStoppingDeploymentId(null);
      setConfirmStop(null);
    }
  };

  if (loading && deployments.length === 0) return <Spinner />;
  if (error && deployments.length === 0) return <ErrorComponent message={error.message} />;

  return (
    <Box sx={{ mb: 4 }}>
      <SubTitle variant="h5">
        Your Tool Servers
        <Tooltip title="Tool servers currently deployed and accessible">
          <InfoIcon />
        </Tooltip>
      </SubTitle>
      
      <StyledTableContainer component={Paper}>
        <StyledTable>
          <TableHead>
            <TableRow>
              <TableHeaderCell>Name</TableHeaderCell>
              <TableHeaderCell>Status</TableHeaderCell>
              <TableHeaderCell>URL</TableHeaderCell>
              <TableHeaderCell>Actions</TableHeaderCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {deployments.length === 0 ? (
              <StyledTableRow className="empty-row">
                <TableBodyCell colSpan={4}>No Tool servers deployed yet</TableBodyCell>
              </StyledTableRow>
            ) : (
              deployments
                .filter(deployment => deployment.status !== 'STOPPED')
                .map((deployment) => (
                  <StyledTableRow key={deployment.id}>
                    <TableBodyCell>
                      {deployment.name.replace('mcp-', '').replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
                    </TableBodyCell>
                    <TableBodyCell>
                      <StatusChip 
                        label={deployment.status} 
                        size="small" 
                        variant="outlined"
                        status={deployment.status}
                      />
                    </TableBodyCell>
                    <TableBodyCell>
                      {deployment.status === 'DEPLOYED' ? (
                        <URLCell>
                          <URLText>{deployment.url}</URLText>
                          <Tooltip title="Copy URL">
                            <IconButton 
                              size="small" 
                              onClick={() => handleCopyURL(deployment.url)}
                              sx={{ color: '#00c07f' }}
                            >
                              <ContentCopyIcon fontSize="small" />
                            </IconButton>
                          </Tooltip>
                        </URLCell>
                      ) : (
                        <Typography variant="body2" color="text.secondary">
                          {deployment.status === 'DEPLOYING' ? 'Deploying...' : 'Not available'}
                        </Typography>
                      )}
                    </TableBodyCell>
                    <TableBodyCell>
                      <Box sx={{ display: 'flex', gap: 1 }}>
                        <ActionButton 
                          onClick={() => handleOpenDetails(deployment.id)}
                          size="small"
                          startIcon={<VisibilityIcon />}
                        >
                          Details
                        </ActionButton>
                        <ActionButton
                          color="error"
                          size="small"
                          startIcon={stoppingDeploymentId === deployment.id ? 
                            <CircularProgress size={16} color="inherit" /> : 
                            <StopIcon />
                          }
                          onClick={() => setConfirmStop(deployment)}
                          disabled={deployment.status === 'STOPPING' || stoppingDeploymentId === deployment.id}
                        >
                          {stoppingDeploymentId === deployment.id ? 'Stopping...' : 'Stop'}
                        </ActionButton>
                      </Box>
                    </TableBodyCell>
                  </StyledTableRow>
                ))
            )}
          </TableBody>
        </StyledTable>
      </StyledTableContainer>
      
      {selectedDeploymentId && (
        <ToolDeploymentDetail
          deploymentId={selectedDeploymentId}
          isOpen={!!selectedDeploymentId}
          onClose={() => setSelectedDeploymentId(null)}
          onDeploymentStopped={fetchDeployments}
        />
      )}

      {confirmStop && (
        <ConfirmModal
          title="Stop Tool Server"
          message={`Are you sure you want to stop "${confirmStop.name}"? You can redeploy it later.`}
          showModal={!!confirmStop}
          onConfirm={() => handleStopDeployment(confirmStop.id)}
          onCancel={() => setConfirmStop(null)}
        />
      )}

      <Snackbar
        open={snackbar.open}
        autoHideDuration={3000}
        onClose={() => setSnackbar({ open: false, message: '' })}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'center' }}
      >
        <Alert severity="success" sx={{ width: '100%' }}>
          {snackbar.message}
        </Alert>
      </Snackbar>
    </Box>
  );
};

export default ToolDeploymentList;