import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  IconButton,
  Typography,
  Box,
  TextField,
  Button,
  Alert,
  AlertTitle,
  Chip,
  Divider,
  Grid,
  Paper,
  Snackbar,
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import ContentCopyIcon from '@mui/icons-material/ContentCopy';
import InfoIcon from '@mui/icons-material/Info';
import TerminalIcon from '@mui/icons-material/Terminal';
import api from '../../utils/api';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 16,
    backgroundColor: theme.palette.background.paper,
    boxShadow: '0 8px 32px rgba(0, 0, 0, 0.24)',
    maxWidth: 800,
    width: '100%',
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.05), rgba(0, 192, 160, 0.05))',
  borderBottom: `1px solid ${theme.palette.divider}`,
  padding: theme.spacing(3),
  paddingBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between',
}));

const DialogTitleText = styled(Typography)(({ theme }) => ({
  fontSize: '1.5rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
}));

const URLSection = styled(Paper)(({ theme }) => ({
  padding: theme.spacing(3),
  marginBottom: theme.spacing(3),
  backgroundColor: 'rgba(0, 192, 127, 0.05)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: 12,
}));

const URLField = styled(TextField)(({ theme }) => ({
  '& .MuiInputBase-root': {
    fontFamily: 'monospace',
    fontSize: '1rem',
    backgroundColor: theme.palette.background.paper,
  },
}));

const CopyButton = styled(Button)(({ theme }) => ({
  minWidth: 100,
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  fontWeight: 600,
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
  },
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const InfoGrid = styled(Grid)(({ theme }) => ({
  '& .MuiGrid-item': {
    marginBottom: theme.spacing(2),
  },
}));

const Label = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontSize: '0.9rem',
  marginBottom: theme.spacing(0.5),
}));

const Value = styled(Typography)(({ theme }) => ({
  fontSize: '1rem',
  fontWeight: 500,
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 500,
  ...(status === 'DEPLOYED' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'DEPLOYING' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    color: '#ff9800',
    borderColor: '#ff9800',
  }),
}));

const UsageAlert = styled(Alert)(({ theme }) => ({
  marginTop: theme.spacing(3),
  '& .MuiAlert-message': {
    width: '100%',
  },
}));

const CodeBlock = styled(Box)(({ theme }) => ({
  backgroundColor: theme.palette.grey[900],
  color: theme.palette.common.white,
  padding: theme.spacing(2),
  borderRadius: 8,
  fontFamily: 'monospace',
  fontSize: '0.9rem',
  marginTop: theme.spacing(1),
  overflowX: 'auto',
}));

const ToolDeploymentDetail = ({ deploymentId, isOpen, onClose, onDeploymentStopped }) => {
  const [deployment, setDeployment] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [snackbar, setSnackbar] = useState({ open: false, message: '' });

  useEffect(() => {
    if (isOpen && deploymentId) {
      fetchDeploymentDetails();
    }
  }, [isOpen, deploymentId]);

  const fetchDeploymentDetails = async () => {
    setLoading(true);
    try {
      const response = await api.get(`/tool/deployment/${deploymentId}`);
      setDeployment(response.data);
    } catch (err) {
      console.error('Error fetching Tool deployment details:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  const handleCopyURL = () => {
    if (deployment?.url) {
      navigator.clipboard.writeText(deployment.url);
      setSnackbar({ open: true, message: 'URL copied to clipboard!' });
    }
  };

  const handleCopyConfig = () => {
    const config = JSON.stringify(
      {
        name: deployment.name,
        url: deployment.url,
      },
      null,
      2
    );
    navigator.clipboard.writeText(config);
    setSnackbar({ open: true, message: 'Configuration copied to clipboard!' });
  };

  if (!isOpen) return null;

  return (
    <StyledDialog open={isOpen} onClose={onClose} maxWidth="md" fullWidth>
      <StyledDialogTitle>
        <DialogTitleText>Tool Server Details</DialogTitleText>
        <IconButton onClick={onClose} size="small">
          <CloseIcon />
        </IconButton>
      </StyledDialogTitle>

      <DialogContent sx={{ p: 3 }}>
        {loading && <Spinner />}
        {error && <ErrorComponent message={error.message} />}

        {deployment && !loading && (
          <>
            {/* Tool URL Section */}
            <URLSection elevation={0}>
              <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
                Tool Server URL
              </Typography>
              <Box sx={{ display: 'flex', gap: 2, alignItems: 'stretch' }}>
                <URLField
                  fullWidth
                  value={deployment.url}
                  InputProps={{ readOnly: true }}
                  variant="outlined"
                />
                <CopyButton
                  variant="contained"
                  startIcon={<ContentCopyIcon />}
                  onClick={handleCopyURL}
                >
                  Copy
                </CopyButton>
              </Box>
            </URLSection>

            {/* Deployment Information */}
            <InfoSection>
              <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
                Deployment Information
              </Typography>
              <InfoGrid container spacing={3}>
                <Grid item xs={12} sm={6}>
                  <Label>Name</Label>
                  <Value>{deployment.name}</Value>
                </Grid>
                <Grid item xs={12} sm={6}>
                  <Label>Status</Label>
                  <StatusChip
                    label={deployment.status}
                    size="small"
                    variant="outlined"
                    status={deployment.status}
                  />
                </Grid>
                <Grid item xs={12} sm={6}>
                  <Label>Created At</Label>
                  <Value>{new Date(deployment.created_at).toLocaleString()}</Value>
                </Grid>
                {deployment.deployed_at && (
                  <Grid item xs={12} sm={6}>
                    <Label>Deployed At</Label>
                    <Value>{new Date(deployment.deployed_at).toLocaleString()}</Value>
                  </Grid>
                )}
              </InfoGrid>
            </InfoSection>

            <Divider sx={{ my: 3 }} />

            {/* Usage Instructions */}
            <UsageAlert severity="info" icon={<InfoIcon />}>
              <AlertTitle>How to use this Tool server</AlertTitle>
              <Typography variant="body2" sx={{ mb: 2 }}>
                1. Copy the URL above
                <br />
                2. Add it to your Tool client configuration (e.g., Claude Desktop, Kaizen chatbot)
                <br />
                3. The server will be available as "
                <strong>{deployment.name.replace('mcp-', '')}</strong>"
              </Typography>

              <Typography variant="body2" sx={{ mt: 2, mb: 1 }}>
                <strong>Example configuration for Claude Desktop:</strong>
              </Typography>
              <CodeBlock>
                {`{
  "mcpServers": {
    "${deployment.name.replace('mcp-', '')}": {
      "url": "${deployment.url}",
      "transport": "sse"
    }
  }
}`}
              </CodeBlock>

              <Button
                size="small"
                startIcon={<ContentCopyIcon />}
                onClick={handleCopyConfig}
                sx={{ mt: 2 }}
              >
                Copy Configuration
              </Button>
            </UsageAlert>

            {/* Environment Variables (if any) */}
            {deployment.env_vars && Object.keys(deployment.env_vars).length > 0 && (
              <>
                <Divider sx={{ my: 3 }} />
                <Box>
                  <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
                    Environment Variables
                  </Typography>
                  <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                    These environment variables are configured for this deployment:
                  </Typography>
                  {Object.entries(deployment.env_vars).map(([key, value]) => (
                    <Box key={key} sx={{ mb: 1 }}>
                      <Typography variant="body2" component="span" sx={{ fontWeight: 600 }}>
                        {key}:
                      </Typography>
                      <Typography
                        variant="body2"
                        component="span"
                        sx={{ ml: 1, fontFamily: 'monospace' }}
                      >
                        {key.includes('KEY') || key.includes('SECRET') ? '••••••••' : value}
                      </Typography>
                    </Box>
                  ))}
                </Box>
              </>
            )}
          </>
        )}
      </DialogContent>

      <Snackbar
        open={snackbar.open}
        autoHideDuration={3000}
        onClose={() => setSnackbar({ open: false, message: '' })}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'center' }}
      >
        <Alert severity="success" sx={{ width: '100%' }}>
          {snackbar.message}
        </Alert>
      </Snackbar>
    </StyledDialog>
  );
};

export default ToolDeploymentDetail;
