import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import { 
  Dialog, 
  DialogTitle, 
  DialogContent, 
  DialogActions,
  TextField,
  Button,
  Box,
  Typography,
  Alert,
  CircularProgress,
  InputAdornment,
  IconButton
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import VisibilityIcon from '@mui/icons-material/Visibility';
import VisibilityOffIcon from '@mui/icons-material/VisibilityOff';
import api from '../../utils/api';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 16,
    backgroundColor: theme.palette.background.paper,
    boxShadow: '0 8px 32px rgba(0, 0, 0, 0.24)',
    minWidth: 500,
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.05), rgba(0, 192, 160, 0.05))',
  borderBottom: `1px solid ${theme.palette.divider}`,
  padding: theme.spacing(3),
  paddingBottom: theme.spacing(2),
}));

const DialogTitleText = styled(Typography)(({ theme }) => ({
  fontSize: '1.5rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
}));

const CloseButton = styled(IconButton)(({ theme }) => ({
  position: 'absolute',
  right: theme.spacing(2),
  top: theme.spacing(2),
  color: theme.palette.grey[500],
}));

const StyledDialogContent = styled(DialogContent)(({ theme }) => ({
  padding: theme.spacing(3),
  paddingTop: theme.spacing(3) + ' !important',
}));

const FormField = styled(TextField)(({ theme }) => ({
  marginBottom: theme.spacing(2.5),
}));

const DeployButton = styled(Button)(({ theme }) => ({
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  fontWeight: 600,
  padding: theme.spacing(1.2, 3),
  borderRadius: 8,
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
  },
  '&:disabled': {
    background: theme.palette.grey[300],
    color: theme.palette.grey[500],
  },
}));

const ToolDeployModal = ({ open, onClose, onSuccess, template = null, isCustom = false }) => {
  const [deploymentName, setDeploymentName] = useState('');
  const [dockerImage, setDockerImage] = useState('');
  const [envVars, setEnvVars] = useState(template?.env_defaults || {});
  const [showEnvVars, setShowEnvVars] = useState({});
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  // Reset form when modal opens
  useEffect(() => {
    if (open) {
      setDeploymentName('');
      setDockerImage('');
      setEnvVars(template?.env_defaults || {});
      setShowEnvVars({});
      setError(null);
    }
  }, [open, template]);

  const handleDeploy = async () => {
    setLoading(true);
    setError(null);
    
    try {
      let response;
      
      if (isCustom) {
        // Deploy custom Tool server
        response = await api.post('/tool/deploy', {
          name: deploymentName,
          image: dockerImage,
          env_vars: envVars
        });
      } else if (template) {
        // Deploy from template
        response = await api.post(`/tool/deploy-template/${template.name}`, {
          name: deploymentName,
          env_vars: envVars
        });
      }
      
      if (response.data) {
        onSuccess();
      }
    } catch (err) {
      console.error('Error deploying Tool server:', err);
      setError(err.response?.data?.detail || 'Failed to deploy Tool server');
    } finally {
      setLoading(false);
    }
  };

  const handleEnvVarChange = (varName, value) => {
    setEnvVars(prev => ({
      ...prev,
      [varName]: value
    }));
  };

  const toggleShowEnvVar = (varName) => {
    setShowEnvVars(prev => ({
      ...prev,
      [varName]: !prev[varName]
    }));
  };

  const isFormValid = () => {
    if (!deploymentName) return false;
    if (isCustom && !dockerImage) return false;
    
    // Check if all required env vars are filled
    if (template?.required_env_vars) {
      for (const varName of template.required_env_vars) {
        if (!envVars[varName] || envVars[varName].trim() === '') {
          return false;
        }
      }
    }
    
    return true;
  };

  return (
    <StyledDialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <StyledDialogTitle>
        <DialogTitleText>
          {isCustom ? 'Deploy Custom Tool Server' : `Deploy ${template?.display_name || template?.name || 'Tool Server'}`}
        </DialogTitleText>
        <CloseButton onClick={onClose}>
          <CloseIcon />
        </CloseButton>
      </StyledDialogTitle>
      
      <StyledDialogContent>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }} onClose={() => setError(null)}>
            {error}
          </Alert>
        )}

        <FormField
          fullWidth
          label="Deployment Name"
          value={deploymentName}
          onChange={(e) => setDeploymentName(e.target.value)}
          placeholder="my-tool-server"
          helperText="A unique name for your tool deployment"
          autoComplete="off"
          required
        />

        {isCustom && (
          <FormField
            fullWidth
            label="Docker Image"
            value={dockerImage}
            onChange={(e) => setDockerImage(e.target.value)}
            placeholder="username/mcp-server:latest"
            helperText="The Docker image containing your tool server"
            autoComplete="off"
            required
          />
        )}

        {template?.required_env_vars && template.required_env_vars.length > 0 && (
          <Box sx={{ mt: 3 }}>
            <Typography variant="subtitle2" sx={{ mb: 2, color: 'text.secondary' }}>
              Required Environment Variables
            </Typography>
            {template.required_env_vars.map((varName) => (
              <FormField
                key={varName}
                fullWidth
                label={varName}
                type={showEnvVars[varName] ? 'text' : 'password'}
                value={envVars[varName] || ''}
                onChange={(e) => handleEnvVarChange(varName, e.target.value)}
                autoComplete="off"
                required
                InputProps={{
                  endAdornment: (
                    <InputAdornment position="end">
                      <IconButton
                        onClick={() => toggleShowEnvVar(varName)}
                        edge="end"
                        size="small"
                      >
                        {showEnvVars[varName] ? <VisibilityOffIcon /> : <VisibilityIcon />}
                      </IconButton>
                    </InputAdornment>
                  ),
                }}
                helperText={`Required for ${template.display_name || template.name}`}
              />
            ))}
          </Box>
        )}

        {template?.env_defaults && Object.keys(template.env_defaults).length > 0 && (
          <Box sx={{ mt: 3 }}>
            <Typography variant="subtitle2" sx={{ mb: 2, color: 'text.secondary' }}>
              Optional Configuration (defaults provided)
            </Typography>
            {Object.entries(template.env_defaults).map(([varName, defaultValue]) => (
              <FormField
                key={varName}
                fullWidth
                label={varName}
                type={varName.toLowerCase().includes('password') || varName.toLowerCase().includes('key') || varName.toLowerCase().includes('secret') 
                  ? (showEnvVars[varName] ? 'text' : 'password')
                  : 'text'}
                value={envVars[varName] || ''}
                onChange={(e) => handleEnvVarChange(varName, e.target.value)}
                placeholder={String(defaultValue)}
                autoComplete="off"
                InputProps={
                  (varName.toLowerCase().includes('password') || varName.toLowerCase().includes('key') || varName.toLowerCase().includes('secret')) ? {
                    endAdornment: (
                      <InputAdornment position="end">
                        <IconButton
                          onClick={() => toggleShowEnvVar(varName)}
                          edge="end"
                          size="small"
                        >
                          {showEnvVars[varName] ? <VisibilityOffIcon /> : <VisibilityIcon />}
                        </IconButton>
                      </InputAdornment>
                    ),
                  } : {}
                }
                helperText={`Default: ${defaultValue}`}
              />
            ))}
          </Box>
        )}

        {!template?.required_env_vars?.length && !template?.env_defaults && !isCustom && (
          <Alert severity="info" sx={{ mt: 2 }}>
            This Tool server doesn't require any additional configuration.
          </Alert>
        )}
      </StyledDialogContent>
      
      <DialogActions sx={{ p: 3, pt: 2 }}>
        <Button onClick={onClose} disabled={loading}>
          Cancel
        </Button>
        <DeployButton
          onClick={handleDeploy}
          disabled={loading || !isFormValid()}
          startIcon={loading ? <CircularProgress size={20} color="inherit" /> : <RocketLaunchIcon />}
        >
          {loading ? 'Deploying...' : 'Deploy'}
        </DeployButton>
      </DialogActions>
    </StyledDialog>
  );
};

export default ToolDeployModal;