import React from 'react';
import { renderWithProviders, screen, fireEvent } from '../../../test-utils';
import TourSettings from '../TourSettings';

// Mock the TourContext
const mockResetTours = jest.fn();
const mockStartTour = jest.fn();
const mockSkipAllTours = jest.fn();

jest.mock('../../../contexts/TourContext', () => ({
  useTour: () => ({
    resetTours: mockResetTours,
    startTour: mockStartTour,
    skipAllTours: mockSkipAllTours,
    isNoviceMode: false
  })
}));

// Mock window.alert
const mockAlert = jest.fn();
global.alert = mockAlert;

describe('TourSettings', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('Tour Visibility Based on Admin Status', () => {
    test('non-admin user sees 3 tours (excluding Cluster Tour)', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      // Should see these tours
      expect(screen.getByText('Homepage Tour')).toBeInTheDocument();
      expect(screen.getByText('Models Tour')).toBeInTheDocument();
      expect(screen.getByText('App Garden Tour')).toBeInTheDocument();

      // Should NOT see Cluster Tour (admin only)
      expect(screen.queryByText('Cluster Tour')).not.toBeInTheDocument();
    });

    test('admin user sees all 4 tours including Cluster Tour', () => {
      renderWithProviders(<TourSettings isAdmin={true} />);

      // Should see all tours including admin-only
      expect(screen.getByText('Homepage Tour')).toBeInTheDocument();
      expect(screen.getByText('Models Tour')).toBeInTheDocument();
      expect(screen.getByText('App Garden Tour')).toBeInTheDocument();
      expect(screen.getByText('Cluster Tour')).toBeInTheDocument();
    });

    test('non-admin user sees 3 Play Tour buttons', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      const playButtons = screen.getAllByRole('button', { name: /Play Tour/i });
      expect(playButtons).toHaveLength(3);
    });

    test('admin user sees 4 Play Tour buttons', () => {
      renderWithProviders(<TourSettings isAdmin={true} />);

      const playButtons = screen.getAllByRole('button', { name: /Play Tour/i });
      expect(playButtons).toHaveLength(4);
    });
  });

  describe('Tour Descriptions', () => {
    test('displays tour descriptions for non-admin', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      expect(screen.getByText('Introduction to Kamiwaza and main features')).toBeInTheDocument();
      expect(screen.getByText('Learn about model selection and deployment')).toBeInTheDocument();
      expect(screen.getByText('Discover pre-built AI applications')).toBeInTheDocument();

      // Should NOT see Cluster Tour description
      expect(screen.queryByText('Understand hardware and resource management')).not.toBeInTheDocument();
    });

    test('displays all tour descriptions for admin', () => {
      renderWithProviders(<TourSettings isAdmin={true} />);

      expect(screen.getByText('Introduction to Kamiwaza and main features')).toBeInTheDocument();
      expect(screen.getByText('Learn about model selection and deployment')).toBeInTheDocument();
      expect(screen.getByText('Discover pre-built AI applications')).toBeInTheDocument();
      expect(screen.getByText('Understand hardware and resource management')).toBeInTheDocument();
    });
  });

  describe('Tour Actions', () => {
    test('clicking Play Tour calls startTour with correct key', () => {
      renderWithProviders(<TourSettings isAdmin={true} />);

      // Find and click the Homepage Tour play button
      const playButtons = screen.getAllByRole('button', { name: /Play Tour/i });

      // Click homepage tour (first in list)
      fireEvent.click(playButtons[0]);
      expect(mockStartTour).toHaveBeenCalledWith('homepage', true);

      // Click models tour (second in list)
      fireEvent.click(playButtons[1]);
      expect(mockStartTour).toHaveBeenCalledWith('models', true);
    });

    test('admin can start Cluster Tour', () => {
      renderWithProviders(<TourSettings isAdmin={true} />);

      const playButtons = screen.getAllByRole('button', { name: /Play Tour/i });

      // Cluster tour is the 4th (last) in the list for admins
      fireEvent.click(playButtons[3]);
      expect(mockStartTour).toHaveBeenCalledWith('cluster', true);
    });

    test('Reset All Tours calls resetTours and shows alert', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      const resetButton = screen.getByRole('button', { name: /Reset All Tours/i });
      fireEvent.click(resetButton);

      expect(mockResetTours).toHaveBeenCalled();
      expect(mockAlert).toHaveBeenCalledWith('All tours have been reset. They will play again when you visit each section.');
    });

    test('Skip All Tours calls skipAllTours and shows alert', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      const skipButton = screen.getByRole('button', { name: /Skip All Tours/i });
      fireEvent.click(skipButton);

      expect(mockSkipAllTours).toHaveBeenCalled();
      expect(mockAlert).toHaveBeenCalledWith("All tours have been marked as completed. You won't see them again unless you reset.");
    });
  });

  describe('UI Elements', () => {
    test('renders Guided Tours heading', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      expect(screen.getByText('Guided Tours')).toBeInTheDocument();
    });

    test('renders management buttons for all users', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      expect(screen.getByRole('button', { name: /Reset All Tours/i })).toBeInTheDocument();
      expect(screen.getByRole('button', { name: /Skip All Tours/i })).toBeInTheDocument();
    });

    test('renders info alert about novice mode', () => {
      renderWithProviders(<TourSettings isAdmin={false} />);

      expect(screen.getByText(/Tours are only automatically shown in Novice Mode/i)).toBeInTheDocument();
    });
  });

  describe('Default Props', () => {
    test('defaults isAdmin to false when not provided', () => {
      renderWithProviders(<TourSettings />);

      // Should not see Cluster Tour when isAdmin defaults to false
      expect(screen.queryByText('Cluster Tour')).not.toBeInTheDocument();

      // Should see non-admin tours
      expect(screen.getByText('Homepage Tour')).toBeInTheDocument();
    });
  });

  describe('Edge Cases', () => {
    test('handles isAdmin=undefined as non-admin', () => {
      renderWithProviders(<TourSettings isAdmin={undefined} />);

      // Should not see Cluster Tour
      expect(screen.queryByText('Cluster Tour')).not.toBeInTheDocument();
      expect(screen.getByText('Homepage Tour')).toBeInTheDocument();
    });
  });
});
