import React from 'react';
import { renderWithProviders, testData, screen, fireEvent } from '../../../test-utils';
import Settings from '../Settings';

// Mock child components to isolate Settings testing
jest.mock('../AuthFederation', () => {
  const MockAuthFederation = () => <div data-testid="auth-federation">AuthFederation Component</div>;
  MockAuthFederation.displayName = 'MockAuthFederation';
  return MockAuthFederation;
});
jest.mock('../ChangePasswordForm', () => {
  const MockChangePasswordForm = () => <div data-testid="change-password-form">ChangePasswordForm Component</div>;
  MockChangePasswordForm.displayName = 'MockChangePasswordForm';
  return MockChangePasswordForm;
});
jest.mock('../LocalUsersSection', () => {
  const MockLocalUsersSection = () => <div data-testid="local-users-section">LocalUsersSection Component</div>;
  MockLocalUsersSection.displayName = 'MockLocalUsersSection';
  return MockLocalUsersSection;
});
jest.mock('../TourSettings', () => {
  const PropTypesModule = require('prop-types');
  const MockTourSettings = ({ isAdmin }) => (
    <div data-testid="tour-settings">TourSettings Component (isAdmin: {isAdmin ? 'true' : 'false'})</div>
  );
  MockTourSettings.displayName = 'MockTourSettings';
  MockTourSettings.propTypes = { isAdmin: PropTypesModule.bool };
  return MockTourSettings;
});

// Mock useSearchParams with stateful behavior
let mockCurrentSearchParams = new URLSearchParams();
const mockSetSearchParams = jest.fn((params) => {
  mockCurrentSearchParams = new URLSearchParams(params);
});

jest.mock('react-router-dom', () => ({
  ...jest.requireActual('react-router-dom'),
  useSearchParams: () => [mockCurrentSearchParams, mockSetSearchParams],
}));

// Mock localStorage
const localStorageMock = {
  getItem: jest.fn(),
  setItem: jest.fn(),
  clear: jest.fn(),
};
Object.defineProperty(window, 'localStorage', { value: localStorageMock });

describe('Settings', () => {
  // Admin user for testing
  const adminUser = { ...testData.user, roles: ['admin'], is_superuser: true };
  // Regular non-admin user
  const regularUser = { ...testData.user, roles: ['user'], is_superuser: false };

  beforeEach(() => {
    jest.clearAllMocks();
    localStorageMock.getItem.mockReturnValue(null);
    // Reset search params for test isolation
    mockCurrentSearchParams = new URLSearchParams();
  });

  describe('Tab Visibility', () => {
    test('non-admin user sees only "My Settings" tab', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      // Should see "My Settings" tab
      expect(screen.getByRole('tab', { name: 'My Settings' })).toBeInTheDocument();

      // Should NOT see "Auth & Users" tab
      expect(screen.queryByRole('tab', { name: 'Auth & Users' })).not.toBeInTheDocument();
    });

    test('admin user sees both tabs', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      // Should see both tabs
      expect(screen.getByRole('tab', { name: 'My Settings' })).toBeInTheDocument();
      expect(screen.getByRole('tab', { name: 'Auth & Users' })).toBeInTheDocument();
    });

    test('user with empty roles array sees only "My Settings" tab', () => {
      const userWithEmptyRoles = { ...testData.user, roles: [] };

      renderWithProviders(<Settings />, {
        authValue: {
          user: userWithEmptyRoles,
          loading: false
        }
      });

      expect(screen.getByRole('tab', { name: 'My Settings' })).toBeInTheDocument();
      expect(screen.queryByRole('tab', { name: 'Auth & Users' })).not.toBeInTheDocument();
    });
  });

  describe('Advanced Developer Mode', () => {
    test('non-admin user does not see Advanced Developer Mode section', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      // Should NOT see Advanced Developer Mode
      expect(screen.queryByText('Advanced Developer Mode')).not.toBeInTheDocument();
      expect(screen.queryByText('Enable advanced interface options')).not.toBeInTheDocument();
    });

    test('admin user sees Advanced Developer Mode section', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      // Should see Advanced Developer Mode
      expect(screen.getByText('Advanced Developer Mode')).toBeInTheDocument();
      expect(screen.getByText('Enable advanced interface options')).toBeInTheDocument();
    });

    test('Advanced Developer Mode switch reflects localStorage value', () => {
      localStorageMock.getItem.mockReturnValue('true');

      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      const switchElement = screen.getByRole('checkbox');
      expect(switchElement).toBeChecked();
    });

    test('Advanced Developer Mode switch is unchecked when localStorage is false', () => {
      localStorageMock.getItem.mockReturnValue('false');

      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      const switchElement = screen.getByRole('checkbox');
      expect(switchElement).not.toBeChecked();
    });
  });

  describe('Tab Content', () => {
    test('My Settings tab shows ChangePasswordForm and TourSettings', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      // Should show personal settings content
      expect(screen.getByTestId('change-password-form')).toBeInTheDocument();
      expect(screen.getByTestId('tour-settings')).toBeInTheDocument();
    });

    test('admin can switch to Auth & Users tab', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      // Initially on My Settings tab
      expect(screen.getByTestId('change-password-form')).toBeInTheDocument();

      // Click Auth & Users tab
      fireEvent.click(screen.getByRole('tab', { name: 'Auth & Users' }));

      // Should show Auth & Users content
      expect(screen.getByTestId('auth-federation')).toBeInTheDocument();
      expect(screen.getByTestId('local-users-section')).toBeInTheDocument();

      // Should hide My Settings content
      expect(screen.queryByTestId('change-password-form')).not.toBeInTheDocument();
    });

    test('tab change updates URL params', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      // Click Auth & Users tab
      fireEvent.click(screen.getByRole('tab', { name: 'Auth & Users' }));

      // Should update URL params
      expect(mockSetSearchParams).toHaveBeenCalledWith({ tab: 'auth-users' });
    });

    test('switching back to My Settings clears URL params', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      // Click Auth & Users tab first
      fireEvent.click(screen.getByRole('tab', { name: 'Auth & Users' }));
      mockSetSearchParams.mockClear();

      // Click My Settings tab
      fireEvent.click(screen.getByRole('tab', { name: 'My Settings' }));

      // Should clear URL params
      expect(mockSetSearchParams).toHaveBeenCalledWith({});
    });
  });

  describe('TourSettings Integration', () => {
    test('passes isAdmin=false to TourSettings for non-admin user', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      expect(screen.getByText('TourSettings Component (isAdmin: false)')).toBeInTheDocument();
    });

    test('passes isAdmin=true to TourSettings for admin user', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: adminUser,
          loading: false
        }
      });

      expect(screen.getByText('TourSettings Component (isAdmin: true)')).toBeInTheDocument();
    });
  });

  describe('Page Structure', () => {
    test('renders Settings title', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      expect(screen.getByRole('heading', { name: 'Settings' })).toBeInTheDocument();
    });

    test('renders Tours & Help card', () => {
      renderWithProviders(<Settings />, {
        authValue: {
          user: regularUser,
          loading: false
        }
      });

      expect(screen.getByText('Tours & Help')).toBeInTheDocument();
    });
  });

  describe('Edge Cases', () => {
    test('handles user with undefined roles', () => {
      const userWithNoRoles = { ...testData.user, roles: undefined };

      renderWithProviders(<Settings />, {
        authValue: {
          user: userWithNoRoles,
          loading: false
        }
      });

      // Should only see My Settings tab
      expect(screen.getByRole('tab', { name: 'My Settings' })).toBeInTheDocument();
      expect(screen.queryByRole('tab', { name: 'Auth & Users' })).not.toBeInTheDocument();

      // Should NOT see Advanced Developer Mode
      expect(screen.queryByText('Advanced Developer Mode')).not.toBeInTheDocument();
    });

    test('handles user with null roles', () => {
      const userWithNullRoles = { ...testData.user, roles: null };

      renderWithProviders(<Settings />, {
        authValue: {
          user: userWithNullRoles,
          loading: false
        }
      });

      // Should only see My Settings tab
      expect(screen.getByRole('tab', { name: 'My Settings' })).toBeInTheDocument();
      expect(screen.queryByRole('tab', { name: 'Auth & Users' })).not.toBeInTheDocument();
    });
  });
});
