import React, { useMemo } from 'react';
import PropTypes from 'prop-types';
import {
  Box,
  Button,
  Typography,
  Stack,
  Paper,
  Alert
} from '@mui/material';
import { useTour } from '../../contexts/TourContext';
import RestartAltIcon from '@mui/icons-material/RestartAlt';
import PlayCircleOutlineIcon from '@mui/icons-material/PlayCircleOutline';

const TourSettings = ({ isAdmin = false }) => {
  const { resetTours, startTour, skipAllTours, isNoviceMode } = useTour();

  const handleResetTours = () => {
    resetTours();
    alert('All tours have been reset. They will play again when you visit each section.');
  };

  const handleSkipAllTours = () => {
    skipAllTours();
    alert('All tours have been marked as completed. You won\'t see them again unless you reset.');
  };

  // All available tours
  const allTours = [
    { key: 'homepage', name: 'Homepage Tour', description: 'Introduction to Kamiwaza and main features' },
    { key: 'models', name: 'Models Tour', description: 'Learn about model selection and deployment' },
    { key: 'appGarden', name: 'App Garden Tour', description: 'Discover pre-built AI applications' },
    { key: 'cluster', name: 'Cluster Tour', description: 'Understand hardware and resource management', adminOnly: true }
  ];

  // Filter tours based on admin status
  const tours = useMemo(() => {
    return allTours.filter(tour => !tour.adminOnly || isAdmin);
  }, [isAdmin]);

  return (
    <Paper sx={{ p: 3, mb: 3 }}>
      <Typography variant="h6" gutterBottom>
        Guided Tours
      </Typography>
      
      {!isNoviceMode && (
        <Alert severity="info" sx={{ mb: 2 }}>
          Tours are only automatically shown in Novice Mode. You can manually replay them here.
        </Alert>
      )}
      
      <Stack spacing={2}>
        <Box>
          <Typography variant="body2" color="text.secondary" gutterBottom>
            Replay specific tours to refresh your memory about features:
          </Typography>
          <Stack spacing={1} sx={{ mt: 2 }}>
            {tours.map(tour => (
              <Box key={tour.key} sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box>
                  <Typography variant="body1">{tour.name}</Typography>
                  <Typography variant="caption" color="text.secondary">{tour.description}</Typography>
                </Box>
                <Button
                  size="small"
                  startIcon={<PlayCircleOutlineIcon />}
                  onClick={() => startTour(tour.key, true)}
                  variant="outlined"
                >
                  Play Tour
                </Button>
              </Box>
            ))}
          </Stack>
        </Box>
        
        <Box sx={{ mt: 3, pt: 2, borderTop: 1, borderColor: 'divider' }}>
          <Typography variant="body2" color="text.secondary" gutterBottom>
            Manage all tours:
          </Typography>
          <Stack direction="row" spacing={2} sx={{ mt: 2 }}>
            <Button
              variant="contained"
              startIcon={<RestartAltIcon />}
              onClick={handleResetTours}
              color="primary"
            >
              Reset All Tours
            </Button>
            <Button
              variant="outlined"
              onClick={handleSkipAllTours}
              color="secondary"
            >
              Skip All Tours
            </Button>
          </Stack>
        </Box>
      </Stack>
    </Paper>
  );
};

TourSettings.propTypes = {
  isAdmin: PropTypes.bool,
};

export default TourSettings;