import React, { useEffect, useMemo, useState } from 'react';
import { useSearchParams } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Tabs,
  Tab,
  Switch,
  FormControlLabel,
  Card,
  CardHeader,
  CardContent,
  Stack,
} from '@mui/material';
import { useIsAdmin } from '../../hooks/useIsAdmin';
import TourSettings from './TourSettings';
import AuthFederation from './AuthFederation';
import ChangePasswordForm from './ChangePasswordForm';
import LocalUsersSection from './LocalUsersSection';

const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(3),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const TabPanel = styled(Box)(({ theme }) => ({
  padding: theme.spacing(3, 3),
}));

const Settings = () => {
  const isAdmin = useIsAdmin();
  const [searchParams, setSearchParams] = useSearchParams();
  const [tabValue, setTabValue] = useState('personal');
  const [advancedMode, setAdvancedMode] = useState(
    localStorage.getItem('advancedDeveloperMode') === 'true'
  );

  // Tabs: non-admins only see "My Settings", admins see both tabs
  const TABS = useMemo(() => {
    const tabs = [{ label: 'My Settings', value: 'personal' }];
    if (isAdmin) {
      tabs.push({ label: 'Auth & Users', value: 'auth-users' });
    }
    return tabs;
  }, [isAdmin]);

  const validTabValues = useMemo(() => new Set(TABS.map((tab) => tab.value)), [TABS]);

  useEffect(() => {
    const requested = searchParams.get('tab');
    if (requested && validTabValues.has(requested) && requested !== tabValue) {
      setTabValue(requested);
    } else if (!requested && tabValue !== 'personal') {
      setTabValue('personal');
    }
  }, [searchParams, tabValue, validTabValues]);

  const handleTabChange = (_event, newValue) => {
    setTabValue(newValue);
    if (newValue === 'personal') {
      setSearchParams({});
    } else {
      setSearchParams({ tab: newValue });
    }
  };

  const handleAdvancedModeToggle = (event) => {
    const newValue = event.target.checked;
    setAdvancedMode(newValue);
    localStorage.setItem('advancedDeveloperMode', newValue.toString());
    window.location.reload();
  };

  return (
    <ContentContainer>
      <SectionTitle variant="h4">Settings</SectionTitle>

      <Paper sx={{ width: '100%' }}>
        <Box sx={{ borderBottom: 1, borderColor: 'divider' }}>
          <Tabs
            value={tabValue}
            onChange={handleTabChange}
            aria-label="settings tabs"
            textColor="primary"
            indicatorColor="primary"
          >
            {TABS.map((tab) => (
              <Tab key={tab.value} label={tab.label} value={tab.value} />
            ))}
          </Tabs>
        </Box>

        {tabValue === 'personal' && (
          <TabPanel role="tabpanel" aria-labelledby="settings-tab-personal">
            <Stack spacing={3}>
              <ChangePasswordForm />

              {isAdmin && (
                <Card>
                  <CardHeader title="Advanced Developer Mode" />
                  <CardContent>
                    <FormControlLabel
                      control={
                        <Switch
                          checked={advancedMode}
                          onChange={handleAdvancedModeToggle}
                          color="primary"
                        />
                      }
                      label="Enable advanced interface options"
                    />
                    <Typography
                      variant="caption"
                      display="block"
                      color="text.secondary"
                      sx={{ mt: 1, ml: 4.5 }}
                    >
                      Turn this off for a streamlined experience. Turn it on for full administrative and developer controls.
                    </Typography>
                  </CardContent>
                </Card>
              )}

              <Card>
                <CardHeader title="Tours & Help" />
                <CardContent>
                  <TourSettings isAdmin={isAdmin} />
                </CardContent>
              </Card>
            </Stack>
          </TabPanel>
        )}

        {tabValue === 'auth-users' && (
          <TabPanel role="tabpanel" aria-labelledby="settings-tab-auth-users">
            <Stack spacing={3}>
              <AuthFederation />
              <LocalUsersSection />
            </Stack>
          </TabPanel>
        )}
      </Paper>
    </ContentContainer>
  );
};

export default Settings;
