import React, { useEffect, useState } from 'react';
import {
  Card,
  CardHeader,
  CardContent,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Button,
  Alert,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Stack,
  Typography,
} from '@mui/material';
import AddIcon from '@mui/icons-material/Add';
import EditIcon from '@mui/icons-material/Edit';
import DeleteIcon from '@mui/icons-material/Delete';
import LockResetIcon from '@mui/icons-material/LockReset';
import api from '../../utils/api';
import Spinner from '../common/Spinner';

const LocalUsersSection = () => {
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(true);
  const [feedback, setFeedback] = useState(null);
  const [error, setError] = useState(null);
  const [idpProvisioned, setIdpProvisioned] = useState(false);

  const [editUser, setEditUser] = useState(null);
  const [editSubmitting, setEditSubmitting] = useState(false);

  const [resetUser, setResetUser] = useState(null);
  const [resetPassword, setResetPassword] = useState('');
  const [resetSubmitting, setResetSubmitting] = useState(false);

  const [deleteUser, setDeleteUser] = useState(null);
  const [deleteSubmitting, setDeleteSubmitting] = useState(false);

  const [addUserOpen, setAddUserOpen] = useState(false);
  const [addUserData, setAddUserData] = useState({ username: '', email: '', password: '' });
  const [addUserSubmitting, setAddUserSubmitting] = useState(false);

  const loadUsers = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await api.get('/auth/users/');
      const localUsers = (response.data || []).filter((user) => !user.is_external && !user.deleted);
      setUsers(localUsers);
    } catch (err) {
      if (err?.response?.status === 403) {
        setError('Admin role required to manage local users.');
      } else {
        setError('Failed to load users.');
      }
    } finally {
      setLoading(false);
    }
  };

  const loadIdpStatus = async () => {
    try {
      const response = await api.get('/auth/idp/providers');
      const data = response.data || {};
      const enabled = data.idp_management_enabled === true;
      const hasProviders = Array.isArray(data.providers) && data.providers.length > 0;
      setIdpProvisioned(enabled && hasProviders);
    } catch (err) {
      setIdpProvisioned(false);
    }
  };

  useEffect(() => {
    loadUsers();
    loadIdpStatus();
  }, []);

  const handleEditFieldChange = (event) => {
    const { name, value } = event.target;
    setEditUser((prev) => ({ ...prev, [name]: value }));
  };

  const handleEditSubmit = async (event) => {
    event.preventDefault();
    if (!editUser) return;
    setFeedback(null);
    setEditSubmitting(true);
    try {
      await api.put(`/auth/users/${editUser.id}`, {
        username: editUser.username,
        email: editUser.email,
      });
      setFeedback({ type: 'success', text: `Updated ${editUser.username}.` });
      setEditUser(null);
      await loadUsers();
    } catch (err) {
      const detail = err?.response?.data?.detail || 'Unable to update user.';
      setFeedback({ type: 'error', text: detail });
    } finally {
      setEditSubmitting(false);
    }
  };

  const handleResetSubmit = async (event) => {
    event.preventDefault();
    if (!resetUser) return;
    setFeedback(null);
    setResetSubmitting(true);
    try {
      await api.post(`/auth/users/${resetUser.id}/password`, {
        new_password: resetPassword,
      });
      setFeedback({ type: 'success', text: `Password reset for ${resetUser.username}.` });
      setResetUser(null);
      setResetPassword('');
    } catch (err) {
      const detail = err?.response?.data?.detail || 'Unable to reset password.';
      setFeedback({ type: 'error', text: detail });
    } finally {
      setResetSubmitting(false);
    }
  };

  const handleDeleteConfirm = async () => {
    if (!deleteUser) return;
    setFeedback(null);
    setDeleteSubmitting(true);
    try {
      await api.delete(`/auth/users/${deleteUser.id}`);
      setFeedback({ type: 'success', text: `User ${deleteUser.username} deleted.` });
      setDeleteUser(null);
      await loadUsers();
    } catch (err) {
      const detail = err?.response?.data?.detail || 'Unable to delete user.';
      setFeedback({ type: 'error', text: detail });
    } finally {
      setDeleteSubmitting(false);
    }
  };

  const handleAddUserFieldChange = (event) => {
    const { name, value } = event.target;
    setAddUserData((prev) => ({ ...prev, [name]: value }));
  };

  const handleAddUserSubmit = async (event) => {
    event.preventDefault();
    setFeedback(null);
    setAddUserSubmitting(true);
    try {
      await api.post('/auth/users/local', {
        username: addUserData.username,
        email: addUserData.email || null,
        password: addUserData.password,
      });
      setFeedback({ type: 'success', text: `User ${addUserData.username} created.` });
      setAddUserOpen(false);
      setAddUserData({ username: '', email: '', password: '' });
      await loadUsers();
    } catch (err) {
      const detail = err?.response?.data?.detail || 'Unable to create user.';
      setFeedback({ type: 'error', text: detail });
    } finally {
      setAddUserSubmitting(false);
    }
  };

  return (
    <Card>
      <CardHeader
        title="Local Users"
        subheader={
          idpProvisioned
            ? 'Provision or auto-link users through your identity provider; local creation in this console is disabled.'
            : 'Manage local user accounts. Configure an identity provider above for centralized user management.'
        }
        action={
          !idpProvisioned && (
            <Button
              variant="contained"
              startIcon={<AddIcon />}
              onClick={() => setAddUserOpen(true)}
              size="small"
            >
              Add User
            </Button>
          )
        }
      />
      <CardContent>
        {idpProvisioned ? (
          <Alert severity="info" sx={{ mb: 2 }}>
            User provisioning is handled by your identity provider (e.g., Keycloak). Configure providers in Authentication Providers above to manage provisioning and auto-linking; this console cannot create new users. CAC/X.509 logins require certificates to map to an existing IdP user.
          </Alert>
        ) : (
          <Alert severity="info" sx={{ mb: 2 }}>
            No identity provider configured. You can create and manage local users below, or configure an identity provider above for centralized user management.
          </Alert>
        )}
        {feedback && (
          <Alert severity={feedback.type} sx={{ mb: 2 }}>
            {feedback.text}
          </Alert>
        )}
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}
        {loading ? (
          <Spinner />
        ) : users.length === 0 ? (
          <Typography variant="body2" color="text.secondary">
            {idpProvisioned
              ? 'No local users found. Create or link users in your identity provider; accounts will appear here after they authenticate.'
              : 'No local users found. Click "Add User" above to create one.'}
          </Typography>
        ) : (
          <TableContainer>
            <Table size="small">
              <TableHead>
                <TableRow>
                  <TableCell>Username</TableCell>
                  <TableCell>Email</TableCell>
                  <TableCell>Roles</TableCell>
                  <TableCell align="right">Actions</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {users.map((user) => (
                  <TableRow key={user.id} hover>
                    <TableCell>{user.username}</TableCell>
                    <TableCell>{user.email || '—'}</TableCell>
                    <TableCell>{(user.roles || []).join(', ') || 'user'}</TableCell>
                    <TableCell align="right">
                      <IconButton
                        aria-label={`Edit ${user.username}`}
                        onClick={() => setEditUser(user)}
                        size="small"
                      >
                        <EditIcon fontSize="small" />
                      </IconButton>
                      <IconButton
                        aria-label={`Reset password for ${user.username}`}
                        onClick={() => {
                          setResetUser(user);
                          setResetPassword('');
                        }}
                        size="small"
                      >
                        <LockResetIcon fontSize="small" />
                      </IconButton>
                      <IconButton
                        aria-label={`Delete ${user.username}`}
                        onClick={() => setDeleteUser(user)}
                        size="small"
                      >
                        <DeleteIcon fontSize="small" />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        )}
      </CardContent>

      <Dialog open={Boolean(editUser)} onClose={() => setEditUser(null)} maxWidth="sm" fullWidth>
        <DialogTitle>Edit User</DialogTitle>
        <DialogContent>
          {editUser && (
            <Stack component="form" spacing={2} onSubmit={handleEditSubmit} sx={{ mt: 1 }}>
              <TextField
                label="Username"
                name="username"
                value={editUser.username}
                onChange={handleEditFieldChange}
                required
                fullWidth
              />
              <TextField
                label="Email"
                name="email"
                value={editUser.email || ''}
                onChange={handleEditFieldChange}
                type="email"
                fullWidth
              />
            </Stack>
          )}
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setEditUser(null)} disabled={editSubmitting}>
            Cancel
          </Button>
          <Button onClick={handleEditSubmit} variant="contained" disabled={editSubmitting}>
            {editSubmitting ? 'Saving…' : 'Save Changes'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog open={Boolean(resetUser)} onClose={() => setResetUser(null)} maxWidth="sm" fullWidth>
        <DialogTitle>Reset Password</DialogTitle>
        <DialogContent>
          {resetUser && (
            <Stack component="form" spacing={2} onSubmit={handleResetSubmit} sx={{ mt: 1 }}>
              <Typography variant="body2">
                Enter a new password for <strong>{resetUser.username}</strong>.
              </Typography>
              <TextField
                label="New Password"
                value={resetPassword}
                onChange={(event) => setResetPassword(event.target.value)}
                type="password"
                required
                fullWidth
              />
            </Stack>
          )}
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setResetUser(null)} disabled={resetSubmitting}>
            Cancel
          </Button>
          <Button onClick={handleResetSubmit} variant="contained" disabled={resetSubmitting}>
            {resetSubmitting ? 'Resetting…' : 'Reset Password'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog open={Boolean(deleteUser)} onClose={() => setDeleteUser(null)} maxWidth="xs">
        <DialogTitle>Delete User</DialogTitle>
        <DialogContent>
          <Typography>
            Are you sure you want to delete <strong>{deleteUser?.username}</strong>? Deleted usernames cannot be reused.
          </Typography>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteUser(null)} disabled={deleteSubmitting}>
            Cancel
          </Button>
          <Button onClick={handleDeleteConfirm} color="error" variant="contained" disabled={deleteSubmitting}>
            {deleteSubmitting ? 'Deleting…' : 'Delete'}
          </Button>
        </DialogActions>
      </Dialog>

      <Dialog
        open={addUserOpen}
        onClose={() => setAddUserOpen(false)}
        maxWidth="sm"
        fullWidth
      >
        <DialogTitle>Add User</DialogTitle>
        <DialogContent>
          <Stack component="form" id="add-user-form" spacing={2} onSubmit={handleAddUserSubmit} sx={{ mt: 1 }}>
            <TextField
              label="Username"
              name="username"
              value={addUserData.username}
              onChange={handleAddUserFieldChange}
              required
              fullWidth
              autoFocus
            />
            <TextField
              label="Email"
              name="email"
              value={addUserData.email}
              onChange={handleAddUserFieldChange}
              type="email"
              fullWidth
            />
            <TextField
              label="Password"
              name="password"
              value={addUserData.password}
              onChange={handleAddUserFieldChange}
              type="password"
              required
              fullWidth
            />
          </Stack>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setAddUserOpen(false)} disabled={addUserSubmitting}>
            Cancel
          </Button>
          <Button type="submit" form="add-user-form" variant="contained" disabled={addUserSubmitting}>
            {addUserSubmitting ? 'Creating…' : 'Create User'}
          </Button>
        </DialogActions>
      </Dialog>
    </Card>
  );
};

export default LocalUsersSection;
