import React, { useState } from 'react';
import {
  Card,
  CardHeader,
  CardContent,
  TextField,
  Button,
  Alert,
  Stack,
} from '@mui/material';
import api from '../../utils/api';

const ChangePasswordForm = () => {
  const [form, setForm] = useState({ current: '', next: '', confirm: '' });
  const [pending, setPending] = useState(false);
  const [message, setMessage] = useState(null);

  const handleFieldChange = (event) => {
    const { name, value } = event.target;
    setForm((prev) => ({ ...prev, [name]: value }));
  };

  const resetForm = () => {
    setForm({ current: '', next: '', confirm: '' });
  };

  const handleSubmit = async (event) => {
    event.preventDefault();
    setMessage(null);

    if (!form.current || !form.next) {
      setMessage({ type: 'error', text: 'Please fill in all required fields.' });
      return;
    }

    if (form.next !== form.confirm) {
      setMessage({ type: 'error', text: 'New password and confirmation do not match.' });
      return;
    }

    setPending(true);
    try {
      await api.post('/auth/users/me/password', {
        current_password: form.current,
        new_password: form.next,
      });
      setMessage({ type: 'success', text: 'Password updated successfully.' });
      resetForm();
    } catch (error) {
      let detail = error?.response?.data?.detail || 'Unable to change password.';
      if (Array.isArray(detail)) {
        detail = detail
          .map((entry) => (entry?.msg ? entry.msg : JSON.stringify(entry)))
          .join('; ');
      } else if (detail && typeof detail === 'object') {
        detail = detail.msg || detail.message || JSON.stringify(detail);
      }
      setMessage({ type: 'error', text: detail });
    } finally {
      setPending(false);
    }
  };

  return (
    <Card>
      <CardHeader title="Change Password" />
      <CardContent>
        {message && (
          <Alert severity={message.type} sx={{ mb: 2 }}>
            {message.text}
          </Alert>
        )}
        <Stack component="form" spacing={2} onSubmit={handleSubmit}>
          <TextField
            label="Current Password"
            type="password"
            name="current"
            value={form.current}
            onChange={handleFieldChange}
            required
            fullWidth
          />
          <TextField
            label="New Password"
            type="password"
            name="next"
            value={form.next}
            onChange={handleFieldChange}
            required
            fullWidth
          />
          <TextField
            label="Confirm New Password"
            type="password"
            name="confirm"
            value={form.confirm}
            onChange={handleFieldChange}
            required
            fullWidth
          />
          <Button
            variant="contained"
            type="submit"
            disabled={pending}
            sx={{ alignSelf: 'flex-start' }}
          >
            {pending ? 'Saving…' : 'Update Password'}
          </Button>
        </Stack>
      </CardContent>
    </Card>
  );
};

export default ChangePasswordForm;
