import React, { useState, useContext, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import {
  Box,
  Typography,
  Paper,
  Tabs,
  Tab,
  TextField,
  Button,
  Alert,
  Stack,
  Avatar,
  Tooltip,
  IconButton,
  Switch,
  Divider,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogContentText,
  DialogActions,
  Snackbar
} from '@mui/material';
import { BASE_URL } from '../../const';
import { AuthContext } from '../../context/AuthContext';
import { Google, Link, Delete, CheckCircle, Block } from '@mui/icons-material';

const Section = styled(Paper)(({ theme }) => ({
  padding: theme.spacing(3),
  borderRadius: 12,
}));

const AuthFederation = () => {
  const { user, sessionToken } = useContext(AuthContext);
  const isAdmin = !!user && Array.isArray(user.roles) && user.roles.includes('admin');
  const [tab, setTab] = useState(0);
  const [loading, setLoading] = useState(false);
  const [providers, setProviders] = useState([]);
  const [listLoading, setListLoading] = useState(false);
  const [updatingAlias, setUpdatingAlias] = useState(null);
  const [toast, setToast] = useState({ open: false, severity: 'success', message: '' });
  const [confirm, setConfirm] = useState({ open: false, alias: null });
  const [idpEnabled, setIdpEnabled] = useState(true);

  // Google form state
  const [gAlias, setGAlias] = useState('google');
  const [gClientId, setGClientId] = useState('');
  const [gClientSecret, setGClientSecret] = useState('');
  const [gHostedDomain, setGHostedDomain] = useState('');

  // Generic OIDC form state
  const [oAlias, setOAlias] = useState('external-oidc');
  const [oIssuer, setOIssuer] = useState('');
  const [oClientId, setOClientId] = useState('');
  const [oClientSecret, setOClientSecret] = useState('');

  const submit = async () => {
    if (!idpEnabled) {
      return;
    }
    setLoading(true);
    try {
      const headers = {
        'Content-Type': 'application/json',
      };
      if (sessionToken) {
        headers['Authorization'] = `Bearer ${sessionToken}`;
      }

      let payload;
      if (tab === 0) {
        payload = {
          provider: 'google',
          google: {
            alias: gAlias || 'google',
            client_id: gClientId,
            client_secret: gClientSecret,
            hosted_domain: gHostedDomain || undefined,
          },
          ensure_redirects: true,
        };
      } else {
        payload = {
          provider: 'oidc',
          oidc: {
            alias: oAlias || 'external-oidc',
            issuer: oIssuer,
            client_id: oClientId,
            client_secret: oClientSecret,
          },
          ensure_redirects: true,
        };
      }

      const resp = await fetch(`${BASE_URL}/auth/idp/register`, {
        method: 'POST',
        headers,
        credentials: 'include',
        body: JSON.stringify(payload),
      });
      const data = await resp.json();
      if (!resp.ok) {
        throw new Error(data.detail || 'Registration failed');
      }
      const providerName = data?.provider?.displayName || data?.provider?.alias || 'provider';
      showToast('success', `Identity provider ${providerName} ${data.status || 'updated'}`);
      await refreshProviders();
    } catch (e) {
      showToast('error', e.message || 'Failed to register provider');
    } finally {
      setLoading(false);
    }
  };

  const refreshProviders = async () => {
    setListLoading(true);
    try {
      const headers = {};
      if (sessionToken) {
        headers['Authorization'] = `Bearer ${sessionToken}`;
      }
      const resp = await fetch(`${BASE_URL}/auth/idp/providers`, {
        credentials: 'include',
        headers,
      });
      const data = await resp.json();
      if (!resp.ok) throw new Error(data.detail || 'Failed to load providers');
      setProviders(data.providers || []);
      setIdpEnabled(data.idp_management_enabled !== false);
    } catch (e) {
      // keep quiet on listing errors; show empty
      setProviders([]);
      setIdpEnabled(false);
    } finally {
      setListLoading(false);
    }
  };

  const requestDelete = (alias) => {
    if (!idpEnabled) {
      return;
    }
    setConfirm({ open: true, alias });
  };

  const handleDeleteConfirmed = async () => {
    const alias = confirm.alias;
    if (!alias) {
      setConfirm({ open: false, alias: null });
      return;
    }
    if (!idpEnabled) {
      setConfirm({ open: false, alias: null });
      return;
    }
    try {
      const headers = {};
      if (sessionToken) {
        headers['Authorization'] = `Bearer ${sessionToken}`;
      }
      const resp = await fetch(`${BASE_URL}/auth/idp/${encodeURIComponent(alias)}`, {
        method: 'DELETE',
        credentials: 'include',
        headers,
      });
      const data = await resp.json().catch(() => ({}));
      if (!resp.ok) throw new Error(data.detail || 'Delete failed');
      showToast('success', `Deleted provider ${alias}`);
      await refreshProviders();
    } catch (e) {
      showToast('error', e.message || 'Delete failed');
    }
    setConfirm({ open: false, alias: null });
  };

  const handleToggle = async (alias, enabled) => {
    if (!idpEnabled) {
      return;
    }
    setUpdatingAlias(alias);
    try {
      const headers = {
        'Content-Type': 'application/json'
      };
      if (sessionToken) {
        headers['Authorization'] = `Bearer ${sessionToken}`;
      }
      const resp = await fetch(`${BASE_URL}/auth/idp/${encodeURIComponent(alias)}`, {
        method: 'PATCH',
        headers,
        credentials: 'include',
        body: JSON.stringify({ enabled })
      });
      const data = await resp.json();
      if (!resp.ok) throw new Error(data.detail || 'Update failed');
      const providerName = data?.provider?.displayName || alias;
      showToast('success', `${providerName} ${enabled ? 'enabled' : 'disabled'}`);
      await refreshProviders();
    } catch (e) {
      showToast('error', e.message || 'Failed to update provider');
    } finally {
      setUpdatingAlias(null);
    }
  };

  const showToast = (severity, message) => {
    setToast({ open: true, severity, message });
  };

  const providerIcon = (providerId) => {
    if (!providerId) return <Link fontSize="small" />;
    const id = providerId.toLowerCase();
    if (id === 'google') return <Google fontSize="small" color="primary" />;
    if (id.includes('saml')) return <Link fontSize="small" color="success" />;
    return <Link fontSize="small" color="action" />;
  };

  useEffect(() => {
    refreshProviders();
  }, []);

  if (!isAdmin) {
    return (
      <Section sx={{ mt: 2 }}>
        <Alert severity="warning">Admin role required to configure federation.</Alert>
      </Section>
    );
  }

  return (
    <Section>
      <Typography variant="h6" gutterBottom>
        Authentication Providers
      </Typography>
      <Typography variant="body2" color="text.secondary" gutterBottom>
        Configure external identity providers (Google or generic OIDC). Changes apply immediately without restart.
      </Typography>

      {!idpEnabled ? (
        <Box sx={{ my: 2 }}>
          <Alert severity="info">Identity provider management is unavailable in Lite mode.</Alert>
        </Box>
      ) : (
        <>
          <Tabs value={tab} onChange={(e, v) => setTab(v)} sx={{ mb: 2 }}>
            <Tab label="Google" />
            <Tab label="Generic OIDC" />
          </Tabs>

          {tab === 0 && (
            <Stack spacing={2}>
              <TextField label="Alias" value={gAlias} onChange={e => setGAlias(e.target.value)} size="small" />
              <TextField label="Client ID" value={gClientId} onChange={e => setGClientId(e.target.value)} size="small" />
              <TextField label="Client Secret" value={gClientSecret} onChange={e => setGClientSecret(e.target.value)} size="small" type="password" />
              <TextField label="Hosted Domain (optional)" value={gHostedDomain} onChange={e => setGHostedDomain(e.target.value)} size="small" />
              <Button variant="contained" onClick={submit} disabled={loading || !gClientId || !gClientSecret}>
                {loading ? 'Saving…' : 'Register Google'}
              </Button>
            </Stack>
          )}

          {tab === 1 && (
            <Stack spacing={2}>
              <TextField label="Alias" value={oAlias} onChange={e => setOAlias(e.target.value)} size="small" />
              <TextField label="Issuer (https://...)" value={oIssuer} onChange={e => setOIssuer(e.target.value)} size="small" />
              <TextField label="Client ID" value={oClientId} onChange={e => setOClientId(e.target.value)} size="small" />
              <TextField label="Client Secret" value={oClientSecret} onChange={e => setOClientSecret(e.target.value)} size="small" type="password" />
              <Button variant="contained" onClick={submit} disabled={loading || !oIssuer || !oClientId || !oClientSecret}>
                {loading ? 'Saving…' : 'Register OIDC'}
              </Button>
            </Stack>
          )}

          <Box sx={{ mt: 4 }}>
            <Typography variant="subtitle1" gutterBottom>Configured Providers</Typography>
            {listLoading ? (
              <Typography variant="body2" color="text.secondary">Loading…</Typography>
            ) : providers.length === 0 ? (
              <Typography variant="body2" color="text.secondary">No identity providers configured.</Typography>
            ) : (
              <Stack spacing={1}>
                {providers.map(p => (
                  <Paper key={p.alias} sx={{ p: 2, display: 'flex', alignItems: 'center', justifyContent: 'space-between', flexWrap: 'wrap', gap: 2 }}>
                    <Stack direction="row" spacing={2} alignItems="center">
                      <Avatar sx={{ bgcolor: 'background.default', color: 'text.primary', width: 36, height: 36 }}>
                        {providerIcon(p.providerId)}
                      </Avatar>
                      <Box>
                        <Typography variant="body2"><strong>{p.displayName || p.alias}</strong> <em>({p.providerId})</em></Typography>
                        <Stack direction="row" spacing={1} alignItems="center">
                          {p.enabled ? (
                            <CheckCircle fontSize="small" color="success" />
                          ) : (
                            <Block fontSize="small" color="warning" />
                          )}
                          <Typography variant="caption" color="text.secondary">{p.enabled ? 'Enabled' : 'Disabled'}</Typography>
                        </Stack>
                      </Box>
                    </Stack>
                    <Stack direction="row" spacing={1} alignItems="center">
                      <Tooltip title={p.enabled ? 'Disable provider' : 'Enable provider'}>
                        <Switch
                          checked={Boolean(p.enabled)}
                          onChange={(e) => handleToggle(p.alias, e.target.checked)}
                          disabled={updatingAlias === p.alias}
                          inputProps={{ 'aria-label': `Toggle ${p.alias}` }}
                        />
                      </Tooltip>
                      <Divider orientation="vertical" flexItem />
                      <Tooltip title="Delete provider">
                        <span>
                          <IconButton
                            color="error"
                            onClick={() => requestDelete(p.alias)}
                            disabled={updatingAlias === p.alias}
                            size="small"
                          >
                            <Delete fontSize="small" />
                          </IconButton>
                        </span>
                      </Tooltip>
                    </Stack>
                  </Paper>
                ))}
              </Stack>
            )}
          </Box>
        </>
      )}

      <Dialog open={confirm.open} onClose={() => setConfirm({ open: false, alias: null })}>
        <DialogTitle>Delete provider?</DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to delete identity provider <strong>{confirm.alias}</strong>? This cannot be undone.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setConfirm({ open: false, alias: null })}>Cancel</Button>
          <Button color="error" onClick={handleDeleteConfirmed}>Delete</Button>
        </DialogActions>
      </Dialog>

      <Snackbar
        open={toast.open}
        autoHideDuration={4000}
        onClose={() => setToast({ ...toast, open: false })}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'center' }}
      >
        <Alert onClose={() => setToast({ ...toast, open: false })} severity={toast.severity} sx={{ width: '100%' }}>
          {toast.message}
        </Alert>
      </Snackbar>
    </Section>
  );
};

export default AuthFederation;
