/**
 * Tests for ConsentGate component.
 *
 * Key behaviors tested:
 * - Shows loading state while fetching config
 * - Renders children when consent not required
 * - Shows consent dialog when consent required and not accepted
 * - Does NOT mount children until consent is accepted (security requirement)
 * - Accept button triggers consent acceptance
 * - Dialog cannot be dismissed with escape key
 */
import React, { useEffect } from 'react';
import { render, screen, fireEvent, waitFor } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import theme from '../../../Theme';
import ConsentGate from '../ConsentGate';
import { SecurityContext } from '../../../context/SecurityContext';

// Mock wrapper for theme
// eslint-disable-next-line react/prop-types
const ThemeWrapper = ({ children }) => (
  <ThemeProvider theme={theme}>{children}</ThemeProvider>
);

// Helper to render with security context
const renderWithSecurityContext = (ui, securityValue) => {
  return render(
    <ThemeWrapper>
      <SecurityContext.Provider value={securityValue}>
        {ui}
      </SecurityContext.Provider>
    </ThemeWrapper>
  );
};

// Default security context mock
const createSecurityContext = (overrides = {}) => ({
  config: {
    consent_enabled: false,
    consent_content: null,
    consent_button_label: null,
    banner_enabled: false,
    banner_top_text: null,
    banner_top_color: null,
    banner_bottom_text: null,
    banner_bottom_color: null,
    ...overrides.config,
  },
  loading: false,
  error: null,
  configLoadFailed: false,
  consentAccepted: false,
  acceptConsent: jest.fn(),
  declineConsent: jest.fn(),
  shouldShowConsentGate: false,
  shouldShowBanners: false,
  refreshConfig: jest.fn(),
  ...overrides,
});

describe('ConsentGate', () => {
  describe('loading state', () => {
    it('shows loading spinner while fetching config', () => {
      const securityValue = createSecurityContext({ loading: true });

      renderWithSecurityContext(
        <ConsentGate>
          <div data-testid="child-content">Child Content</div>
        </ConsentGate>,
        securityValue
      );

      // Should show loading spinner
      expect(screen.getByRole('progressbar')).toBeInTheDocument();
      // Should not show children
      expect(screen.queryByTestId('child-content')).not.toBeInTheDocument();
    });
  });

  describe('consent not required', () => {
    it('renders children when consent is disabled', () => {
      const securityValue = createSecurityContext({
        config: { consent_enabled: false },
        shouldShowConsentGate: false,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div data-testid="child-content">Child Content</div>
        </ConsentGate>,
        securityValue
      );

      expect(screen.getByTestId('child-content')).toBeInTheDocument();
      expect(screen.getByText('Child Content')).toBeInTheDocument();
    });

    it('renders children when consent already accepted', () => {
      const securityValue = createSecurityContext({
        config: { consent_enabled: true, consent_content: '<p>Terms</p>' },
        consentAccepted: true,
        shouldShowConsentGate: false,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div data-testid="child-content">Child Content</div>
        </ConsentGate>,
        securityValue
      );

      expect(screen.getByTestId('child-content')).toBeInTheDocument();
    });
  });

  describe('consent required', () => {
    it('shows consent dialog when consent required and not accepted', () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Please accept terms</p>',
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div data-testid="child-content">Child Content</div>
        </ConsentGate>,
        securityValue
      );

      // Should show dialog
      expect(screen.getByRole('dialog')).toBeInTheDocument();
      expect(screen.getByText('System Access Agreement')).toBeInTheDocument();
      expect(screen.getByRole('button', { name: /accept/i })).toBeInTheDocument();
    });

    it('does NOT mount children when consent gate is shown', () => {
      // This is a critical security test - children should not mount pre-consent
      // because their effects could make API calls
      const childEffectFn = jest.fn();

      const ChildWithEffect = () => {
        useEffect(() => {
          childEffectFn();
        }, []);
        return <div data-testid="child-content">Child Content</div>;
      };

      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Terms</p>',
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <ChildWithEffect />
        </ConsentGate>,
        securityValue
      );

      // Child should NOT be mounted
      expect(screen.queryByTestId('child-content')).not.toBeInTheDocument();
      // Child effect should NOT have run
      expect(childEffectFn).not.toHaveBeenCalled();
    });

    it('displays default message when no consent content provided', () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: null,
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      expect(screen.getByText('Please accept the terms to continue.')).toBeInTheDocument();
    });
  });

  describe('accept button', () => {
    it('calls acceptConsent when Accept button is clicked', () => {
      const acceptConsent = jest.fn();
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Terms</p>',
        },
        shouldShowConsentGate: true,
        acceptConsent,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      const acceptButton = screen.getByRole('button', { name: /accept/i });
      fireEvent.click(acceptButton);

      expect(acceptConsent).toHaveBeenCalledTimes(1);
    });

    it('displays custom button label when configured', () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Terms</p>',
          consent_button_label: 'I Agree',
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      expect(screen.getByRole('button', { name: 'I Agree' })).toBeInTheDocument();
    });

    it('uses default "Accept" label when no custom label configured', () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Terms</p>',
          consent_button_label: null,
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      expect(screen.getByRole('button', { name: 'Accept' })).toBeInTheDocument();
    });
  });

  describe('dialog behavior', () => {
    it('dialog has disableEscapeKeyDown prop', () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Terms</p>',
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      const dialog = screen.getByRole('dialog');
      expect(dialog).toBeInTheDocument();

      // Try pressing Escape - dialog should still be open
      fireEvent.keyDown(dialog, { key: 'Escape', code: 'Escape' });
      expect(screen.getByRole('dialog')).toBeInTheDocument();
    });
  });

  describe('HTML content sanitization', () => {
    it('renders sanitized HTML content', async () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<h2>Terms</h2><p>Please read these terms.</p>',
        },
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      // Wait for content to render
      await waitFor(() => {
        expect(screen.getByText('Terms')).toBeInTheDocument();
        expect(screen.getByText('Please read these terms.')).toBeInTheDocument();
      });
    });

    it('strips dangerous HTML tags', async () => {
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Safe content</p><script>alert("XSS")</script>',
        },
        shouldShowConsentGate: true,
      });

      const { container } = renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      // Wait for content to render
      await waitFor(() => {
        expect(screen.getByText('Safe content')).toBeInTheDocument();
      });

      // Script tag should be stripped
      expect(container.querySelector('script')).not.toBeInTheDocument();
    });
  });

  describe('fail-closed behavior (config load failure)', () => {
    it('shows consent gate when config fails to load (fail-closed security)', () => {
      // When config fetch fails, consent should be enabled to prevent bypass
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true, // Fail-closed sets this to true
          consent_content:
            '<p><strong>System Configuration Unavailable</strong></p>' +
            '<p>Unable to load security configuration from the server.</p>',
          consent_button_label: 'Retry',
        },
        error: 'Failed to fetch security config: Network error',
        configLoadFailed: true,
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div data-testid="child-content">Child Content</div>
        </ConsentGate>,
        securityValue
      );

      // Should show dialog with error message
      expect(screen.getByRole('dialog')).toBeInTheDocument();
      expect(screen.getByText('System Configuration Unavailable')).toBeInTheDocument();
      // Children should NOT be mounted (security requirement)
      expect(screen.queryByTestId('child-content')).not.toBeInTheDocument();
    });

    it('shows Retry button when config fails to load', () => {
      const acceptConsent = jest.fn();
      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Configuration unavailable</p>',
          consent_button_label: 'Retry',
        },
        configLoadFailed: true,
        shouldShowConsentGate: true,
        acceptConsent,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <div>Children</div>
        </ConsentGate>,
        securityValue
      );

      // Should show Retry button
      const retryButton = screen.getByRole('button', { name: 'Retry' });
      expect(retryButton).toBeInTheDocument();

      // Clicking should call acceptConsent (which triggers refresh in fail state)
      fireEvent.click(retryButton);
      expect(acceptConsent).toHaveBeenCalledTimes(1);
    });

    it('does NOT mount children when config load fails (security requirement)', () => {
      // Critical: Children must not mount during backend failure
      // This prevents users from bypassing consent during outages
      const childEffectFn = jest.fn();

      const ChildWithEffect = () => {
        useEffect(() => {
          childEffectFn();
        }, []);
        return <div data-testid="child-content">Child Content</div>;
      };

      const securityValue = createSecurityContext({
        config: {
          consent_enabled: true,
          consent_content: '<p>Config unavailable</p>',
        },
        configLoadFailed: true,
        shouldShowConsentGate: true,
      });

      renderWithSecurityContext(
        <ConsentGate>
          <ChildWithEffect />
        </ConsentGate>,
        securityValue
      );

      // Child should NOT be mounted
      expect(screen.queryByTestId('child-content')).not.toBeInTheDocument();
      // Child effect should NOT have run
      expect(childEffectFn).not.toHaveBeenCalled();
    });
  });
});
