/**
 * Tests for Banner component and related utilities.
 *
 * Key behaviors tested:
 * - Returns null when banners disabled
 * - Renders top and bottom banners correctly
 * - Displays correct text and color from config
 * - Uses default color when not specified
 * - Has proper accessibility attributes
 * - BannerSpacer renders correctly
 * - useBannerConfig hook returns correct values
 */
import React from 'react';
import { render, screen, renderHook } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import theme from '../../../Theme';
import Banner, { BANNER_HEIGHT, BannerSpacer, useBannerConfig } from '../Banner';
import { SecurityContext } from '../../../context/SecurityContext';

// Mock wrapper for theme
// eslint-disable-next-line react/prop-types
const ThemeWrapper = ({ children }) => (
  <ThemeProvider theme={theme}>{children}</ThemeProvider>
);

// Helper to render with security context
const renderWithSecurityContext = (ui, securityValue) => {
  return render(
    <ThemeWrapper>
      <SecurityContext.Provider value={securityValue}>
        {ui}
      </SecurityContext.Provider>
    </ThemeWrapper>
  );
};

// Default security context mock
const createSecurityContext = (overrides = {}) => ({
  config: {
    consent_enabled: false,
    consent_content: null,
    consent_button_label: null,
    banner_enabled: false,
    banner_top_text: null,
    banner_top_color: null,
    banner_bottom_text: null,
    banner_bottom_color: null,
    ...overrides.config,
  },
  loading: false,
  error: null,
  consentAccepted: false,
  acceptConsent: jest.fn(),
  declineConsent: jest.fn(),
  shouldShowConsentGate: false,
  shouldShowBanners: false,
  refreshConfig: jest.fn(),
  ...overrides,
});

describe('Banner', () => {
  describe('BANNER_HEIGHT constant', () => {
    it('exports BANNER_HEIGHT as 24 pixels', () => {
      expect(BANNER_HEIGHT).toBe(24);
    });
  });

  describe('when banners disabled', () => {
    it('returns null when shouldShowBanners is false', () => {
      const securityValue = createSecurityContext({
        shouldShowBanners: false,
      });

      const { container } = renderWithSecurityContext(
        <Banner position="top" />,
        securityValue
      );

      expect(container.firstChild).toBeNull();
    });
  });

  describe('when banners enabled', () => {
    it('renders top banner with correct text', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'SECRET',
          banner_top_color: '#C8102E',
          banner_bottom_text: 'SECRET',
          banner_bottom_color: '#C8102E',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      expect(banner).toBeInTheDocument();
      expect(screen.getByText('SECRET')).toBeInTheDocument();
    });

    it('renders bottom banner at correct position', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'UNCLASSIFIED',
          banner_top_color: '#00A651',
          banner_bottom_text: 'UNCLASSIFIED',
          banner_bottom_color: '#00A651',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="bottom" />, securityValue);

      const banner = screen.getByRole('banner');
      expect(banner).toBeInTheDocument();
      expect(screen.getByText('UNCLASSIFIED')).toBeInTheDocument();
    });

    it('renders different text for top and bottom banners', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'TOP SECRET',
          banner_top_color: '#FF6600',
          banner_bottom_text: 'HANDLE VIA COMINT',
          banner_bottom_color: '#FFCC00',
        },
        shouldShowBanners: true,
      });

      const { rerender } = renderWithSecurityContext(<Banner position="top" />, securityValue);
      expect(screen.getByText('TOP SECRET')).toBeInTheDocument();

      rerender(
        <ThemeProvider theme={theme}>
          <SecurityContext.Provider value={securityValue}>
            <Banner position="bottom" />
          </SecurityContext.Provider>
        </ThemeProvider>
      );
      expect(screen.getByText('HANDLE VIA COMINT')).toBeInTheDocument();
    });

    it('uses configured banner color', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'CUI',
          banner_top_color: '#502B85',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      // Check that the background color style is applied
      expect(banner).toHaveStyle({ backgroundColor: '#502B85' });
    });

    it('uses default color when banner_top_color is null', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'TEST',
          banner_top_color: null,
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      // Default color is #C8102E
      expect(banner).toHaveStyle({ backgroundColor: '#C8102E' });
    });
  });

  describe('accessibility', () => {
    it('has role="banner" attribute', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'SECRET',
          banner_top_color: '#C8102E',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      expect(screen.getByRole('banner')).toBeInTheDocument();
    });

    it('has correct aria-label with classification text', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'TOP SECRET',
          banner_top_color: '#FF6600',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      expect(banner).toHaveAttribute('aria-label', 'Classification: TOP SECRET');
    });
  });

  describe('styling', () => {
    it('has fixed positioning', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'SECRET',
          banner_top_color: '#C8102E',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      expect(banner).toHaveStyle({ position: 'fixed' });
    });

    it('has correct height', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'SECRET',
          banner_top_color: '#C8102E',
        },
        shouldShowBanners: true,
      });

      renderWithSecurityContext(<Banner position="top" />, securityValue);

      const banner = screen.getByRole('banner');
      expect(banner).toHaveStyle({ height: '24px' });
    });
  });
});

describe('BannerSpacer', () => {
  describe('when banners disabled', () => {
    it('returns null', () => {
      const securityValue = createSecurityContext({
        shouldShowBanners: false,
      });

      const { container } = renderWithSecurityContext(
        <BannerSpacer />,
        securityValue
      );

      expect(container.firstChild).toBeNull();
    });
  });

  describe('when banners enabled', () => {
    it('renders spacer with correct height', () => {
      const securityValue = createSecurityContext({
        config: {
          banner_enabled: true,
          banner_top_text: 'SECRET',
          banner_top_color: '#C8102E',
        },
        shouldShowBanners: true,
      });

      const { container } = renderWithSecurityContext(
        <BannerSpacer />,
        securityValue
      );

      const spacer = container.firstChild;
      expect(spacer).toBeInTheDocument();
      expect(spacer).toHaveStyle({ height: '24px' });
    });
  });
});

describe('useBannerConfig', () => {
  const createWrapper = (securityValue) => {
    // eslint-disable-next-line react/display-name, react/prop-types
    return ({ children }) => (
      <ThemeWrapper>
        <SecurityContext.Provider value={securityValue}>
          {children}
        </SecurityContext.Provider>
      </ThemeWrapper>
    );
  };

  it('returns enabled: false when banners disabled', () => {
    const securityValue = createSecurityContext({
      shouldShowBanners: false,
    });

    const { result } = renderHook(() => useBannerConfig(), {
      wrapper: createWrapper(securityValue),
    });

    expect(result.current.enabled).toBe(false);
    expect(result.current.height).toBe(BANNER_HEIGHT);
  });

  it('returns all banner values when no position specified', () => {
    const securityValue = createSecurityContext({
      config: {
        banner_enabled: true,
        banner_top_text: 'TOP SECRET',
        banner_top_color: '#FF6600',
        banner_bottom_text: 'HANDLE VIA COMINT',
        banner_bottom_color: '#FFCC00',
      },
      shouldShowBanners: true,
    });

    const { result } = renderHook(() => useBannerConfig(), {
      wrapper: createWrapper(securityValue),
    });

    expect(result.current.enabled).toBe(true);
    expect(result.current.height).toBe(24);
    expect(result.current.topText).toBe('TOP SECRET');
    expect(result.current.topColor).toBe('#FF6600');
    expect(result.current.bottomText).toBe('HANDLE VIA COMINT');
    expect(result.current.bottomColor).toBe('#FFCC00');
  });

  it('returns position-specific values when position is top', () => {
    const securityValue = createSecurityContext({
      config: {
        banner_enabled: true,
        banner_top_text: 'TOP SECRET',
        banner_top_color: '#FF6600',
        banner_bottom_text: 'BOTTOM TEXT',
        banner_bottom_color: '#00FF00',
      },
      shouldShowBanners: true,
    });

    const { result } = renderHook(() => useBannerConfig('top'), {
      wrapper: createWrapper(securityValue),
    });

    expect(result.current.enabled).toBe(true);
    expect(result.current.text).toBe('TOP SECRET');
    expect(result.current.color).toBe('#FF6600');
  });

  it('returns position-specific values when position is bottom', () => {
    const securityValue = createSecurityContext({
      config: {
        banner_enabled: true,
        banner_top_text: 'TOP SECRET',
        banner_top_color: '#FF6600',
        banner_bottom_text: 'BOTTOM TEXT',
        banner_bottom_color: '#00FF00',
      },
      shouldShowBanners: true,
    });

    const { result } = renderHook(() => useBannerConfig('bottom'), {
      wrapper: createWrapper(securityValue),
    });

    expect(result.current.enabled).toBe(true);
    expect(result.current.text).toBe('BOTTOM TEXT');
    expect(result.current.color).toBe('#00FF00');
  });
});
