/**
 * ConsentGate Component
 *
 * A blocking modal that appears before login when consent is enabled.
 * Users must accept the consent terms to proceed.
 * If declined (by navigating away), they cannot access the application.
 *
 * SECURITY NOTE: This component renders HTML content using dangerouslySetInnerHTML.
 * This is SAFE because:
 * 1. Content originates from admin-controlled server file ($KAMIWAZA_ROOT/config/security/consent.html)
 * 2. Content is sanitized with DOMPurify using a strict tag/attribute whitelist
 * 3. This is the standard pattern for rendering trusted, formatted consent content
 */
import React from 'react';
import PropTypes from 'prop-types';
import DOMPurify from 'dompurify';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  CircularProgress,
  Typography,
} from '@mui/material';
import { useSecurity } from '../../context/SecurityContext';

/**
 * Configure DOMPurify to add security attributes to links.
 * This hook runs after sanitization to ensure all links have
 * target="_blank" and rel="noopener noreferrer" for security.
 */
DOMPurify.addHook('afterSanitizeAttributes', (node) => {
  if (node.tagName === 'A') {
    node.setAttribute('target', '_blank');
    node.setAttribute('rel', 'noopener noreferrer');
  }
});

/**
 * Sanitize HTML content with DOMPurify using a strict whitelist.
 * Only allows basic formatting tags needed for consent content.
 *
 * @param {string} htmlContent - Raw HTML content to sanitize
 * @returns {string} Sanitized HTML safe for rendering
 */
const sanitizeHtml = (htmlContent) => {
  if (!htmlContent) return '';

  return DOMPurify.sanitize(htmlContent, {
    // Only allow basic formatting tags
    ALLOWED_TAGS: [
      'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
      'p', 'br', 'hr',
      'ul', 'ol', 'li',
      'strong', 'b', 'em', 'i', 'u',
      'a', 'span', 'div',
    ],
    // Only allow safe attributes
    ALLOWED_ATTR: ['href', 'target', 'rel', 'class'],
    FORCE_BODY: true,
  });
};

/**
 * ConsentGate wraps children and blocks access until consent is accepted.
 *
 * When consent_enabled is true and user hasn't accepted:
 * - Shows a full-screen modal with consent content
 * - Single "Accept" button to proceed
 * - No decline button - user navigates away to decline
 * - Modal cannot be dismissed with Escape key
 *
 * Once accepted (stored in sessionStorage), children render normally.
 */
const ConsentGate = ({ children }) => {
  const {
    config,
    loading,
    acceptConsent,
    shouldShowConsentGate,
  } = useSecurity();

  // Show loading spinner while fetching config
  if (loading) {
    return (
      <Box
        sx={{
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
          minHeight: '100vh',
          bgcolor: 'background.default',
        }}
      >
        <CircularProgress />
      </Box>
    );
  }

  // If consent not required or already accepted, render children
  if (!shouldShowConsentGate) {
    return children;
  }

  // Sanitize admin-controlled HTML content before rendering
  const sanitizedContent = sanitizeHtml(config.consent_content);

  // SECURITY: Do NOT mount children until consent is accepted.
  // This prevents React effects and API calls from running pre-consent.
  return (
    <Dialog
      open={true}
      onClose={() => {}} // No close on backdrop click
      disableEscapeKeyDown // Cannot dismiss with Escape key
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: {
          maxHeight: '80vh',
          borderRadius: 2,
        },
      }}
    >
      <DialogTitle
        sx={{
          borderBottom: 1,
          borderColor: 'divider',
          pb: 2,
        }}
      >
        <Typography variant="h5" component="div" fontWeight="bold">
          System Access Agreement
        </Typography>
      </DialogTitle>

      <DialogContent sx={{ mt: 2, overflowY: 'auto' }}>
        {sanitizedContent ? (
          <ConsentContent html={sanitizedContent} />
        ) : (
          <Typography color="text.secondary">
            Please accept the terms to continue.
          </Typography>
        )}
      </DialogContent>

      <DialogActions
        sx={{
          borderTop: 1,
          borderColor: 'divider',
          px: 3,
          py: 2,
        }}
      >
        <Button
          variant="contained"
          color="primary"
          onClick={acceptConsent}
          size="large"
          sx={{ minWidth: 120 }}
        >
          {config.consent_button_label || 'Accept'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};

/**
 * ConsentContent renders sanitized HTML content with proper styling.
 * Content is already sanitized by sanitizeHtml() before reaching this component.
 *
 * SECURITY: This uses dangerouslySetInnerHTML which is safe here because:
 * - Content is sanitized with DOMPurify strict whitelist
 * - Source is admin-controlled server file, not user input
 */
const ConsentContent = ({ html }) => {
  // Create element ref for setting innerHTML safely
  const containerRef = React.useRef(null);

  React.useEffect(() => {
    if (containerRef.current && html) {
      // Set sanitized HTML content
      containerRef.current.innerHTML = html;
    }
  }, [html]);

  return (
    <Box
      ref={containerRef}
      sx={{
        '& h1, & h2, & h3, & h4, & h5, & h6': {
          mt: 2,
          mb: 1,
        },
        '& p': {
          mb: 2,
        },
        '& ul, & ol': {
          pl: 3,
          mb: 2,
        },
        '& li': {
          mb: 0.5,
        },
        '& a': {
          color: 'primary.main',
          textDecoration: 'underline',
        },
      }}
    />
  );
};

ConsentContent.propTypes = {
  html: PropTypes.string.isRequired,
};

ConsentGate.propTypes = {
  children: PropTypes.node.isRequired,
};

export default ConsentGate;
