/**
 * Banner Component
 *
 * Displays a fixed classification banner at the top or bottom of the page.
 * Used to show security classification markings (e.g., UNCLASSIFIED, SECRET).
 */
import React from 'react';
import PropTypes from 'prop-types';
import { Box, Typography } from '@mui/material';
import { useSecurity } from '../../context/SecurityContext';

/**
 * Banner height in pixels - used for positioning and body padding.
 * Exported for use by other components that need to account for banner space.
 */
export const BANNER_HEIGHT = 24;

/**
 * Banner displays a fixed banner with classification text.
 *
 * @param {Object} props
 * @param {string} props.position - 'top' or 'bottom' position for the banner
 */
const Banner = ({ position }) => {
  const { config, shouldShowBanners } = useSecurity();

  // Don't render if banners are disabled or no text configured
  if (!shouldShowBanners) {
    return null;
  }

  // Get position-specific text and color
  const banner_text = position === 'top' ? config.banner_top_text : config.banner_bottom_text;
  const banner_color = position === 'top' ? config.banner_top_color : config.banner_bottom_color;

  return (
    <Box
      role="banner"
      aria-label={`Classification: ${banner_text}`}
      sx={{
        position: 'fixed',
        left: 0,
        right: 0,
        [position]: 0,
        height: BANNER_HEIGHT,
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        backgroundColor: banner_color || '#C8102E',
        zIndex: 99999, // Above everything including modals
        boxShadow: position === 'top'
          ? '0 2px 4px rgba(0,0,0,0.2)'
          : '0 -2px 4px rgba(0,0,0,0.2)',
      }}
    >
      <Typography
        variant="body2"
        sx={{
          color: '#ffffff',
          fontWeight: 600,
          fontSize: '12px',
          letterSpacing: '0.5px',
          textTransform: 'uppercase',
          textShadow: '0 1px 2px rgba(0,0,0,0.3)',
          fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
        }}
      >
        {banner_text}
      </Typography>
    </Box>
  );
};

Banner.propTypes = {
  position: PropTypes.oneOf(['top', 'bottom']).isRequired,
};

/**
 * BannerSpacer adds padding to account for fixed banners.
 * Use this component to prevent content from being hidden behind banners.
 */
export const BannerSpacer = () => {
  const { shouldShowBanners } = useSecurity();

  if (!shouldShowBanners) {
    return null;
  }

  return <Box sx={{ height: BANNER_HEIGHT }} />;
};

/**
 * Hook to get banner configuration for custom implementations.
 * @param {string} position - Optional 'top' or 'bottom' to get position-specific config
 * @returns {Object} Banner configuration including height and enabled state
 */
export const useBannerConfig = (position = null) => {
  const { shouldShowBanners, config } = useSecurity();

  // If position specified, return position-specific values
  if (position === 'top' || position === 'bottom') {
    return {
      enabled: shouldShowBanners,
      height: BANNER_HEIGHT,
      text: position === 'top' ? config.banner_top_text : config.banner_bottom_text,
      color: position === 'top' ? config.banner_top_color : config.banner_bottom_color,
    };
  }

  // Return top values as default for backwards compatibility
  return {
    enabled: shouldShowBanners,
    height: BANNER_HEIGHT,
    topText: config.banner_top_text,
    topColor: config.banner_top_color,
    bottomText: config.banner_bottom_text,
    bottomColor: config.banner_bottom_color,
  };
};

export default Banner;
