import React, { useState, useEffect } from 'react';
import { Typography, Box, Paper, CircularProgress, Alert, Chip, Grid, IconButton, Tooltip, Button } from '@mui/material';
import { styled } from '@mui/material/styles';
import { Refresh as RefreshIcon, ExpandMore, ExpandLess, CalendarToday, LinkOutlined, FirstPage, LastPage, NavigateBefore, NavigateNext } from '@mui/icons-material';
import ReactMarkdown from 'react-markdown';
import axios from 'axios';

const ContentCard = styled(Paper)(({ theme }) => ({
  padding: theme.spacing(3),
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: theme.spacing(1),
  marginBottom: theme.spacing(3),
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.05) 0%, rgba(255, 255, 255, 0.02) 100%)',
  backdropFilter: 'blur(10px)',
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.1)',
  height: '100%',
}));

const QuadrantCard = styled(Paper)(({ theme }) => ({
  padding: theme.spacing(2),
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(0, 192, 127, 0.3)',
  borderRadius: theme.spacing(1),
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.05) 0%, rgba(255, 255, 255, 0.02) 100%)',
  backdropFilter: 'blur(10px)',
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.1)',
  minHeight: '200px',
  display: 'flex',
  flexDirection: 'column',
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  marginBottom: theme.spacing(2),
  fontWeight: 600,
  position: 'relative',
  color: theme.palette.primary.main,
  '&::after': {
    content: '""',
    position: 'absolute',
    bottom: -8,
    left: 0,
    width: 50,
    height: 3,
    backgroundColor: theme.palette.primary.main,
    borderRadius: 2,
  },
}));

const FeaturedBadge = styled(Chip)(({ theme }) => ({
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.primary.contrastText,
  fontSize: '0.75rem',
  height: 20,
  marginLeft: theme.spacing(1),
}));

const CategoryChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.1)',
  color: theme.palette.text.secondary,
  fontSize: '0.7rem',
  height: 18,
  marginRight: theme.spacing(0.5),
}));

const HeaderContainer = styled(Box)({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  marginBottom: '2rem',
});

const RefreshButton = styled(IconButton)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  border: '1px solid rgba(0, 192, 127, 0.3)',
  color: theme.palette.primary.main,
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.2)',
  },
}));

const PaginationControls = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'center',
  alignItems: 'center',
  gap: theme.spacing(0.5),
  marginTop: theme.spacing(2),
  '& .MuiIconButton-root': {
    padding: 4,
    fontSize: '0.9rem',
    border: '1px solid rgba(0, 192, 127, 0.2)',
    borderRadius: 4,
    '&:hover': {
      backgroundColor: 'rgba(0, 192, 127, 0.1)',
    },
    '&:disabled': {
      opacity: 0.3,
      border: '1px solid rgba(255, 255, 255, 0.1)',
    },
  },
}));

const PageInfo = styled(Typography)(({ theme }) => ({
  fontSize: '0.75rem',
  color: theme.palette.text.secondary,
  minWidth: 60,
  textAlign: 'center',
  margin: '0 4px',
}));

// Static fallback content
const STATIC_NEWS_DATA = {
  metadata: {
    version: "1.0",
    lastUpdated: new Date().toISOString(),
    totalItems: 1
  },
  categories: ["updates", "news", "events", "general"],
  quadrants: {
    updates: {
      title: "Updates",
      position: "top-left",
      items: []
    },
    news: {
      title: "Latest News",
      position: "top-right",
      items: [
        {
          id: "check-back-static",
          title: "Loading News",
          date: new Date().toISOString().split('T')[0],
          category: "news",
          content: "Check back for the latest news",
          featured: false,
          author: "Kamiwaza Team",
          tags: ["status"]
        }
      ]
    },
    events: {
      title: "Events",
      position: "bottom-left",
      items: []
    },
    general: {
      title: "General",
      position: "bottom-right",
      items: []
    }
  }
};

// Cache configuration
const CACHE_KEY = 'kamiwaza_news_cache';
const CACHE_EXPIRY_HOURS = 4;
const CACHE_EXPIRY_MS = CACHE_EXPIRY_HOURS * 60 * 60 * 1000; // 4 hours in milliseconds
const CONTENT_PREVIEW_LENGTH = 200; // Longer preview for news page
const ITEMS_PER_PAGE = 3; // Show 3 items per quadrant page

const News = ({ variant = 'full' }) => {
  // variant can be 'full' for /news page or 'preview' for homepage
  const [newsData, setNewsData] = useState(STATIC_NEWS_DATA);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [lastFetched, setLastFetched] = useState(null);
  const [usingCachedData, setUsingCachedData] = useState(false);
  const [expandedItems, setExpandedItems] = useState({});
  const [quadrantPages, setQuadrantPages] = useState({
    updates: 0,
    news: 0,
    events: 0,
    general: 0
  });

  const getCachedData = () => {
    try {
      const cached = localStorage.getItem(CACHE_KEY);
      if (cached) {
        const parsedCache = JSON.parse(cached);
        const now = Date.now();
        if (now - parsedCache.timestamp < CACHE_EXPIRY_MS) {
          return {
            data: parsedCache.data,
            timestamp: parsedCache.timestamp,
            isValid: true
          };
        }
      }
    } catch (e) {
      console.warn('Failed to parse cached news data:', e);
    }
    return { isValid: false };
  };

  const setCachedData = (data) => {
    try {
      const cacheData = {
        data: data,
        timestamp: Date.now()
      };
      localStorage.setItem(CACHE_KEY, JSON.stringify(cacheData));
    } catch (e) {
      console.warn('Failed to cache news data:', e);
    }
  };

  const fetchNews = async (forceRefresh = false) => {
    // Check cache first unless force refresh
    if (!forceRefresh) {
      const cached = getCachedData();
      if (cached.isValid) {
        setNewsData(cached.data);
        setLastFetched(new Date(cached.timestamp));
        setUsingCachedData(true);
        return;
      }
    }

    setLoading(true);
    setError(null);
    setUsingCachedData(false);

    try {
      // Fetch structured news data from S3 endpoint
      const newsResponse = await axios.get('https://info.kamiwaza.ai/news/news.json');
      if (newsResponse.data) {
        setNewsData(newsResponse.data);
        setCachedData(newsResponse.data);
        setLastFetched(new Date());
      }
    } catch (err) {
      console.error('Error fetching news:', err);
      setError('Failed to fetch latest news. Showing cached content.');
      
      // Try to use cached data even if expired
      const cached = getCachedData();
      if (cached.data) {
        setNewsData(cached.data);
        setLastFetched(new Date(cached.timestamp));
        setUsingCachedData(true);
      } else {
        // Fall back to static data
        setNewsData(STATIC_NEWS_DATA);
      }
    } finally {
      setLoading(false);
    }
  };

  const handleRefresh = () => {
    fetchNews(true); // Force refresh
  };

  useEffect(() => {
    fetchNews();
  }, []);

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    const today = new Date();
    const diffTime = today - date;
    const diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays === 0) return 'Today';
    if (diffDays === 1) return 'Yesterday';
    if (diffDays < 7) return `${diffDays} days ago`;
    if (diffDays < 30) return `${Math.floor(diffDays / 7)} weeks ago`;
    
    return date.toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
  };

  const toggleItemExpanded = (itemId) => {
    setExpandedItems(prev => ({ ...prev, [itemId]: !prev[itemId] }));
  };

  const formatLastFetched = (date) => {
    if (!date) return 'Never';
    return date.toLocaleString('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };


  const handlePageChange = (quadrantKey, newPage) => {
    setQuadrantPages(prev => ({
      ...prev,
      [quadrantKey]: newPage
    }));
  };

  const renderQuadrantContent = (quadrant, quadrantKey) => {
    if (!quadrant || !quadrant.items || quadrant.items.length === 0) {
      return (
        <Box sx={{ textAlign: 'center', py: 3 }}>
          <Typography variant="body2" sx={{ color: 'text.secondary', fontStyle: 'italic', mb: 1 }}>
            No items available yet
          </Typography>
          <Typography variant="caption" sx={{ color: 'text.secondary' }}>
            Check back soon for updates!
          </Typography>
        </Box>
      );
    }

    const currentPage = quadrantPages[quadrantKey] || 0;
    const totalPages = Math.ceil(quadrant.items.length / ITEMS_PER_PAGE);
    const startIndex = currentPage * ITEMS_PER_PAGE;
    const endIndex = startIndex + ITEMS_PER_PAGE;
    const pageItems = quadrant.items.slice(startIndex, endIndex);

    return (
      <Box sx={{ flex: 1, display: 'flex', flexDirection: 'column' }}>
        <Box sx={{ flex: 1 }}>
          {pageItems.map((item, index) => {
          const isExpanded = expandedItems[item.id];
          const hasLongContent = item.content && item.content.length > CONTENT_PREVIEW_LENGTH;
          const displayContent = hasLongContent && !isExpanded 
            ? item.content.substring(0, CONTENT_PREVIEW_LENGTH) + '...'
            : item.content;
          
          return (
          <Box key={item.id} sx={{ mb: 2, pb: 2, borderBottom: index < quadrant.items.length - 1 ? '1px solid rgba(255, 255, 255, 0.1)' : 'none' }}>
            <Box sx={{ display: 'flex', alignItems: 'start', justifyContent: 'space-between', mb: 1 }}>
              <Box sx={{ flex: 1 }}>
                <Typography variant="subtitle1" sx={{ fontWeight: 600, fontSize: '1rem', mb: 0.5 }}>
                  {item.title}
                </Typography>
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1.5, flexWrap: 'wrap' }}>
                  {item.date && (
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                      <CalendarToday sx={{ fontSize: '0.85rem', color: 'text.secondary' }} />
                      <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                        {formatDate(item.date)}
                      </Typography>
                    </Box>
                  )}
                  {item.author && (
                    <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                      by {item.author}
                    </Typography>
                  )}
                </Box>
              </Box>
              {item.featured && <FeaturedBadge label="Featured" size="small" />}
            </Box>
            {item.content && item.content.trim() && (
              <>
                <Box sx={{ color: 'text.secondary', fontSize: '0.9rem', lineHeight: 1.6, mb: 1 }}>
                  <ReactMarkdown
                    components={{
                      p: ({children}) => <Typography variant="body2" sx={{ color: 'inherit', fontSize: 'inherit', mb: 1 }}>{children}</Typography>,
                      a: ({children, href}) => (
                        <Box component="span" sx={{ display: 'inline-flex', alignItems: 'center', gap: 0.5 }}>
                          <LinkOutlined sx={{ fontSize: '0.9rem' }} />
                          <a href={href} target="_blank" rel="noopener noreferrer" style={{ color: '#00c07f', textDecoration: 'none' }}>
                            {children}
                          </a>
                        </Box>
                      ),
                      strong: ({children}) => <strong style={{ color: '#00c07f' }}>{children}</strong>,
                      ul: ({children}) => <ul style={{ marginTop: 4, marginBottom: 4, paddingLeft: 20 }}>{children}</ul>,
                      li: ({children}) => <li style={{ marginBottom: 2 }}>{children}</li>,
                      code: ({children}) => <code style={{ backgroundColor: 'rgba(255,255,255,0.1)', padding: '2px 4px', borderRadius: 3 }}>{children}</code>,
                    }}
                  >
                    {displayContent}
                  </ReactMarkdown>
                </Box>
                {hasLongContent && (
                  <Button
                    size="small"
                    onClick={() => toggleItemExpanded(item.id)}
                    startIcon={isExpanded ? <ExpandLess /> : <ExpandMore />}
                    sx={{ textTransform: 'none', fontSize: '0.85rem' }}
                  >
                    {isExpanded ? 'Show less' : 'Read more'}
                  </Button>
                )}
              </>
            )}
            {item.tags && item.tags.length > 0 && (
              <Box sx={{ display: 'flex', gap: 0.5, flexWrap: 'wrap', mt: 0.5 }}>
                {item.tags.slice(0, 3).map((tag, tagIndex) => (
                  <CategoryChip key={tagIndex} label={`#${tag}`} size="small" />
                ))}
              </Box>
            )}
          </Box>
          );
        })}
        </Box>
        
        {/* Pagination controls */}
        {totalPages > 1 && (
          <PaginationControls>
            <Tooltip title="First page">
              <IconButton 
                size="small" 
                onClick={() => handlePageChange(quadrantKey, 0)}
                disabled={currentPage === 0}
              >
                <FirstPage fontSize="small" />
              </IconButton>
            </Tooltip>
            <Tooltip title="Previous page">
              <IconButton 
                size="small" 
                onClick={() => handlePageChange(quadrantKey, currentPage - 1)}
                disabled={currentPage === 0}
              >
                <NavigateBefore fontSize="small" />
              </IconButton>
            </Tooltip>
            <PageInfo>
              {currentPage + 1} / {totalPages}
            </PageInfo>
            <Tooltip title="Next page">
              <IconButton 
                size="small" 
                onClick={() => handlePageChange(quadrantKey, currentPage + 1)}
                disabled={currentPage >= totalPages - 1}
              >
                <NavigateNext fontSize="small" />
              </IconButton>
            </Tooltip>
            <Tooltip title="Last page">
              <IconButton 
                size="small" 
                onClick={() => handlePageChange(quadrantKey, totalPages - 1)}
                disabled={currentPage >= totalPages - 1}
              >
                <LastPage fontSize="small" />
              </IconButton>
            </Tooltip>
          </PaginationControls>
        )}
      </Box>
    );
  };

  return (
    <Box sx={{ padding: variant === 'full' ? 4 : 0 }}>
      {variant === 'full' && (
        <HeaderContainer>
          <Typography variant="h3" sx={{ fontWeight: 700 }}>
            News & Updates
          </Typography>
          <Tooltip title="Refresh news content">
            <RefreshButton onClick={handleRefresh} disabled={loading}>
              <RefreshIcon />
            </RefreshButton>
          </Tooltip>
        </HeaderContainer>
      )}
      
      {variant === 'preview' && (
        <Typography variant="h4" sx={{ fontWeight: 700, mb: 2 }}>
          News
        </Typography>
      )}

      {loading && (
        <Box sx={{ display: 'flex', justifyContent: 'center', py: 4 }}>
          <CircularProgress />
        </Box>
      )}

      {error && (
        <Alert severity="warning" sx={{ mb: 3 }}>
          {error}
        </Alert>
      )}

      {/* 4-Quadrant Layout with responsive ordering */}
      {newsData && newsData.quadrants && (
        <Grid container spacing={3} sx={{ mb: 4 }}>
          {/* Updates - Order 1 on mobile */}
          <Grid item xs={12} md={6} sx={{ order: { xs: 1, md: 1 } }}>
            <QuadrantCard>
              <SectionTitle variant="h5">
                {newsData.quadrants.updates?.title || 'Updates'}
              </SectionTitle>
              {renderQuadrantContent(newsData.quadrants.updates, 'updates')}
            </QuadrantCard>
          </Grid>

          {/* News - Order 2 on mobile */}
          <Grid item xs={12} md={6} sx={{ order: { xs: 2, md: 2 } }}>
            <QuadrantCard>
              <SectionTitle variant="h5">
                {newsData.quadrants.news?.title || 'Latest News'}
              </SectionTitle>
              {renderQuadrantContent(newsData.quadrants.news, 'news')}
            </QuadrantCard>
          </Grid>

          {/* Events - Order 3 on mobile */}
          <Grid item xs={12} md={6} sx={{ order: { xs: 3, md: 3 } }}>
            <QuadrantCard>
              <SectionTitle variant="h5">
                {newsData.quadrants.events?.title || 'Events'}
              </SectionTitle>
              {renderQuadrantContent(newsData.quadrants.events, 'events')}
            </QuadrantCard>
          </Grid>

          {/* General - Order 4 on mobile */}
          <Grid item xs={12} md={6} sx={{ order: { xs: 4, md: 4 } }}>
            <QuadrantCard>
              <SectionTitle variant="h5">
                {newsData.quadrants.general?.title || 'General'}
              </SectionTitle>
              {renderQuadrantContent(newsData.quadrants.general, 'general')}
            </QuadrantCard>
          </Grid>
        </Grid>
      )}

      {/* Metadata Footer - only show for full variant */}
      {variant === 'full' && newsData && newsData.metadata && (
        <ContentCard>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', flexWrap: 'wrap', gap: 2 }}>
            <Typography variant="body2" sx={{ color: 'text.secondary' }}>
              {newsData.metadata.totalItems} total items across all categories
            </Typography>
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
              {usingCachedData && (
                <Chip 
                  label="Cached" 
                  size="small" 
                  color="info" 
                  sx={{ backgroundColor: 'rgba(33, 150, 243, 0.1)' }}
                />
              )}
              <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                Last fetched: {formatLastFetched(lastFetched)}
              </Typography>
            </Box>
          </Box>
        </ContentCard>
      )}
    </Box>
  );
};

export default News;