import React from 'react';
import { render, screen, fireEvent, waitFor } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import axios from 'axios';
import TestConversationModal from '../TestConversationModal';
import { RoutingConfigContext } from '../../../context/RoutingConfigContext';

// Mock axios
jest.mock('axios');

// Mock routing config value for tests
const mockRoutingConfig = {
  routingMode: 'dual',
  rawRoutingMode: 'dual',
  isLoading: false,
  error: null,
  isDualMode: true,
  normalizedMode: 'port',
};

// Helper to render with routing context
const renderWithRouting = (ui) => {
  return render(
    <RoutingConfigContext.Provider value={mockRoutingConfig}>
      {ui}
    </RoutingConfigContext.Provider>
  );
};
const mockedAxios = axios;

// Mock window.location
delete window.location;
window.location = {
  hostname: 'localhost',
  protocol: 'https:',
  port: '',
  origin: 'https://localhost',
};

// Mock MUI components to simplify testing
jest.mock('@mui/material/Dialog', () => {
  // eslint-disable-next-line react/prop-types
  return function MockDialog({ children, open }) {
    if (!open) return null;
    return <div data-testid="dialog">{children}</div>;
  };
});

// Mock scrollIntoView
window.HTMLElement.prototype.scrollIntoView = jest.fn();

describe('TestConversationModal', () => {
  const mockDeployment = {
    id: 'test-deployment-123',
    m_name: 'test-model',
    lb_port: 8080,
    engine_type: 'vllm'
  };

  const defaultProps = {
    deployment: mockDeployment,
    open: true,
    onClose: jest.fn()
  };

beforeEach(() => {
  jest.clearAllMocks();
  window.location.port = '';
  window.location.origin = 'https://localhost';
  // Silence console.error for expected errors in tests
  jest.spyOn(console, 'error').mockImplementation(() => {});
});

  afterEach(() => {
    // Restore console.error
    console.error.mockRestore();
  });

  describe('Component Rendering', () => {
    it('should render when open', () => {
      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      expect(screen.getByTestId('dialog')).toBeInTheDocument();
      expect(screen.getByText(/Test Conversation - test-model/i)).toBeInTheDocument();
      expect(screen.getByPlaceholderText(/Type your message/i)).toBeInTheDocument();
      expect(screen.getByText(/Start a conversation with the model/i)).toBeInTheDocument();
    });

    it('should not render when closed', () => {
      renderWithRouting(<TestConversationModal {...defaultProps} open={false} />);
      
      expect(screen.queryByTestId('dialog')).not.toBeInTheDocument();
    });

    it('should call onClose when close button is clicked', () => {
      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      // Find the close button by its SVG test id or parent button
      const buttons = screen.getAllByRole('button');
      const closeButton = buttons.find(button => {
        return button.querySelector('svg[data-testid="CloseIcon"]');
      });
      
      if (closeButton) {
        fireEvent.click(closeButton);
        expect(defaultProps.onClose).toHaveBeenCalledTimes(1);
      }
    });
  });

  describe('Message Sending', () => {
    it('should use sanitized m_name for vLLM engine', async () => {
      const vllmDeployment = {
        ...mockDeployment,
        engine_name: 'vllm'
      };

      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: 'Response from vLLM'
            }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} deployment={vllmDeployment} />);

      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Hello vLLM!');
      fireEvent.click(screen.getByText('Send'));

      await waitFor(() => {
        expect(mockedAxios.post).toHaveBeenCalledWith(
          'https://localhost:8080/v1/chat/completions',
          expect.objectContaining({
            model: 'test-model', // Uses sanitized m_name from deployment
            messages: expect.any(Array)
          }),
          expect.any(Object)
        );
      });
    });

    it('should use actual model name for non-vLLM engines', async () => {
      const llamacppDeployment = {
        ...mockDeployment,
        engine_name: 'llamacpp'
      };
      
      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: 'Response from llamacpp'
            }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} deployment={llamacppDeployment} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Hello llamacpp!');
      fireEvent.click(screen.getByText('Send'));
      
      await waitFor(() => {
        expect(mockedAxios.post).toHaveBeenCalledWith(
          'https://localhost:8080/v1/chat/completions',
          expect.objectContaining({
            model: 'test-model', // Should use actual model name
            messages: expect.any(Array)
          }),
          expect.any(Object)
        );
      });
    });

    it('should use m_name regardless of engine case', async () => {
      const vllmUppercaseDeployment = {
        ...mockDeployment,
        engine_name: 'VLLM'
      };

      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: 'Response'
            }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} deployment={vllmUppercaseDeployment} />);

      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Test');
      fireEvent.click(screen.getByText('Send'));

      await waitFor(() => {
        expect(mockedAxios.post).toHaveBeenCalledWith(
          'https://localhost:8080/v1/chat/completions',
          expect.objectContaining({
            model: 'test-model', // Uses sanitized m_name from deployment
            messages: expect.any(Array)
          }),
          expect.any(Object)
        );
      });
    });

    it('uses path-based endpoint when serve_path is provided', async () => {
      const user = userEvent.setup();
      window.location.port = '';
      window.location.origin = 'https://localhost';
      const pathDeployment = {
        ...mockDeployment,
        lb_port: 0,
        serve_path: '/runtime/models/path-test',
        access_path: '/runtime/models/path-test',
      };

      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: 'Path response',
            },
          }],
        },
      });

      renderWithRouting(<TestConversationModal {...defaultProps} deployment={pathDeployment} />);

      const input = screen.getByPlaceholderText(/Type your message/i);
      await user.type(input, 'Hello path!');
      fireEvent.click(screen.getByText('Send'));

      await waitFor(() => {
        expect(mockedAxios.post).toHaveBeenCalledWith(
          'https://localhost/runtime/models/path-test/v1/chat/completions',
          expect.objectContaining({
            messages: expect.any(Array),
          }),
          expect.any(Object)
        );
      });

      window.location.port = '';
      window.location.origin = 'https://localhost';
    });

    it('should send message and display response', async () => {
      const user = userEvent.setup();
      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: 'Hello! How can I help you today?'
            }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      const sendButton = screen.getByText('Send');
      
      // Type message
      await user.type(input, 'Hello!');
      expect(input).toHaveValue('Hello!');
      
      // Send message
      fireEvent.click(sendButton);
      
      // Verify API was called
      await waitFor(() => {
        expect(mockedAxios.post).toHaveBeenCalledWith(
          'https://localhost:8080/v1/chat/completions',
          expect.objectContaining({
            model: 'test-model',
            messages: expect.arrayContaining([
              expect.objectContaining({
                role: 'user',
                content: 'Hello!'
              })
            ])
          }),
          expect.any(Object)
        );
      });

      // Check response appears
      await waitFor(() => {
        expect(screen.getByText('Hello!')).toBeInTheDocument();
        expect(screen.getByText('Hello! How can I help you today?')).toBeInTheDocument();
      });
    });

    it('should clear input after sending', async () => {
      const user = userEvent.setup();
      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: { content: 'Response' }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await user.type(input, 'Test message');
      
      fireEvent.click(screen.getByText('Send'));
      
      await waitFor(() => {
        expect(input).toHaveValue('');
      });
    });
  });

  describe('Thinking/Reasoning Display', () => {
    it('should parse and display thinking content', async () => {
      const user = userEvent.setup();
      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: {
              content: '<thinking>Let me think about this...</thinking>The answer is 42.'
            }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await user.type(input, 'What is the meaning of life?');
      fireEvent.click(screen.getByText('Send'));
      
      await waitFor(() => {
        // Response should be visible
        expect(screen.getByText('The answer is 42.')).toBeInTheDocument();
        // Should have a button to show thinking
        expect(screen.getByText(/Show thinking/i)).toBeInTheDocument();
      });

      // Click to show thinking
      fireEvent.click(screen.getByText(/Show thinking/i));
      
      // Thinking content should now be visible
      expect(screen.getByText('Let me think about this...')).toBeInTheDocument();
      expect(screen.getByText(/Hide thinking/i)).toBeInTheDocument();
    });

    it('should handle multiple thinking tag formats', async () => {
      const testCases = [
        { tag: 'think', content: '<think>Thinking...</think>Response' },
        { tag: 'thinking', content: '<thinking>Thinking...</thinking>Response' },
        { tag: 'thought', content: '<thought>Thinking...</thought>Response' }
      ];

      for (const testCase of testCases) {
        mockedAxios.post.mockResolvedValueOnce({
          data: {
            choices: [{
              message: { content: testCase.content }
            }]
          }
        });

        const { unmount } = renderWithRouting(<TestConversationModal {...defaultProps} />);
        
        const input = screen.getByPlaceholderText(/Type your message/i);
        await userEvent.type(input, 'Test');
        fireEvent.click(screen.getByText('Send'));
        
        await waitFor(() => {
          expect(screen.getByText('Response')).toBeInTheDocument();
          expect(screen.getByText(/Show thinking/i)).toBeInTheDocument();
        });

        unmount();
      }
    });
  });

  describe('Error Handling', () => {
    it('should display error when API fails', async () => {
      mockedAxios.post.mockRejectedValueOnce({
        response: {
          status: 500,
          statusText: 'Internal Server Error'
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Test');
      fireEvent.click(screen.getByText('Send'));
      
      const errorMessages = await screen.findAllByText(/Error: 500 - Internal Server Error/i);
      expect(errorMessages.length).toBeGreaterThan(0);
    });

    it('should handle network errors', async () => {
      mockedAxios.post.mockRejectedValueOnce({
        request: {}
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Test');
      fireEvent.click(screen.getByText('Send'));
      
      const errorMessages = await screen.findAllByText(/No response from model/i);
      expect(errorMessages.length).toBeGreaterThan(0);
    });

    it('should handle malformed responses', async () => {
      mockedAxios.post.mockResolvedValueOnce({
        data: {} // Missing choices array
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Test');
      fireEvent.click(screen.getByText('Send'));
      
      await screen.findByText(/Invalid response format from model/i);
      await screen.findByText(/response\.data\.choices is missing or empty/i);
    });
  });

  describe('UI State Management', () => {
    it('should disable send button when loading', async () => {
      // Mock a slow response
      mockedAxios.post.mockImplementation(() => 
        new Promise(resolve => setTimeout(resolve, 1000))
      );

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      const input = screen.getByPlaceholderText(/Type your message/i);
      const sendButton = screen.getByText('Send');
      
      await userEvent.type(input, 'Test');
      fireEvent.click(sendButton);
      
      // Button should be disabled while loading
      expect(sendButton).toBeDisabled();
    });

    it('should clear conversation', async () => {
      mockedAxios.post.mockResolvedValueOnce({
        data: {
          choices: [{
            message: { content: 'Response' }
          }]
        }
      });

      renderWithRouting(<TestConversationModal {...defaultProps} />);
      
      // Send a message
      const input = screen.getByPlaceholderText(/Type your message/i);
      await userEvent.type(input, 'Test');
      fireEvent.click(screen.getByText('Send'));
      
      await waitFor(() => {
        expect(screen.getByText('Test')).toBeInTheDocument();
        expect(screen.getByText('Response')).toBeInTheDocument();
      });

      // Clear conversation
      const clearButton = screen.getByText('Clear');
      fireEvent.click(clearButton);
      
      // Messages should be gone
      expect(screen.queryByText('Test')).not.toBeInTheDocument();
      expect(screen.queryByText('Response')).not.toBeInTheDocument();
    });
  });
});
