import React from 'react';
import { act, within } from '@testing-library/react';
import { renderWithProviders, userEvent } from '../../../test-utils';
import ModelList from '../ModelList';
import axios from 'axios';

// Mock axios
jest.mock('axios');
const mockedAxios = axios;

// Mock the BASE_URL
jest.mock('../../../const', () => ({
  BASE_URL: 'http://localhost:8000',
  API_PORT: '7777',
  LAB_PORT: '8888',
  isDefaultPort: (port) => port === '' || port === '80' || port === '443',
}));

// Mock DownloadModal component
jest.mock('../DownloadModal', () => {
  // eslint-disable-next-line react/prop-types
  const MockDownloadModal = ({ model, closeModal, showModal }) => {
    return showModal ? (
      <div data-testid="download-modal">
        {/* eslint-disable-next-line react/prop-types */}
        <p>Download Modal for {model.name}</p>
        <button onClick={closeModal}>Close</button>
      </div>
    ) : null;
  };
  return MockDownloadModal;
});

// Mock ModelFilesContext
const mockContext = {
  refreshModelList: false
};

jest.mock('../ModelFilesContext', () => {
  const React = require('react');
  const MockContext = React.createContext(mockContext);
  // eslint-disable-next-line react/prop-types
  const MockProvider = ({ children }) => React.createElement('div', { 'data-testid': 'model-files-provider' }, children);
  MockProvider.displayName = 'MockModelFilesProvider';
  MockContext.Provider = MockProvider;
  return MockContext;
});

delete window.location;
window.location = {
  hostname: 'localhost',
  protocol: 'https:',
  port: '',
  origin: 'https://localhost',
};

describe('ModelList', () => {
  const mockModels = [
    {
      id: '1',
      name: 'Test Model 1',
      hub: 'HubsHf',
      repo_modelId: 'test/model1',
      version: 'v1.0',
      source_repository: 'https://huggingface.co/test/model1'
    },
    {
      id: '2', 
      name: 'Test Model 2',
      hub: 'HubsHf',
      repo_modelId: 'test/model2',
      version: 'v2.0',
      source_repository: 'https://huggingface.co/test/model2'
    }
  ];

  const mockSearchResults = [
    {
      repo_modelId: 'search/result1',
      hub: 'HubsHf',
      version: 'latest'
    },
    {
      repo_modelId: 'search/result2', 
      hub: 'HubsHf',
      version: 'latest'
    }
  ];

  beforeEach(() => {
    jest.clearAllMocks();
    mockedAxios.get.mockResolvedValue({ status: 200, data: mockModels });
    mockedAxios.post.mockResolvedValue({ status: 200, data: [] });
    mockedAxios.delete.mockResolvedValue({ status: 200 });
    window.location.port = '3000';
    window.location.origin = 'https://localhost';
  });

  describe('Rendering', () => {
    test('renders model list with provided models', () => {
      const { getByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      expect(getByText('Your Models')).toBeInTheDocument();
      expect(getByText('Test Model 1')).toBeInTheDocument();
      expect(getByText('Test Model 2')).toBeInTheDocument();
    });

    test('renders search results when search is true', () => {
      const { getByText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} />
      );

      expect(getByText('Model Search Results')).toBeInTheDocument();
    });

    test('displays no models message when list is empty', () => {
      const { getByText } = renderWithProviders(
        <ModelList models={[]} />
      );

      expect(getByText('No models available.')).toBeInTheDocument();
    });

    test('renders table headers correctly', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const headers = container.querySelectorAll('th');
      const headerTexts = Array.from(headers).map(th => th.textContent);
      
      expect(headerTexts).toContain('Model Name');
      expect(headerTexts).toContain('Provider');
      expect(headerTexts).toContain('Size');
      expect(headerTexts).toContain('Type');
      expect(headerTexts).toContain('Deployment IDs');
      expect(headerTexts).toContain('Actions');
    });

    test('hides deployments column in search mode', () => {
      const { queryByText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} />
      );

      expect(queryByText('Deployments')).not.toBeInTheDocument();
    });

    test('renders provider links correctly', () => {
      const { getAllByText, container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Check that provider names are displayed (both models have 'test' as provider)
      const providerLinks = getAllByText('test');
      expect(providerLinks).toHaveLength(2);
      
      // Check that they are links to the provider page
      const links = container.querySelectorAll('a[href="https://huggingface.co/test"]');
      expect(links).toHaveLength(2);
      expect(links[0]).toHaveAttribute('target', '_blank');
      expect(links[0]).toHaveAttribute('rel', 'noopener noreferrer');
      
      // The tooltip functionality is provided by Material-UI and will work in the browser
      // Testing the actual tooltip behavior would require more complex setup
    });

    test('handles non-HuggingFace hub providers', () => {
      const modelsWithDifferentHub = [
        {
          id: '3',
          name: 'Custom Model',
          hub: 'CustomHub',
          repo_modelId: 'custom/model',
          version: 'v1.0'
        }
      ];

      const { getByText } = renderWithProviders(
        <ModelList models={modelsWithDifferentHub} />
      );

      // Should display provider name without link for non-HuggingFace hubs
      const provider = getByText('custom');
      expect(provider).toBeInTheDocument();
      expect(provider.tagName).not.toBe('A'); // Should not be a link
    });

    test('handles models without repo_modelId', () => {
      const modelsWithoutRepoId = [
        {
          id: '4',
          name: 'Local Model',
          hub: 'HubsHf',
          m_files: [] // Add empty files array to avoid undefined issues
        }
      ];

      const { container } = renderWithProviders(
        <ModelList models={modelsWithoutRepoId} />
      );

      // Should display '-' when no provider can be extracted
      // Find all cells in the first row to debug
      const rows = container.querySelectorAll('tbody tr');
      const cells = rows[0].querySelectorAll('td');
      
      // Find the provider cell - it contains just '-' text
      let providerCell = null;
      for (let i = 0; i < cells.length; i++) {
        if (cells[i].textContent === '-') {
          providerCell = cells[i];
          break;
        }
      }
      
      expect(providerCell).not.toBeNull();
      expect(providerCell.textContent).toBe('-');
    });
  });

  describe('Data Fetching', () => {
    test('fetches models from API when no models prop provided', async () => {
      renderWithProviders(<ModelList />);

      expect(mockedAxios.get).toHaveBeenCalledWith('http://localhost:8000/models/?load_files=true');
    });

    test('does not fetch models when models prop is provided', () => {
      renderWithProviders(<ModelList models={mockModels} />);

      // Should not fetch models, but will fetch deployments
      expect(mockedAxios.get).not.toHaveBeenCalledWith('http://localhost:8000/models/?load_files=true');
      expect(mockedAxios.get).toHaveBeenCalledWith('http://localhost:8000/serving/deployments');
    });

    test('displays loading spinner while fetching', async () => {
      // Mock a pending promise that we can control
      let resolvePromise;
      const pendingPromise = new Promise((resolve) => {
        resolvePromise = resolve;
      });
      
      mockedAxios.get.mockReturnValue(pendingPromise);

      const { getByTestId } = renderWithProviders(<ModelList />);

      // Should show loading spinner immediately
      expect(getByTestId('loading-spinner')).toBeInTheDocument();
      
      // Resolve the promise to avoid hanging and wait for the update
      await act(async () => {
        resolvePromise({ status: 200, data: mockModels });
        await pendingPromise;
      });
    });

    test('handles API fetch error gracefully', async () => {
      mockedAxios.get.mockRejectedValue(new Error('Network error'));

      const { findByText } = renderWithProviders(<ModelList />);

      expect(await findByText(/network error/i)).toBeInTheDocument();
    });

    test('handles non-200 response status', async () => {
      mockedAxios.get.mockResolvedValue({ status: 500, data: null });

      const { findByText } = renderWithProviders(<ModelList />);

      expect(await findByText(/failed to fetch models/i)).toBeInTheDocument();
    });
  });

  describe('Model Actions', () => {
    test('displays download buttons for all models', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const downloadIcons = container.querySelectorAll('[data-testid="DownloadIcon"]');
      expect(downloadIcons).toHaveLength(2);
    });

    test('displays delete buttons for owned models only', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const deleteIcons = container.querySelectorAll('[data-testid="DeleteIcon"]');
      expect(deleteIcons).toHaveLength(2);
    });

    test('opens download modal when download button clicked', async () => {
      const user = userEvent.setup();
      
      const { container, findByTestId } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const downloadIcon = container.querySelector('[data-testid="DownloadIcon"]');
      const downloadButton = downloadIcon.closest('button');
      await user.click(downloadButton);

      expect(await findByTestId('download-modal')).toBeInTheDocument();
    });

    test('opens confirm delete modal when delete button clicked', async () => {
      const user = userEvent.setup();
      
      const { container, findByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const deleteIcon = container.querySelector('[data-testid="DeleteIcon"]');
      const deleteButton = deleteIcon.closest('button');
      await user.click(deleteButton);

      expect(await findByText('Confirm Delete')).toBeInTheDocument();
    });
  });

  describe('Download Functionality', () => {
    test('fetches model files when download is clicked', async () => {
      const user = userEvent.setup();
      
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const downloadIcon = container.querySelector('[data-testid="DownloadIcon"]');
      const downloadButton = downloadIcon.closest('button');
      await user.click(downloadButton);

      expect(mockedAxios.post).toHaveBeenCalledWith(
        'http://localhost:8000/model_files/search/',
        {
          hub: 'HubsHf',
          model: 'test/model1',
          version: 'v1.0'
        }
      );
    });

    test('handles download API error gracefully', async () => {
      mockedAxios.post.mockRejectedValue(new Error('Download failed'));
      const user = userEvent.setup();
      
      const { container, findByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const downloadIcon = container.querySelector('[data-testid="DownloadIcon"]');
      const downloadButton = downloadIcon.closest('button');
      await user.click(downloadButton);

      expect(await findByText(/download failed/i)).toBeInTheDocument();
    });

    test('closes download modal when close button is clicked', async () => {
      const user = userEvent.setup();
      
      const { container, findByTestId, queryByTestId } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Open modal
      const downloadIcon = container.querySelector('[data-testid="DownloadIcon"]');
      const downloadButton = downloadIcon.closest('button');
      await user.click(downloadButton);
      
      expect(await findByTestId('download-modal')).toBeInTheDocument();

      // Close modal
      const closeButton = await findByTestId('download-modal');
      const closeBtn = closeButton.querySelector('button');
      await user.click(closeBtn);

      expect(queryByTestId('download-modal')).not.toBeInTheDocument();
    });
  });

  describe('Delete Functionality', () => {
    test('deletes model when confirmed', async () => {
      const user = userEvent.setup();
      
      const { container, findByText, queryByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Click delete button
      const deleteIcon = container.querySelector('[data-testid="DeleteIcon"]');
      const deleteButton = deleteIcon.closest('button');
      await user.click(deleteButton);

      // Confirm deletion
      const confirmButton = await findByText('Yes');
      await user.click(confirmButton);

      expect(mockedAxios.delete).toHaveBeenCalledWith('http://localhost:8000/models/1');
      
      // Model should be removed from list
      expect(queryByText('Test Model 1')).not.toBeInTheDocument();
    });

    test('handles delete API error gracefully', async () => {
      mockedAxios.delete.mockRejectedValue(new Error('Delete failed'));
      const user = userEvent.setup();
      
      const { container, findByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Click delete and confirm
      const deleteIcon = container.querySelector('[data-testid="DeleteIcon"]');
      const deleteButton = deleteIcon.closest('button');
      await user.click(deleteButton);
      
      const confirmButton = await findByText('Yes');
      await user.click(confirmButton);

      expect(await findByText(/delete failed/i)).toBeInTheDocument();
    });

    test('cancels deletion when cancel is clicked', async () => {
      const user = userEvent.setup();
      
      const { container, findByText, queryByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Click delete button
      const deleteIcon = container.querySelector('[data-testid="DeleteIcon"]');
      const deleteButton = deleteIcon.closest('button');
      await user.click(deleteButton);

      // Cancel deletion
      const cancelButton = await findByText('No');
      await user.click(cancelButton);

      expect(mockedAxios.delete).not.toHaveBeenCalled();
      expect(queryByText('Confirm Delete')).not.toBeInTheDocument();
    });
  });

  describe('Model Display', () => {
    test('displays model names as links for owned models', () => {
      const { getByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      const modelLink = getByText('Test Model 1').closest('a');
      expect(modelLink).toHaveAttribute('href', '/models/1');
    });

    test('displays model repo ID for search results without links', () => {
      const { getByText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} />
      );

      expect(getByText('search/result1')).toBeInTheDocument();
      expect(getByText('search/result1').closest('a')).toBeNull();
    });

    test('truncates long model names', () => {
      const longNameModel = [{
        ...mockModels[0],
        name: 'A'.repeat(150)
      }];

      const { getByText } = renderWithProviders(
        <ModelList models={longNameModel} />
      );

      // Model names are truncated at 60 characters
      const displayedName = getByText('A'.repeat(60));
      expect(displayedName).toBeInTheDocument();
    });

    test('displays source repository links', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // Source repository is now shown as an icon link
      const sourceLinks = container.querySelectorAll('a[href="https://huggingface.co/test/model1"]');
      expect(sourceLinks.length).toBeGreaterThan(0);
      expect(sourceLinks[0]).toHaveAttribute('target', '_blank');
      expect(sourceLinks[0]).toHaveAttribute('rel', 'noopener noreferrer');
    });

    test('handles long source repository URLs', () => {
      const longUrlModel = [{
        ...mockModels[0],
        source_repository: 'https://very-long-domain-name-that-exceeds-sixty-characters.com/model'
      }];

      const { container } = renderWithProviders(
        <ModelList models={longUrlModel} />
      );

      // Source repository URLs are not truncated anymore, they're shown as icon links
      const sourceLink = container.querySelector('a[href="https://very-long-domain-name-that-exceeds-sixty-characters.com/model"]');
      expect(sourceLink).toBeInTheDocument();
    });
  });

  describe('Deployment Display', () => {
    test('shows port-based deployment endpoints', async () => {
      const deployment = {
        id: 'deployment-1',
        m_id: '1',
        status: 'DEPLOYED',
        lb_port: 9000,
      };

      mockedAxios.get.mockResolvedValueOnce({ status: 200, data: [deployment] });

      const { findByText } = renderWithProviders(<ModelList models={mockModels} />);

      expect(await findByText('deployment-1')).toBeInTheDocument();
    });

    test('shows runtime paths when serve_path is available', async () => {
      const deployment = {
        id: 'deployment-2',
        m_id: '1',
        status: 'DEPLOYED',
        lb_port: 0,
        serve_path: '/runtime/models/deployment-2',
      };

      mockedAxios.get.mockResolvedValueOnce({ status: 200, data: [deployment] });

      const { findByText } = renderWithProviders(<ModelList models={mockModels} />);

      expect(await findByText('deployment-2')).toBeInTheDocument();
    });
  });

  describe('Search Results', () => {
    test('shows clear search results button when in search mode', () => {
      const mockClear = jest.fn();
      
      const { getByLabelText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} clearSearchResults={mockClear} />
      );

      expect(getByLabelText('clear results')).toBeInTheDocument();
    });

    test('calls clearSearchResults when clear button is clicked', async () => {
      const mockClear = jest.fn();
      const user = userEvent.setup();
      
      const { getByLabelText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} clearSearchResults={mockClear} />
      );

      await user.click(getByLabelText('clear results'));
      expect(mockClear).toHaveBeenCalledTimes(1);
    });

    test('does not show clear button when not in search mode', () => {
      const { queryByLabelText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      expect(queryByLabelText('clear results')).not.toBeInTheDocument();
    });
  });

  describe('Accessibility', () => {
    test('has proper table structure', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      expect(container.querySelector('table')).toBeInTheDocument();
      expect(container.querySelector('thead')).toBeInTheDocument();
      expect(container.querySelector('tbody')).toBeInTheDocument();
    });

    test('download buttons have proper labels', () => {
      const { getAllByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // In normal mode, download buttons say "Re-download"
      const downloadButtons = getAllByText('Re-download');
      expect(downloadButtons.length).toBeGreaterThan(0);
      downloadButtons.forEach(button => {
        expect(button).toBeInTheDocument();
      });
    });

    test('external links have proper attributes', () => {
      const { container } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      // External links are the source repository icon links
      const externalLinks = container.querySelectorAll('a[target="_blank"]');
      expect(externalLinks.length).toBeGreaterThan(0);
      externalLinks.forEach(link => {
        expect(link).toHaveAttribute('rel', 'noopener noreferrer');
      });
    });

    test('clear button has proper aria-label', () => {
      const { getByLabelText } = renderWithProviders(
        <ModelList models={mockSearchResults} search={true} clearSearchResults={jest.fn()} />
      );

      expect(getByLabelText('clear results')).toBeInTheDocument();
    });
  });

  describe('Edge Cases', () => {
    test('hides download controls for external models', () => {
      const externalModel = {
        id: 'ext-1',
        name: 'External Model',
        hub: 'external',
        purpose: 'external',
        modelfamily: 'external',
        repo_modelId: 'external/model',
        default_config: {
          config: {
            external_endpoint: JSON.stringify({
              service: 'aws_bedrock',
              display_name: 'Bedrock Claude',
            }),
          },
        },
      };

      const { container, queryByText } = renderWithProviders(
        <ModelList models={[externalModel]} />
      );

      const row = container.querySelector('tbody tr');
      expect(row).toBeInTheDocument();
      const rowUtils = within(row);

      expect(rowUtils.getByText('AWS Bedrock')).toBeInTheDocument();
      expect(rowUtils.getByText('External Endpoint')).toBeInTheDocument();
      expect(rowUtils.getByText('—')).toBeInTheDocument();
      expect(queryByText('Re-download')).not.toBeInTheDocument();
    });

    test('handles models without names gracefully', () => {
      const modelsWithoutNames = [{
        id: '1',
        hub: 'huggingface',
        repo_modelId: 'test/model'
      }];

      const { getByText } = renderWithProviders(
        <ModelList models={modelsWithoutNames} />
      );

      expect(getByText('Unnamed Model')).toBeInTheDocument();
    });

    test('handles models without repository URLs', () => {
      const modelsWithoutUrls = [{
        id: '1',
        name: 'Test Model',
        hub: 'huggingface',
        repo_modelId: 'test/model'
      }];

      const { container } = renderWithProviders(
        <ModelList models={modelsWithoutUrls} />
      );

      // Models without source_repository should not have external link icons
      const externalLinks = container.querySelectorAll('a[target="_blank"][rel="noopener noreferrer"]');
      // Should only find model name links, not source repository links
      const sourceLinks = Array.from(externalLinks).filter(link => 
        link.href.includes('huggingface.co')
      );
      expect(sourceLinks.length).toBe(0);
    });

    test('handles missing ModelFilesContext gracefully', () => {
      // This test ensures the component doesn't crash when context is null
      const { getByText } = renderWithProviders(
        <ModelList models={mockModels} />
      );

      expect(getByText('Your Models')).toBeInTheDocument();
    });

    test('renders without crashing with minimal props', () => {
      const { container } = renderWithProviders(<ModelList />);
      expect(container).toBeInTheDocument();
    });
  });
});
