import React, { useContext } from 'react';
import { renderHook, act, waitFor } from '@testing-library/react';
import axios from 'axios';
import ModelFilesContext, { ModelFilesProvider } from '../ModelFilesContext';

// Mock axios
jest.mock('axios');
const mockedAxios = axios;

// Mock BASE_URL
jest.mock('../../../const', () => ({
  BASE_URL: 'http://localhost:8000'
}));

// Custom hook to use the context
const useModelFilesContext = () => {
  const context = useContext(ModelFilesContext);
  if (!context) {
    throw new Error('useModelFilesContext must be used within ModelFilesProvider');
  }
  return context;
};

describe('ModelFilesContext', () => {
  const wrapper = ({ children }) => (
    <ModelFilesProvider>{children}</ModelFilesProvider>
  );

  beforeEach(() => {
    jest.clearAllMocks();
    jest.useFakeTimers();
    // Default mock for download status - empty array
    mockedAxios.get.mockResolvedValue({ data: [] });
    mockedAxios.delete.mockResolvedValue({ data: { result: true } });
  });

  afterEach(async () => {
    // Let all pending timers and promises resolve
    await act(async () => {
      jest.runOnlyPendingTimers();
      await Promise.resolve();
    });
    jest.useRealTimers();
    jest.clearAllMocks();
  });

  describe('Initial State', () => {
    test('provides initial context values', async () => {
      let result;
      
      await act(async () => {
        const hook = renderHook(
          () => useModelFilesContext(),
          { wrapper }
        );
        result = hook.result;
        // Allow initial useEffect to run
        jest.advanceTimersByTime(0);
      });

      expect(result.current.statusModelFiles).toEqual([]);
      expect(result.current.showDownloadStatus).toBe(false);
      expect(result.current.hasActiveDownloads).toBe(false);
      expect(result.current.modelListRefreshTrigger).toBe(0);
      expect(result.current.toast).toEqual({ open: false, message: '', severity: 'info' });
    });
  });

  describe('Download Status Fetching', () => {
    test('fetches download status on mount', async () => {
      const mockStatusData = [
        {
          id: '1',
          name: 'model1.bin',
          is_downloading: true,
          dl_requested_at: '2024-01-01',
          storage_location: null,
          download_percentage: 50
        }
      ];

      mockedAxios.get.mockResolvedValueOnce({
        data: mockStatusData
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial fetch to be called
      await act(async () => {
        jest.advanceTimersByTime(100);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledWith(
          'http://localhost:8000/model_files/download_status/'
        );
      });

      // Let all state updates complete
      await act(async () => {
        jest.advanceTimersByTime(100);
      });

      expect(result.current.statusModelFiles).toEqual(mockStatusData);
      expect(result.current.showDownloadStatus).toBe(true);
      expect(result.current.hasActiveDownloads).toBe(true);
    });

    test('handles fetch error gracefully', async () => {
      mockedAxios.get.mockRejectedValueOnce(new Error('Network error'));

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial fetch to be called
      await act(async () => {
        jest.advanceTimersByTime(100);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalled();
      });

      // Let error handling complete
      await act(async () => {
        jest.advanceTimersByTime(100);
      });

      expect(result.current.statusModelFiles).toEqual([]);
      expect(result.current.showDownloadStatus).toBe(false);
    });

    test('shows warning toast for 503 errors', async () => {
      mockedAxios.get.mockRejectedValueOnce({
        response: { status: 503 }
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial fetch and error handling
      await act(async () => {
        jest.advanceTimersByTime(100);
      });

      await waitFor(() => {
        expect(result.current.toast.message).toBe('Service temporarily unavailable. Retrying...');
        expect(result.current.toast.severity).toBe('warning');
      });
    });
  });

  describe('Polling Mechanism', () => {
    test.skip('starts polling when active downloads exist', async () => {
      const mockActiveDownload = [{
        id: '1',
        name: 'model.bin',
        is_downloading: true,
        dl_requested_at: '2024-01-01',
        storage_location: null
      }];

      mockedAxios.get
        .mockResolvedValueOnce({ data: mockActiveDownload })
        .mockResolvedValueOnce({ data: mockActiveDownload })
        .mockResolvedValueOnce({ data: mockActiveDownload });

      renderHook(() => React.useContext(ModelFilesContext), { wrapper });

      // Wait for initial fetch and polling to start
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });

      // Wait a bit for state updates to complete
      await act(async () => {
        await new Promise(resolve => setTimeout(resolve, 0));
      });

      // Fast-forward 2 seconds
      await act(async () => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(2);
      });

      // Fast-forward another 2 seconds
      await act(async () => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(3);
      });
    });

    test.skip('stops polling when no active downloads', async () => {
      const mockActiveDownload = [{
        id: '1',
        name: 'model.bin',
        is_downloading: true,
        dl_requested_at: '2024-01-01',
        storage_location: null
      }];

      const mockCompletedDownload = [{
        id: '1',
        name: 'model.bin',
        is_downloading: false,
        dl_requested_at: null,
        storage_location: '/path/to/file'
      }];

      mockedAxios.get
        .mockResolvedValueOnce({ data: mockActiveDownload })
        .mockResolvedValueOnce({ data: mockCompletedDownload })
        .mockResolvedValueOnce({ data: [] }); // Should not be called

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial fetch
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
        expect(result.current.hasActiveDownloads).toBe(true);
      });

      // Wait for state to settle
      await act(async () => {
        await new Promise(resolve => setTimeout(resolve, 0));
      });

      // Fast-forward 2 seconds - should fetch again
      await act(async () => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(2);
        expect(result.current.hasActiveDownloads).toBe(false);
      });

      // Fast-forward 2 more seconds - should NOT fetch again
      await act(async () => {
        jest.advanceTimersByTime(2000);
      });

      // Give it a moment to ensure no additional calls
      await act(async () => {
        await new Promise(resolve => setTimeout(resolve, 100));
      });

      expect(mockedAxios.get).toHaveBeenCalledTimes(2); // Still 2
    });

    test.skip('maintains 2-second polling interval', async () => {
      const mockActiveDownload = [{
        id: '1',
        is_downloading: true,
        dl_requested_at: '2024-01-01',
        storage_location: null
      }];

      mockedAxios.get.mockResolvedValue({ data: mockActiveDownload });

      renderHook(() => React.useContext(ModelFilesContext), { wrapper });

      // Initial fetch
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });

      // Wait for polling to start
      await act(async () => {
        await new Promise(resolve => setTimeout(resolve, 0));
      });

      // Test exact 2-second intervals
      for (let i = 1; i <= 3; i++) {
        // Advance just under 2 seconds - should not trigger
        await act(async () => {
          jest.advanceTimersByTime(1999);
          await new Promise(resolve => setTimeout(resolve, 0));
        });
        
        expect(mockedAxios.get).toHaveBeenCalledTimes(i);

        // Advance the remaining 1ms to complete 2 seconds
        await act(async () => {
          jest.advanceTimersByTime(1);
        });

        await waitFor(() => {
          expect(mockedAxios.get).toHaveBeenCalledTimes(i + 1);
        });
      }
    });
  });

  describe('File Metadata Preservation', () => {
    test('preserves file names across status updates', async () => {
      const initialData = [{
        id: '1',
        name: 'important-model.bin',
        size: 1000,
        m_id: 'model123',
        is_downloading: true,
        download_percentage: 0
      }];

      const updateData = [{
        id: '1',
        name: 'Unnamed file', // Backend returns generic name
        is_downloading: true,
        download_percentage: 50
      }];

      mockedAxios.get
        .mockResolvedValueOnce({ data: initialData })
        .mockResolvedValueOnce({ data: updateData });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial fetch
      await waitFor(() => {
        expect(result.current.statusModelFiles[0].name).toBe('important-model.bin');
      });

      // Trigger polling update
      act(() => {
        jest.advanceTimersByTime(2000);
      });

      // Name should be preserved despite backend returning 'Unnamed file'
      await waitFor(() => {
        expect(result.current.statusModelFiles[0].name).toBe('important-model.bin');
        expect(result.current.statusModelFiles[0].download_percentage).toBe(50);
      });
    });

    test('handles files without initial names', async () => {
      const dataWithoutName = [{
        id: '1',
        name: null,
        is_downloading: true
      }];

      mockedAxios.get.mockResolvedValueOnce({ data: dataWithoutName });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      await waitFor(() => {
        expect(result.current.statusModelFiles[0].name).toBe('Unnamed file');
      });
    });
  });

  describe('Cancel Operations', () => {
    test('cancelDownload removes file from status and triggers refresh', async () => {
      const mockFiles = [{
        id: '1',
        name: 'file1.bin',
        is_downloading: true
      }, {
        id: '2',
        name: 'file2.bin',
        is_downloading: true
      }];

      mockedAxios.get.mockResolvedValueOnce({ data: mockFiles });
      mockedAxios.delete.mockResolvedValueOnce({ 
        data: { result: true, message: 'Cancelled' }
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Wait for initial data
      await waitFor(() => {
        expect(result.current.statusModelFiles).toHaveLength(2);
      });

      // Cancel first download
      await act(async () => {
        await result.current.cancelDownload('1', 'file1.bin');
      });

      // Verify API call
      expect(mockedAxios.delete).toHaveBeenCalledWith(
        'http://localhost:8000/model_files/1/download'
      );

      // File should be removed immediately
      expect(result.current.statusModelFiles).toHaveLength(1);
      expect(result.current.statusModelFiles[0].id).toBe('2');

      // Toast should show
      expect(result.current.toast.message).toBe('Download cancelled for file1.bin');
      expect(result.current.toast.severity).toBe('success');

      // Model list refresh should trigger after delay
      const initialTrigger = result.current.modelListRefreshTrigger;
      
      act(() => {
        jest.advanceTimersByTime(500);
      });

      expect(result.current.modelListRefreshTrigger).toBe(initialTrigger + 1);
    });

    test('handles cancel errors gracefully', async () => {
      mockedAxios.get.mockResolvedValueOnce({ data: [{ id: '1', name: 'file.bin' }] });
      mockedAxios.delete.mockRejectedValueOnce({
        response: { data: { detail: 'Cannot cancel completed download' } }
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      await waitFor(() => {
        expect(result.current.statusModelFiles).toHaveLength(1);
      });

      await act(async () => {
        const response = await result.current.cancelDownload('1', 'file.bin');
        expect(response.success).toBe(false);
      });

      expect(result.current.toast.message).toBe('Error: Cannot cancel completed download');
      expect(result.current.toast.severity).toBe('error');
    });

    test('cancelAllDownloads handles both active and queued', async () => {
      const mockFiles = [
        { id: '1', name: 'active.bin', is_downloading: true },
        { id: '2', name: 'queued.bin', dl_requested_at: '2024-01-01', is_downloading: false }
      ];

      mockedAxios.get.mockResolvedValueOnce({ data: mockFiles });
      mockedAxios.delete.mockResolvedValueOnce({
        data: {
          result: true,
          cancelled_files: ['1', '2'],
          cancelled_count: 2,
          failed_count: 0,
          total_downloads: 2
        }
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      await waitFor(() => {
        expect(result.current.statusModelFiles).toHaveLength(2);
      });

      await act(async () => {
        await result.current.cancelAllDownloads();
      });

      expect(mockedAxios.delete).toHaveBeenCalledWith(
        'http://localhost:8000/model_files/downloads/cancel_all'
      );

      // All files should be removed
      expect(result.current.statusModelFiles).toHaveLength(0);
      expect(result.current.toast.message).toBe('Successfully cancelled all 2 downloads');
    });

    test('cancelAllDownloads reports partial failures', async () => {
      mockedAxios.get.mockResolvedValueOnce({ data: [{ id: '1' }, { id: '2' }] });
      mockedAxios.delete.mockResolvedValueOnce({
        data: {
          result: true,
          cancelled_files: ['1'],
          cancelled_count: 1,
          failed_count: 1,
          total_downloads: 2
        }
      });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      await waitFor(() => {
        expect(result.current.statusModelFiles).toHaveLength(2);
      });

      await act(async () => {
        await result.current.cancelAllDownloads();
      });

      expect(result.current.statusModelFiles).toHaveLength(1);
      expect(result.current.toast.message).toBe('Cancelled 1 downloads, 1 failed');
      expect(result.current.toast.severity).toBe('warning');
    });
  });

  describe('State Management', () => {
    test('updateStatusModelFiles merges data correctly', () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      const initialFiles = [
        { id: '1', name: 'file1.bin', download_percentage: 0 },
        { id: '2', name: 'file2.bin', download_percentage: 0 }
      ];

      act(() => {
        result.current.updateStatusModelFiles(initialFiles, true);
      });

      expect(result.current.statusModelFiles).toEqual(initialFiles);

      // Update with partial data
      const updates = [
        { id: '1', download_percentage: 50 },
        { id: '2', download_percentage: 75 }
      ];

      act(() => {
        result.current.updateStatusModelFiles(updates, false);
      });

      // Names should be preserved
      expect(result.current.statusModelFiles[0].name).toBe('file1.bin');
      expect(result.current.statusModelFiles[0].download_percentage).toBe(50);
      expect(result.current.statusModelFiles[1].name).toBe('file2.bin');
      expect(result.current.statusModelFiles[1].download_percentage).toBe(75);
    });

    test('clearCompletedDownloads removes only completed files', () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      const mixedFiles = [
        { id: '1', storage_location: '/path', dl_requested_at: null }, // Completed
        { id: '2', storage_location: null, dl_requested_at: '2024-01-01' }, // Active
        { id: '3', storage_location: '/path', dl_requested_at: null }, // Completed
      ];

      act(() => {
        result.current.updateStatusModelFiles(mixedFiles, true);
      });

      act(() => {
        result.current.clearCompletedDownloads();
      });

      expect(result.current.statusModelFiles).toHaveLength(1);
      expect(result.current.statusModelFiles[0].id).toBe('2');
    });

    test('refreshModelList increments trigger', () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      const initialTrigger = result.current.modelListRefreshTrigger;

      act(() => {
        result.current.refreshModelList();
      });

      expect(result.current.modelListRefreshTrigger).toBe(initialTrigger + 1);

      act(() => {
        result.current.refreshModelList();
      });

      expect(result.current.modelListRefreshTrigger).toBe(initialTrigger + 2);
    });

    test('startPolling initiates polling', async () => {
      mockedAxios.get.mockResolvedValue({ data: [] });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Clear initial fetch
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });

      jest.clearAllMocks();

      // Start polling manually
      act(() => {
        result.current.startPolling();
      });

      // Should start fetching
      act(() => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });
    });
  });

  describe('Toast Management', () => {
    test('showToast updates toast state', () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      act(() => {
        result.current.showToast('Test message', 'error');
      });

      expect(result.current.toast).toEqual({
        open: true,
        message: 'Test message',
        severity: 'error'
      });
    });

    test('closeToast closes toast', () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      act(() => {
        result.current.showToast('Test message', 'info');
      });

      act(() => {
        result.current.closeToast();
      });

      expect(result.current.toast.open).toBe(false);
    });
  });

  describe('Race Condition Prevention', () => {
    test('handles rapid status updates without losing data', async () => {
      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      // Simulate rapid updates
      const updates = [
        [{ id: '1', name: 'file.bin', download_percentage: 10 }],
        [{ id: '1', download_percentage: 20 }],
        [{ id: '1', download_percentage: 30 }],
        [{ id: '1', download_percentage: 40 }],
        [{ id: '1', download_percentage: 50 }]
      ];

      // First update with name
      act(() => {
        result.current.updateStatusModelFiles(updates[0], true);
      });

      // Rapid subsequent updates
      updates.slice(1).forEach(update => {
        act(() => {
          result.current.updateStatusModelFiles(update, false);
        });
      });

      // Name should be preserved through all updates
      expect(result.current.statusModelFiles[0].name).toBe('file.bin');
      expect(result.current.statusModelFiles[0].download_percentage).toBe(50);
    });

    test('prevents multiple polling intervals', async () => {
      const mockActiveDownload = [{
        id: '1',
        is_downloading: true,
        dl_requested_at: '2024-01-01'
      }];

      mockedAxios.get.mockResolvedValue({ data: mockActiveDownload });

      const { result } = renderHook(
        () => React.useContext(ModelFilesContext),
        { wrapper }
      );

      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });

      // Try to start polling multiple times
      act(() => {
        result.current.startPolling();
        result.current.startPolling();
        result.current.startPolling();
      });

      // Advance time
      act(() => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        // Should only have one additional call, not multiple
        expect(mockedAxios.get).toHaveBeenCalledTimes(2);
      });

      // Advance time again
      act(() => {
        jest.advanceTimersByTime(2000);
      });

      await waitFor(() => {
        // Still just one interval running
        expect(mockedAxios.get).toHaveBeenCalledTimes(3);
      });
    });
  });

  describe('Critical: Polling Precision', () => {
    test.skip('polls at exactly 2-second intervals when downloads are active', async () => {
      const activeDownload = [{
        id: '1',
        name: 'model.bin',
        is_downloading: true,
        dl_requested_at: '2024-01-01'
      }];

      // Keep returning active download to maintain polling
      mockedAxios.get.mockResolvedValue({ data: activeDownload });

      const { result } = renderHook(() => React.useContext(ModelFilesContext), { wrapper });

      // Wait for initial mount and verify polling started
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
        expect(result.current.hasActiveDownloads).toBe(true);
      });

      // Wait for state to settle
      await act(async () => {
        await new Promise(resolve => setTimeout(resolve, 50));
      });

      // Clear the mock to count only polling calls
      mockedAxios.get.mockClear();

      // Test 2-second interval
      act(() => jest.advanceTimersByTime(2000));
      
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(1);
      });

      // Test another interval
      act(() => jest.advanceTimersByTime(2000));
      
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(2);
      });

      // Test third interval
      act(() => jest.advanceTimersByTime(2000));
      
      await waitFor(() => {
        expect(mockedAxios.get).toHaveBeenCalledTimes(3);
      });
    });
  });

  describe('Critical: Download State Detection on Mount', () => {
    test.skip('correctly identifies download states from API response', async () => {
      // Test various states by mocking different API responses
      const testCases = [
        {
          name: 'active download',
          files: [{ id: '1', is_downloading: true, dl_requested_at: '2024-01-01', name: 'test.bin' }],
          expectedActive: true
        },
        {
          name: 'queued download',
          files: [{ id: '1', is_downloading: false, dl_requested_at: '2024-01-01', name: 'test.bin' }],
          expectedActive: true
        },
        {
          name: 'completed download',
          files: [{ id: '1', is_downloading: false, dl_requested_at: null, storage_location: '/path', name: 'test.bin' }],
          expectedActive: false
        }
      ];

      for (const testCase of testCases) {
        // Clear mocks for each test
        jest.clearAllMocks();
        
        // Set up the mock response
        mockedAxios.get.mockResolvedValueOnce({ data: testCase.files });

        // Render a fresh instance
        const { result } = renderHook(
          () => React.useContext(ModelFilesContext),
          { wrapper }
        );

        // Wait for the initial fetch to complete
        await waitFor(() => {
          expect(mockedAxios.get).toHaveBeenCalledWith('http://localhost:8000/model_files/download_status/');
        });

        // Check the active downloads state
        expect(result.current.hasActiveDownloads).toBe(testCase.expectedActive);
      }
    });
  });
});