import React from 'react';
import { act, waitFor, screen } from '@testing-library/react';
import { renderWithProviders } from '../../../test-utils';
import ModelDetail from '../ModelDetail';
import ModelFilesContext from '../ModelFilesContext';
import axios from 'axios';

// Mock axios
jest.mock('axios');
const mockedAxios = axios;

// Mock react-router-dom
jest.mock('react-router-dom', () => ({
  ...jest.requireActual('react-router-dom'),
  useParams: () => ({ model_id: 'test-model-123' })
}));

// Mock the BASE_URL
jest.mock('../../../const', () => ({
  BASE_URL: 'http://localhost:8000'
}));

// Mock child components to simplify testing
jest.mock('../ModelConfigList', () => {
  const MockModelConfigList = () => <div data-testid="model-config-list">Model Config List</div>;
  MockModelConfigList.displayName = 'MockModelConfigList';
  return MockModelConfigList;
});

jest.mock('../ModelDeploymentList', () => {
  const MockModelDeploymentList = () => <div data-testid="model-deployment-list">Model Deployment List</div>;
  MockModelDeploymentList.displayName = 'MockModelDeploymentList';
  return MockModelDeploymentList;
});

jest.mock('../AddModelConfigForm', () => {
  const MockAddModelConfigForm = () => <div data-testid="add-model-config-form">Add Model Config Form</div>;
  MockAddModelConfigForm.displayName = 'MockAddModelConfigForm';
  return MockAddModelConfigForm;
});

describe('ModelDetail', () => {
  const mockModel = {
    id: 'test-model-123',
    name: 'Test Model',
    family: 'GPT',
    purpose: 'Text Generation',
    version: '1.0',
    author: 'Test Author',
    description: 'Test Description',
    created_timestamp: '2024-01-01T00:00:00Z',
    modified_timestamp: '2024-01-02T00:00:00Z',
    m_files: [
      {
        id: 'file-1',
        name: 'model.bin',
        size: 1024 * 1024 * 1024, // 1GB
        storage_location: '/models/test-model/model.bin',
        dl_requested_at: null
      },
      {
        id: 'file-2',
        name: 'config.json',
        size: 1024,
        storage_location: null,
        dl_requested_at: '2024-01-03T00:00:00Z'
      }
    ]
  };

  const mockConfigs = [
    { id: 'config-1', name: 'Default Config', default: true },
    { id: 'config-2', name: 'Custom Config', default: false }
  ];

  const mockStatusModelFiles = [
    {
      id: 'file-2',
      m_id: 'test-model-123',
      name: 'config.json',
      size: 1024,
      is_downloading: true,
      download_percentage: 45,
      download_throughput: '1.2 MB/s',
      dl_requested_at: '2024-01-03T00:00:00Z',
      storage_location: null
    }
  ];

  const defaultContextValue = {
    statusModelFiles: [],
    showDownloadStatus: false,
    setShowDownloadStatus: jest.fn(),
    toast: { open: false, message: '', severity: 'info' },
    showToast: jest.fn(),
    closeToast: jest.fn(),
    refreshModelList: jest.fn(),
    modelListRefreshTrigger: 0,
    startPolling: jest.fn(),
    hasActiveDownloads: false
  };

  beforeEach(() => {
    jest.clearAllMocks();
    
    // Setup default axios responses
    mockedAxios.get.mockImplementation((url) => {
      if (url.includes('/configs')) {
        return Promise.resolve({ data: mockConfigs });
      }
      if (url.includes('/models/test-model-123')) {
        return Promise.resolve({ data: mockModel });
      }
      return Promise.reject(new Error('Unknown URL'));
    });
  });

  const renderModelDetail = (contextValue = defaultContextValue) => {
    return renderWithProviders(
      <ModelFilesContext.Provider value={contextValue}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );
  };

  it('should render model details correctly', async () => {
    renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText('Test Model')).toBeInTheDocument();
    });

    expect(screen.getByText('GPT')).toBeInTheDocument();
    expect(screen.getByText('Text Generation')).toBeInTheDocument();
    expect(screen.getByText('Test Author')).toBeInTheDocument();
    expect(screen.getByText('Test Description')).toBeInTheDocument();
  });

  it('should display model files with their status', async () => {
    renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText('model.bin')).toBeInTheDocument();
      expect(screen.getByText('config.json')).toBeInTheDocument();
    });

    // Check file sizes are formatted correctly
    expect(screen.getByText('1.0G')).toBeInTheDocument(); // 1GB file
    expect(screen.getByText('1.0K')).toBeInTheDocument(); // 1KB file
  });

  it('should show download progress from context without re-fetching', async () => {
    const { rerender } = renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: mockStatusModelFiles
    });

    await waitFor(() => {
      expect(screen.getByText('model.bin')).toBeInTheDocument();
    });

    // Verify download status is shown (without percentage or throughput)
    expect(screen.getByText('downloading')).toBeInTheDocument();

    // Verify axios was called only once for initial load
    expect(mockedAxios.get).toHaveBeenCalledTimes(2); // model + configs

    // Update download progress
    const updatedStatus = [{
      ...mockStatusModelFiles[0],
      download_percentage: 75,
      download_throughput: '2.5 MB/s'
    }];

    rerender(
      <ModelFilesContext.Provider value={{
        ...defaultContextValue,
        statusModelFiles: updatedStatus
      }}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );

    // Status should remain downloading (no percentage shown anymore)
    await waitFor(() => {
      expect(screen.getByText('downloading')).toBeInTheDocument();
    });

    // Verify no additional API calls were made
    expect(mockedAxios.get).toHaveBeenCalledTimes(2);
  });

  it('should NOT re-fetch when modelListRefreshTrigger changes', async () => {
    const { rerender } = renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText('Test Model')).toBeInTheDocument();
    });

    expect(mockedAxios.get).toHaveBeenCalledTimes(2);

    // Update the refresh trigger (this used to cause re-fetch)
    rerender(
      <ModelFilesContext.Provider value={{
        ...defaultContextValue,
        modelListRefreshTrigger: 1
      }}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );

    // Wait a bit to ensure no re-fetch happens
    await act(async () => {
      await new Promise(resolve => setTimeout(resolve, 100));
    });

    // Verify no additional API calls were made
    expect(mockedAxios.get).toHaveBeenCalledTimes(2);
  });

  it('should handle download completion without re-fetching', async () => {
    const { rerender } = renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: mockStatusModelFiles
    });

    await waitFor(() => {
      expect(screen.getByText('downloading')).toBeInTheDocument();
    });

    // Simulate download completion
    const completedStatus = [{
      ...mockStatusModelFiles[0],
      is_downloading: false,
      download_percentage: 100,
      storage_location: '/models/test-model/config.json',
      dl_requested_at: null
    }];

    rerender(
      <ModelFilesContext.Provider value={{
        ...defaultContextValue,
        statusModelFiles: completedStatus
      }}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );

    // File should show as available without re-fetching
    await waitFor(() => {
      expect(screen.queryByText(/downloading/)).not.toBeInTheDocument();
    });

    // Both files should now have checkboxes checked
    const checkboxes = screen.getAllByRole('checkbox');
    expect(checkboxes).toHaveLength(2);
    expect(checkboxes[0]).toBeChecked(); // model.bin was already downloaded
    expect(checkboxes[1]).toBeChecked(); // config.json just completed

    // Verify no additional API calls
    expect(mockedAxios.get).toHaveBeenCalledTimes(2);
  });

  it('should handle API errors gracefully', async () => {
    mockedAxios.get.mockRejectedValueOnce(new Error('Network error'));

    renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText(/Network error/)).toBeInTheDocument();
    });
  });

  it('should handle 404 errors with specific message', async () => {
    mockedAxios.get.mockRejectedValueOnce({
      response: { status: 404 }
    });

    renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText(/Model not found/)).toBeInTheDocument();
    });
  });

  it('should merge file status correctly with model files', async () => {
    renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: mockStatusModelFiles
    });

    await waitFor(() => {
      expect(screen.getByText('config.json')).toBeInTheDocument();
    });

    // File from context should show download status (without percentage)
    expect(screen.getByText('downloading')).toBeInTheDocument();
    
    // File not in download status should show original state
    const checkboxes = screen.getAllByRole('checkbox');
    expect(checkboxes[0]).toBeChecked(); // model.bin (has storage_location)
    expect(checkboxes[1]).not.toBeChecked(); // config.json (downloading)
  });

  it('should only show queued status for files in active download context', async () => {
    // Model file has dl_requested_at but is not in active downloads
    const modelWithQueuedFile = {
      ...mockModel,
      m_files: [
        ...mockModel.m_files,
        {
          id: 'file-3',
          name: 'completed-file.bin',
          size: 2048,
          storage_location: null,
          dl_requested_at: '2024-01-03T00:00:00Z' // Has request but not in context
        }
      ]
    };

    mockedAxios.get.mockImplementation((url) => {
      if (url.includes('/configs')) {
        return Promise.resolve({ data: mockConfigs });
      }
      if (url.includes('/models/test-model-123')) {
        return Promise.resolve({ data: modelWithQueuedFile });
      }
      return Promise.reject(new Error('Unknown URL'));
    });

    renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: [] // No active downloads
    });

    await waitFor(() => {
      expect(screen.getByText('completed-file.bin')).toBeInTheDocument();
    });

    // Should NOT show as queued since it's not in active context
    expect(screen.queryByText('queued')).not.toBeInTheDocument();
    
    // Should show as available (checked) since it has dl_requested_at but not in active status
    const checkboxes = screen.getAllByRole('checkbox');
    expect(checkboxes).toHaveLength(3); // model.bin, config.json, completed-file.bin
    expect(checkboxes[0]).toBeChecked(); // model.bin (has storage_location)
    expect(checkboxes[1]).toBeChecked(); // config.json (has dl_requested_at, not in active status)
    expect(checkboxes[2]).toBeChecked(); // completed-file.bin (has dl_requested_at, not in active)
  });

  it('should calculate isModelDownloading correctly', async () => {
    const { rerender } = renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: []
    });

    await waitFor(() => {
      expect(screen.getByText('Test Model')).toBeInTheDocument();
    });

    // Initially no downloads
    expect(screen.getByTestId('model-config-list')).toBeInTheDocument();

    // Add active download
    rerender(
      <ModelFilesContext.Provider value={{
        ...defaultContextValue,
        statusModelFiles: mockStatusModelFiles
      }}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );

    // ModelConfigList should still be rendered (it receives isModelDownloading prop)
    expect(screen.getByTestId('model-config-list')).toBeInTheDocument();
  });

  it('should correctly identify file availability states', async () => {
    const modelWithVariousStates = {
      ...mockModel,
      m_files: [
        // Case 1: File with storage_location - always available
        {
          id: 'file-1',
          name: 'downloaded.bin',
          size: 1024,
          storage_location: '/path/to/file',
          dl_requested_at: '2024-01-01T00:00:00Z' // Should be ignored
        },
        // Case 2: File with dl_requested_at but not in active status - likely completed
        {
          id: 'file-2', 
          name: 'completed.bin',
          size: 2048,
          storage_location: null,
          dl_requested_at: '2024-01-02T00:00:00Z'
        },
        // Case 3: File with no download info - not available
        {
          id: 'file-3',
          name: 'not-downloaded.bin', 
          size: 4096,
          storage_location: null,
          dl_requested_at: null
        }
      ]
    };

    mockedAxios.get.mockImplementation((url) => {
      if (url.includes('/configs')) {
        return Promise.resolve({ data: mockConfigs });
      }
      if (url.includes('/models/test-model-123')) {
        return Promise.resolve({ data: modelWithVariousStates });
      }
      return Promise.reject(new Error('Unknown URL'));
    });

    // Add one file to active status
    const activeStatus = [{
      id: 'file-4',
      m_id: 'test-model-123',
      name: 'downloading.bin',
      size: 8192,
      is_downloading: true,
      download_percentage: 50,
      dl_requested_at: '2024-01-03T00:00:00Z',
      storage_location: null
    }];

    renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: activeStatus
    });

    await waitFor(() => {
      expect(screen.getByText('downloaded.bin')).toBeInTheDocument();
    });

    const checkboxes = screen.getAllByRole('checkbox');
    expect(checkboxes).toHaveLength(3);
    
    // File with storage_location is always available
    expect(checkboxes[0]).toBeChecked(); // downloaded.bin
    
    // File with dl_requested_at but not in active status is considered completed
    expect(checkboxes[1]).toBeChecked(); // completed.bin
    
    // File with no download info is not available
    expect(checkboxes[2]).not.toBeChecked(); // not-downloaded.bin
  });

  it('should handle files in active status correctly regardless of original data', async () => {
    // This tests the vocab.json scenario - file has storage_location in original data
    // but is queued in active status
    const modelWithStoredFile = {
      ...mockModel,
      m_files: [{
        id: 'vocab-file',
        name: 'vocab.json',
        size: 2600000, // 2.6MB
        storage_location: '/models/vocab.json', // Has storage location in original data
        dl_requested_at: '2024-01-01T00:00:00Z'
      }]
    };

    mockedAxios.get.mockImplementation((url) => {
      if (url.includes('/configs')) {
        return Promise.resolve({ data: mockConfigs });
      }
      if (url.includes('/models/test-model-123')) {
        return Promise.resolve({ data: modelWithStoredFile });
      }
      return Promise.reject(new Error('Unknown URL'));
    });

    // File is in active status as queued
    const activeStatus = [{
      id: 'vocab-file',
      m_id: 'test-model-123',
      name: 'vocab.json',
      size: 2600000,
      is_downloading: false,
      download_percentage: 0,
      dl_requested_at: '2024-01-03T00:00:00Z', // Still has request
      storage_location: '/models/vocab.json' // Has location but still queued
    }];

    renderModelDetail({
      ...defaultContextValue,
      statusModelFiles: activeStatus
    });

    await waitFor(() => {
      expect(screen.getByText('vocab.json')).toBeInTheDocument();
    });

    // Should show as queued, not checked
    expect(screen.getByText('queued')).toBeInTheDocument();
    
    const checkbox = screen.getByRole('checkbox');
    expect(checkbox).not.toBeChecked(); // Should NOT be checked even though it has storage_location
  });

  it('should only re-fetch when model_id changes', async () => {
    const { rerender } = renderModelDetail();

    await waitFor(() => {
      expect(screen.getByText('Test Model')).toBeInTheDocument();
    });

    expect(mockedAxios.get).toHaveBeenCalledTimes(2);

    // Mock different model_id
    jest.spyOn(require('react-router-dom'), 'useParams').mockReturnValue({ model_id: 'different-model' });
    
    mockedAxios.get.mockImplementation((url) => {
      if (url.includes('/models/different-model')) {
        return Promise.resolve({ 
          data: { ...mockModel, id: 'different-model', name: 'Different Model' } 
        });
      }
      if (url.includes('/models/different-model/configs')) {
        return Promise.resolve({ data: [] });
      }
      return Promise.reject(new Error('Unknown URL'));
    });

    rerender(
      <ModelFilesContext.Provider value={defaultContextValue}>
        <ModelDetail />
      </ModelFilesContext.Provider>
    );

    await waitFor(() => {
      expect(screen.getByText('Different Model')).toBeInTheDocument();
    });

    // Should have made 2 more calls (model + configs)
    expect(mockedAxios.get).toHaveBeenCalledTimes(4);
  });
});