import { render, screen, waitFor, within } from "@testing-library/react";
import userEvent from "@testing-library/user-event";
import axios from "axios";
import React from "react";
import ModelDeploymentList from "../ModelDeploymentList";

// Mock axios
jest.mock("axios");
const mockedAxios = axios;

// Mock child components
jest.mock("../../common/Spinner", () => {
	return function MockSpinner() {
		return <div data-testid="spinner">Loading...</div>;
	};
});

jest.mock("../../common/ErrorComponent", () => {
	// eslint-disable-next-line react/prop-types
	return function MockErrorComponent({ message }) {
		return <div data-testid="error-component">Error: {message}</div>;
	};
});

/* eslint-disable react/prop-types */
jest.mock("../ModelDeploymentDetail", () => {
	return function MockModelDeploymentDetailModal({
		deploymentId,
		isOpen,
		onClose,
		onDeploymentStopped,
	}) {
		return isOpen ? (
			<div data-testid="deployment-detail-modal">
				<div>Deployment ID: {deploymentId}</div>
				<button onClick={onClose}>Close</button>
				<button onClick={onDeploymentStopped}>Deployment Stopped</button>
			</div>
		) : null;
	};
});
/* eslint-enable react/prop-types */

// Mock BASE_URL
jest.mock("../../../const", () => ({
	BASE_URL: "http://localhost:8000",
	API_PORT: "7777",
	LAB_PORT: "8888",
	isDefaultPort: (port) => port === "" || port === "80" || port === "443",
}));

// Mock RoutingConfigContext with configurable mode
let mockRoutingMode = "port";
jest.mock("../../../context/RoutingConfigContext", () => ({
	useRoutingConfig: () => ({
		routingMode: mockRoutingMode,
		rawRoutingMode: mockRoutingMode,
		isLoading: false,
		error: null,
		isDualMode: mockRoutingMode === "dual",
		normalizedMode: mockRoutingMode === "path" ? "path" : "port",
	}),
}));

// Mock window.location
delete window.location;
window.location = {
	hostname: "localhost",
	protocol: "https:",
	port: "",
	origin: "https://localhost",
};

// Mock MUI Dialog to make it testable
jest.mock("@mui/material/Dialog", () => {
	// eslint-disable-next-line react/prop-types, no-unused-vars
	return function MockDialog({ open, children, onClose }) {
		return open ? <div role="dialog">{children}</div> : null;
	};
});

// Test data factory
const createMockDeployment = (overrides = {}) => ({
	id: `dep-${Math.random().toString(36).substr(2, 9)}`,
	m_id: "model-123",
	m_name: "Test Model",
	engine_name: "vllm",
	m_config_name: "Config A",
	lb_port: 8080,
	status: "RUNNING",
	...overrides,
});

describe("ModelDeploymentList", () => {
	beforeEach(() => {
		jest.clearAllMocks();
		// Default mock for successful empty response
		mockedAxios.get.mockResolvedValue({ data: [] });
		mockedAxios.delete.mockResolvedValue({ data: { success: true } });
	});

	describe("Rendering States", () => {
		test("renders loading state initially", () => {
			render(<ModelDeploymentList />);
			expect(screen.getByTestId("spinner")).toBeInTheDocument();
		});

		test("renders error state on fetch failure", async () => {
			mockedAxios.get.mockRejectedValueOnce(new Error("Network error"));

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByTestId("error-component")).toBeInTheDocument();
				expect(screen.getByText("Error: Network error")).toBeInTheDocument();
			});
		});

		test("renders empty state when no deployments", async () => {
			mockedAxios.get.mockResolvedValueOnce({ data: [] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(
					screen.getByText("No deployed models found"),
				).toBeInTheDocument();
			});
		});

		test("renders deployment table with data", async () => {
			const mockDeployments = [
				createMockDeployment({ m_name: "GPT-4" }),
				createMockDeployment({ m_name: "Claude-3" }),
			];

			mockedAxios.get.mockResolvedValueOnce({ data: mockDeployments });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("GPT-4")).toBeInTheDocument();
				expect(screen.getByText("Claude-3")).toBeInTheDocument();
			});
		});

		test("shows selection checkboxes when onSelectionChange provided", async () => {
			const mockDeployments = [createMockDeployment()];
			mockedAxios.get.mockResolvedValueOnce({ data: mockDeployments });

			const handleSelectionChange = jest.fn();
			render(<ModelDeploymentList onSelectionChange={handleSelectionChange} />);

			await waitFor(() => {
				// Should have header checkbox and row checkbox
				const checkboxes = screen.getAllByRole("checkbox");
				expect(checkboxes).toHaveLength(2);
			});
		});

		test("hides selection checkboxes when onSelectionChange not provided", async () => {
			const mockDeployments = [createMockDeployment()];
			mockedAxios.get.mockResolvedValueOnce({ data: mockDeployments });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const checkboxes = screen.queryAllByRole("checkbox");
				expect(checkboxes).toHaveLength(0);
			});
		});
	});

	describe("Data Fetching", () => {
		test("fetches deployments on mount", async () => {
			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledWith(
					"http://localhost:8000/serving/deployments",
				);
			});
		});

		test("includes model_id in query when provided", async () => {
			render(<ModelDeploymentList model_id="model-456" />);

			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledWith(
					"http://localhost:8000/serving/deployments?model_id=model-456",
				);
			});
		});

		test("refetches when refreshDeployments prop changes", async () => {
			const { rerender } = render(
				<ModelDeploymentList refreshDeployments={1} />,
			);

			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledTimes(1);
			});

			rerender(<ModelDeploymentList refreshDeployments={2} />);

			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledTimes(2);
			});
		});
	});

	describe("Deployment Display", () => {
		test("displays deployment information correctly", async () => {
			const deployment = createMockDeployment({
				id: "dep-123",
				m_name: "Test Model",
				engine_name: "vllm",
				m_config_name: "Custom Config",
				lb_port: 9090,
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const row = screen.getByText("Test Model").closest("tr");
				expect(row).toBeInTheDocument();
				expect(within(row).getByText("vllm")).toBeInTheDocument();
				expect(within(row).getByText("Custom Config")).toBeInTheDocument();
				expect(
					within(row).getByText("https://localhost:9090"),
				).toBeInTheDocument();
				expect(
					within(row).getByText("/v1/chat/completions"),
				).toBeInTheDocument();
			});
		});

		test("shows defaults for missing fields", async () => {
			const deployment = createMockDeployment({
				engine_name: null,
				m_config_name: null,
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Unknown")).toBeInTheDocument();
				expect(screen.getByText("Default Config")).toBeInTheDocument();
			});
		});

		test("filters out stopped deployments", async () => {
			const deployments = [
				createMockDeployment({ m_name: "Active Model", status: "RUNNING" }),
				createMockDeployment({ m_name: "Stopped Model", status: "STOPPED" }),
			];

			mockedAxios.get.mockResolvedValueOnce({ data: deployments });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Active Model")).toBeInTheDocument();
				expect(screen.queryByText("Stopped Model")).not.toBeInTheDocument();
			});
		});

		test("shows Failed status for failed deployments", async () => {
			const deployment = createMockDeployment({
				status: "FAILED",
				lb_port: 8080,
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Failed")).toBeInTheDocument();
				expect(
					screen.queryByText("/v1/chat/completions"),
				).not.toBeInTheDocument();
			});
		});

		test("shows deployments with INITIALIZING status", async () => {
			const deployment = createMockDeployment({
				id: "dep-init",
				status: "INITIALIZING",
				m_name: "Initializing Model",
				lb_port: 8080,
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const row = screen.getByText("Initializing Model").closest("tr");
				expect(row).toBeInTheDocument();
				const endpoint = within(row).getByText("/v1/chat/completions");
				expect(endpoint.closest("a")).toHaveAttribute(
					"href",
					"https://localhost:8080/v1/chat/completions",
				);
			});
		});
	});

	describe("API Links", () => {
		test("displays chat completions endpoint", async () => {
			const deployment = createMockDeployment({
				id: "dep-chat",
				lb_port: 8080,
				m_name: "Chat Model",
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const row = screen.getByText("Chat Model").closest("tr");
				const endpoint = within(row).getByText("/v1/chat/completions");
				expect(endpoint.closest("a")).toHaveAttribute(
					"href",
					"https://localhost:8080/v1/chat/completions",
				);
			});
		});

		test("shows API Docs link only for vllm engine", async () => {
			const vllmDeployment = createMockDeployment({
				id: "dep-vllm",
				engine_name: "vllm",
				lb_port: 8080,
			});
			const otherDeployment = createMockDeployment({
				id: "dep-llama",
				engine_name: "llamacpp",
				lb_port: 8081,
			});

			mockedAxios.get.mockResolvedValueOnce({
				data: [vllmDeployment, otherDeployment],
			});

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const docsLinks = screen.getAllByText("/docs");
				expect(docsLinks).toHaveLength(1);
				// In port mode without explicit access_path, the openapi path is just /openapi.json
				expect(docsLinks[0].closest("a")).toHaveAttribute(
					"href",
					"https://localhost:8080/docs?url=%2Fopenapi.json",
				);
			});
		});

		test("handles deployments without port by showing runtime path when access_path provided", async () => {
			// In port mode, path endpoints only shown when backend provides explicit access_path
			const deployment = createMockDeployment({
				id: "dep-nopath",
				m_name: "No Path Model",
				lb_port: 0,
				access_path: "/runtime/models/dep-nopath",
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await screen.findByText("No Path Model");
			const row = screen.getByText("No Path Model").closest("tr");
			const endpoint = within(row).getByText("/v1/chat/completions");
			expect(endpoint.closest("a")).toHaveAttribute(
				"href",
				"https://localhost/runtime/models/dep-nopath/v1/chat/completions",
			);
		});

		test("displays path-based access information when access_path is provided", async () => {
			// In port mode, path endpoints require explicit access_path from backend
			const deployment = createMockDeployment({
				id: "dep-path",
				lb_port: 0,
				serve_path: "/runtime/models/dep-path",
				access_path: "/runtime/models/dep-path",
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const row = screen.getByText("Test Model").closest("tr");
				const docsLink = within(row).getByText("/docs");
				expect(docsLink.closest("a")).toHaveAttribute(
					"href",
					"https://localhost/runtime/models/dep-path/docs?url=%2Fruntime%2Fmodels%2Fdep-path%2Fopenapi.json",
				);
			});
		});
	});

	describe("Selection Functionality", () => {
		test("select all checkbox selects all active deployments", async () => {
			const user = userEvent.setup();
			const deployments = [
				createMockDeployment({
					id: "dep-1",
					m_name: "Model 1",
					status: "RUNNING",
				}),
				createMockDeployment({
					id: "dep-2",
					m_name: "Model 2",
					status: "RUNNING",
				}),
				createMockDeployment({
					id: "dep-3",
					m_name: "Model 3",
					status: "STOPPED",
				}), // Should be filtered out
			];

			mockedAxios.get.mockResolvedValueOnce({ data: deployments });

			const handleSelectionChange = jest.fn();
			render(<ModelDeploymentList onSelectionChange={handleSelectionChange} />);

			await waitFor(() => {
				expect(screen.getByText("Model 1")).toBeInTheDocument();
			});

			const headerCheckbox = screen.getAllByRole("checkbox")[0];
			await user.click(headerCheckbox);

			expect(handleSelectionChange).toHaveBeenCalledWith(["dep-1", "dep-2"]);
		});

		test("individual checkbox toggles selection", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment({ id: "dep-1" });

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			const handleSelectionChange = jest.fn();
			render(
				<ModelDeploymentList
					onSelectionChange={handleSelectionChange}
					selectedDeployments={[]}
				/>,
			);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			const rowCheckbox = screen.getAllByRole("checkbox")[1];
			await user.click(rowCheckbox);

			expect(handleSelectionChange).toHaveBeenCalledWith(["dep-1"]);
		});

		test("shows indeterminate state for partial selection", async () => {
			const deployments = [
				createMockDeployment({ id: "dep-1", m_name: "Model 1" }),
				createMockDeployment({ id: "dep-2", m_name: "Model 2" }),
			];

			mockedAxios.get.mockResolvedValueOnce({ data: deployments });

			render(
				<ModelDeploymentList
					onSelectionChange={jest.fn()}
					selectedDeployments={["dep-1"]}
				/>,
			);

			await waitFor(() => {
				expect(screen.getByText("Model 1")).toBeInTheDocument();
			});

			// For MUI checkboxes, the indeterminate state is represented by data attributes
			const checkboxes = screen.getAllByRole("checkbox");
			// headerCheckbox is checkboxes[0] - used implicitly in length check

			// Check if the checkbox has both checked and unchecked items (indeterminate state)
			expect(checkboxes).toHaveLength(3); // header + 2 row checkboxes
			expect(checkboxes[1]).toBeChecked(); // First row is selected
			expect(checkboxes[2]).not.toBeChecked(); // Second row is not selected
		});
	});

	describe("Stop Deployment", () => {
		test("opens confirmation dialog on stop click", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment({ m_name: "Model to Stop" });

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Model to Stop")).toBeInTheDocument();
			});

			const stopButton = screen.getByText("Stop");
			await user.click(stopButton);

			await waitFor(() => {
				// Use getAllByText since "Stop Deployment" appears as both dialog title and button text
				const stopDeploymentElements = screen.getAllByText("Stop Deployment");
				expect(stopDeploymentElements.length).toBeGreaterThan(0);
			});

			expect(
				screen.getByText(/Are you sure you want to stop the deployment for/),
			).toBeInTheDocument();

			// Use getAllByText since the model name appears in both table and dialog
			const modelNameElements = screen.getAllByText("Model to Stop");
			expect(modelNameElements.length).toBeGreaterThan(1);
		});

		test("cancels stop operation", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment();

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			await user.click(screen.getByText("Stop"));

			await waitFor(() => {
				const stopDeploymentElements = screen.getAllByText("Stop Deployment");
				expect(stopDeploymentElements.length).toBeGreaterThan(0);
			});

			await user.click(screen.getByText("Cancel"));

			await waitFor(() => {
				expect(screen.queryByRole("dialog")).not.toBeInTheDocument();
			});

			expect(mockedAxios.delete).not.toHaveBeenCalled();
		});

		test("confirms stop operation and refreshes list", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment({ id: "dep-123" });

			mockedAxios.get
				.mockResolvedValueOnce({ data: [deployment] })
				.mockResolvedValueOnce({ data: [] }); // After stop

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			await user.click(screen.getByText("Stop"));

			const confirmButton = screen.getByRole("button", {
				name: "Stop Deployment",
			});
			await user.click(confirmButton);

			await waitFor(() => {
				expect(mockedAxios.delete).toHaveBeenCalledWith(
					"http://localhost:8000/serving/deployment/dep-123",
					{ params: { force: true } },
				);
			});

			// Should refresh the list
			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledTimes(2);
				expect(
					screen.getByText("No deployed models found"),
				).toBeInTheDocument();
			});
		});

		test("shows loading state while stopping", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment({ id: "dep-123" });

			// Delay the delete response
			mockedAxios.delete.mockImplementation(
				() =>
					new Promise((resolve) =>
						setTimeout(() => resolve({ data: { success: true } }), 100),
					),
			);

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			const stopButton = screen.getByText("Stop");
			await user.click(stopButton);

			const confirmButton = screen.getByRole("button", {
				name: "Stop Deployment",
			});
			await user.click(confirmButton);

			// Should show loading state
			expect(screen.getByText("Stopping...")).toBeInTheDocument();

			// Button in table should also show loading
			const tableStopButton = screen.getByText("Stopping");
			expect(tableStopButton).toBeInTheDocument();
		});

		test("handles stop error with alert", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment();

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });
			mockedAxios.delete.mockRejectedValueOnce(new Error("Stop failed"));

			// Mock window.alert
			const alertSpy = jest.spyOn(window, "alert").mockImplementation();

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			await user.click(screen.getByText("Stop"));
			await user.click(screen.getByRole("button", { name: "Stop Deployment" }));

			await waitFor(() => {
				expect(alertSpy).toHaveBeenCalledWith(
					"Failed to stop deployment. Please try again.",
				);
			});

			alertSpy.mockRestore();
		});
	});

	describe("Details Modal", () => {
		test("opens detail modal on Details click", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment({ id: "dep-123" });

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			const detailsButton = screen.getByText("Details");
			await user.click(detailsButton);

			expect(screen.getByTestId("deployment-detail-modal")).toBeInTheDocument();
			expect(screen.getByText("Deployment ID: dep-123")).toBeInTheDocument();
		});

		test("closes modal on close callback", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment();

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			await user.click(screen.getByText("Details"));

			const closeButton = within(
				screen.getByTestId("deployment-detail-modal"),
			).getByText("Close");
			await user.click(closeButton);

			expect(
				screen.queryByTestId("deployment-detail-modal"),
			).not.toBeInTheDocument();
		});

		test("refreshes list when deployment stopped from modal", async () => {
			const user = userEvent.setup();
			const deployment = createMockDeployment();

			mockedAxios.get
				.mockResolvedValueOnce({ data: [deployment] })
				.mockResolvedValueOnce({ data: [] }); // After refresh

			render(<ModelDeploymentList />);

			await waitFor(() => {
				expect(screen.getByText("Test Model")).toBeInTheDocument();
			});

			await user.click(screen.getByText("Details"));

			const stoppedButton = within(
				screen.getByTestId("deployment-detail-modal"),
			).getByText("Deployment Stopped");
			await user.click(stoppedButton);

			await waitFor(() => {
				expect(mockedAxios.get).toHaveBeenCalledTimes(2);
			});
		});
	});

	describe("Model Link Navigation", () => {
		test("model name links to model page", async () => {
			const deployment = createMockDeployment({
				m_id: "model-456",
				m_name: "GPT-4",
			});

			mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

			render(<ModelDeploymentList />);

			await waitFor(() => {
				const modelLink = screen.getByRole("link", { name: "GPT-4" });
				expect(modelLink).toHaveAttribute("href", "/models/model-456");
			});
		});
	});

	describe("Routing Mode Presentation", () => {
		afterEach(() => {
			// Reset to default routing mode after each test
			mockRoutingMode = "port";
		});

		describe("port mode", () => {
			beforeEach(() => {
				mockRoutingMode = "port";
			});

			test("displays port-based URL when in port mode", async () => {
				const deployment = createMockDeployment({
					id: "dep-port-test",
					m_name: "Port Model",
					lb_port: 9090,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(
						screen.getByText("https://localhost:9090"),
					).toBeInTheDocument();
				});
			});

			test("falls back to path URL when no port available but access_path provided", async () => {
				// In port mode, fallback to path only works when backend provides explicit access_path
				const deployment = createMockDeployment({
					id: "dep-no-port",
					m_name: "No Port Model",
					lb_port: 0,
					access_path: "/runtime/models/dep-no-port",
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(
						screen.getByText("https://localhost/runtime/models/dep-no-port"),
					).toBeInTheDocument();
				});
			});

			test("shows unavailable when no port and no access_path in port mode", async () => {
				// In port mode without explicit access_path, no path endpoints are generated
				const deployment = createMockDeployment({
					id: "dep-unavailable",
					m_name: "Unavailable Model",
					lb_port: 0,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(screen.getByText("Unavailable Model")).toBeInTheDocument();
				});

				// Should show "unavailable" instead of a path URL
				const row = screen.getByText("Unavailable Model").closest("tr");
				expect(within(row).getByText("unavailable")).toBeInTheDocument();
			});
		});

		describe("path mode", () => {
			beforeEach(() => {
				mockRoutingMode = "path";
			});

			test("displays path-based URL when in path mode", async () => {
				const deployment = createMockDeployment({
					id: "dep-path-test",
					m_name: "Path Model",
					lb_port: 9090,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(
						screen.getByText("https://localhost/runtime/models/dep-path-test"),
					).toBeInTheDocument();
				});
			});

			test("does not show port URL even when port is available", async () => {
				const deployment = createMockDeployment({
					id: "dep-path-only",
					m_name: "Path Only Model",
					lb_port: 8888,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(screen.getByText("Path Only Model")).toBeInTheDocument();
				});

				// Should show path URL, not port URL
				expect(
					screen.getByText("https://localhost/runtime/models/dep-path-only"),
				).toBeInTheDocument();
				expect(
					screen.queryByText("https://localhost:8888"),
				).not.toBeInTheDocument();
			});
		});

		describe("dual mode", () => {
			beforeEach(() => {
				mockRoutingMode = "dual";
			});

			test("displays both port and path URLs when in dual mode", async () => {
				const deployment = createMockDeployment({
					id: "dep-dual-test",
					m_name: "Dual Model",
					lb_port: 7777,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(screen.getByText("Dual Model")).toBeInTheDocument();
				});

				// Should show both URLs
				expect(screen.getByText("https://localhost:7777")).toBeInTheDocument();
				expect(
					screen.getByText("https://localhost/runtime/models/dep-dual-test"),
				).toBeInTheDocument();
			});

			test("displays Port and Path labels when both routes available", async () => {
				const deployment = createMockDeployment({
					id: "dep-dual-labels",
					m_name: "Dual Labels Model",
					lb_port: 6666,
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(screen.getByText("Dual Labels Model")).toBeInTheDocument();
				});

				// Should show labels for each route type
				expect(screen.getByText("Port")).toBeInTheDocument();
				expect(screen.getByText("Path")).toBeInTheDocument();
			});

			test("does not show labels when only one route available in dual mode", async () => {
				const deployment = createMockDeployment({
					id: "dep-dual-single",
					m_name: "Single Route Model",
					lb_port: 0, // No port, only path available
				});

				mockedAxios.get.mockResolvedValueOnce({ data: [deployment] });

				render(<ModelDeploymentList />);

				await waitFor(() => {
					expect(screen.getByText("Single Route Model")).toBeInTheDocument();
				});

				// Should show path URL without labels since only one route
				expect(
					screen.getByText("https://localhost/runtime/models/dep-dual-single"),
				).toBeInTheDocument();
				expect(screen.queryByText("Port")).not.toBeInTheDocument();
				expect(screen.queryByText("Path")).not.toBeInTheDocument();
			});
		});
	});
});
