import { ThemeProvider } from "@mui/material/styles";
import { fireEvent, render, screen, waitFor } from "@testing-library/react";
import axios from "axios";
// biome-ignore lint/correctness/noUnusedImports: React is required for JSX transform
import React from "react";
import theme from "../../../Theme";
import ModelDeploymentDetailModal from "../ModelDeploymentDetail";

// Mock axios
jest.mock("axios");

// Mock the TestConversationModal component
jest.mock("../TestConversationModal", () => {
	return function MockTestConversationModal() {
		return <div>Test Conversation Modal</div>;
	};
});

// Mock the ContainerLogViewer component to avoid LogPatternAlert issues
jest.mock("../ContainerLogViewer", () => {
	return function MockContainerLogViewer() {
		return <div data-testid="container-log-viewer">Container Log Viewer</div>;
	};
});

// Mock RoutingConfigContext with configurable mode
let mockRoutingMode = "port";
jest.mock("../../../context/RoutingConfigContext", () => ({
	useRoutingConfig: () => ({
		routingMode: mockRoutingMode,
		rawRoutingMode: mockRoutingMode,
		isLoading: false,
		error: null,
		isDualMode: mockRoutingMode === "dual",
		normalizedMode: mockRoutingMode === "dual" ? "port" : mockRoutingMode,
	}),
}));

describe("ModelDeploymentDetailModal", () => {
	beforeAll(() => {
		delete window.location;
		window.location = {
			hostname: "localhost",
			protocol: "https:",
			port: "",
			origin: "https://localhost",
		};
	});

	const mockOnClose = jest.fn();
	const mockOnDeploymentStopped = jest.fn();

	const mockDeploymentData = {
		id: "deployment-123",
		m_name: "Test Model",
		m_config_name: "Test Config",
		lb_port: 8080,
		duration: 60,
		engine_name: "vllm",
		status: "DEPLOYED",
		autoscaling: true,
		min_copies: 1,
		starting_copies: 2,
		max_copies: 4,
		gpu_allocation: true,
		vram_allocation: 4000000000,
		instances: [
			{
				id: "1",
				host_name: "localhost",
				listen_port: 8081,
				container_id: "abc123",
				deployed_at: new Date().toISOString(),
			},
		],
	};

	beforeEach(() => {
		jest.clearAllMocks();
		axios.get.mockResolvedValue({ data: mockDeploymentData });
	});

	const renderComponent = (props = {}) => {
		const defaultProps = {
			deploymentId: "test-deployment-id",
			isOpen: true,
			onClose: mockOnClose,
			onDeploymentStopped: mockOnDeploymentStopped,
			...props,
		};

		return render(
			<ThemeProvider theme={theme}>
				<ModelDeploymentDetailModal {...defaultProps} />
			</ThemeProvider>,
		);
	};

	it("renders the modal when open", async () => {
		renderComponent();

		await waitFor(() => {
			expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
		});
	});

	it("does not render when isOpen is false", () => {
		renderComponent({ isOpen: false });

		expect(screen.queryByText("Deployment Detail")).not.toBeInTheDocument();
	});

	describe("Close button functionality", () => {
		it("renders close button with proper attributes", async () => {
			renderComponent();

			await waitFor(() => {
				const closeButton = screen.getByLabelText("close");
				expect(closeButton).toBeInTheDocument();
				expect(closeButton).toHaveClass("MuiIconButton-sizeSmall");
			});
		});

		it("calls onClose when close button is clicked", async () => {
			renderComponent();

			await waitFor(() => {
				const closeButton = screen.getByLabelText("close");
				fireEvent.click(closeButton);
				expect(mockOnClose).toHaveBeenCalledTimes(1);
			});
		});

		it("close button is positioned correctly in the modal header", async () => {
			renderComponent();

			await waitFor(() => {
				const closeButton = screen.getByLabelText("close");
				const modalHeader = closeButton.closest("div");

				// Check that the parent has flex layout
				expect(modalHeader).toHaveStyle({
					display: "flex",
					justifyContent: "space-between",
				});
			});
		});

		it("close button has correct size attribute", async () => {
			renderComponent();

			await waitFor(() => {
				const closeButton = screen.getByLabelText("close");
				expect(closeButton).toHaveAttribute("class");
				expect(closeButton.className).toMatch(/MuiIconButton-sizeSmall/);
			});
		});
	});

	it("shows loading spinner while fetching data", () => {
		axios.get.mockReturnValueOnce(new Promise(() => {})); // Never resolves
		renderComponent();

		expect(screen.getByRole("progressbar")).toBeInTheDocument();
	});

	it("displays deployment details when data is loaded", async () => {
		renderComponent();

		await waitFor(() => {
			expect(screen.getByText("Test Model")).toBeInTheDocument();
			expect(screen.getByText("Test Config")).toBeInTheDocument();
			expect(screen.getByText("vllm")).toBeInTheDocument();
		});
	});

	it("renders path-based routing details when serve_path is provided", async () => {
		// Use path routing mode for this test
		mockRoutingMode = "path";

		const pathDeployment = {
			...mockDeploymentData,
			lb_port: 0,
			serve_path: "/runtime/models/mock-deployment-path",
		};

		axios.get.mockResolvedValueOnce({ data: pathDeployment });

		renderComponent();

		await waitFor(() => {
			expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
		});

		const baseLink = screen.getByRole("link", {
			name: "https://localhost/runtime/models/deployment-123",
		});
		expect(baseLink).toBeInTheDocument();

		// Reset routing mode for other tests
		mockRoutingMode = "port";
	});

	it("handles error state gracefully", async () => {
		const errorMessage = "Failed to load deployment";
		axios.get.mockRejectedValueOnce(new Error(errorMessage));

		renderComponent();

		await waitFor(() => {
			// Look for the error message in the ErrorComponent
			expect(screen.getByText(`Error: ${errorMessage}`)).toBeInTheDocument();
		});
	});

	it('shows "not found" message when deploymentDetail is null', async () => {
		axios.get.mockResolvedValueOnce({ data: null });

		renderComponent();

		await waitFor(() => {
			expect(
				screen.getByText("Deployment detail not found."),
			).toBeInTheDocument();
		});
	});

	describe("Routing Mode Presentation", () => {
		afterEach(() => {
			// Reset to default routing mode after each test
			mockRoutingMode = "port";
		});

		describe("port mode", () => {
			beforeEach(() => {
				mockRoutingMode = "port";
			});

			it("displays port-based URL when in port mode", async () => {
				axios.get.mockResolvedValueOnce({ data: mockDeploymentData });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should show port-based URL
				expect(
					screen.getByRole("link", { name: "https://localhost:8080" }),
				).toBeInTheDocument();
			});

			it("falls back to path URL when no port available but access_path provided", async () => {
				// In port mode, fallback to path only works when backend provides explicit access_path
				const noPortDeployment = {
					...mockDeploymentData,
					lb_port: 0,
					access_path: "/runtime/models/deployment-123",
				};

				axios.get.mockResolvedValueOnce({ data: noPortDeployment });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should fall back to path-based URL when access_path is provided
				expect(
					screen.getByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).toBeInTheDocument();
			});

			it("shows unavailable when no port and no access_path in port mode", async () => {
				// In port mode without explicit access_path, no path endpoints are generated
				const noPortDeployment = {
					...mockDeploymentData,
					lb_port: 0,
				};

				axios.get.mockResolvedValueOnce({ data: noPortDeployment });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should not show path-based URL without explicit access_path
				expect(
					screen.queryByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).not.toBeInTheDocument();
			});
		});

		describe("path mode", () => {
			beforeEach(() => {
				mockRoutingMode = "path";
			});

			it("displays path-based URL when in path mode", async () => {
				axios.get.mockResolvedValueOnce({ data: mockDeploymentData });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should show path-based URL
				expect(
					screen.getByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).toBeInTheDocument();
			});

			it("does not show port URL even when port is available", async () => {
				axios.get.mockResolvedValueOnce({ data: mockDeploymentData });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should show path URL, not port URL
				expect(
					screen.getByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).toBeInTheDocument();
				expect(
					screen.queryByRole("link", { name: "https://localhost:8080" }),
				).not.toBeInTheDocument();
			});
		});

		describe("dual mode", () => {
			beforeEach(() => {
				mockRoutingMode = "dual";
			});

			it("displays both port and path URLs when in dual mode", async () => {
				axios.get.mockResolvedValueOnce({ data: mockDeploymentData });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should show both URLs
				expect(
					screen.getByRole("link", { name: "https://localhost:8080" }),
				).toBeInTheDocument();
				expect(
					screen.getByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).toBeInTheDocument();
			});

			it("displays Port and Path labels when both routes available", async () => {
				axios.get.mockResolvedValueOnce({ data: mockDeploymentData });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Labels appear in both "API Access" and "OpenAI Compatible" sections
				// So we expect 2 instances of each label
				expect(screen.getAllByText("Port (8080)")).toHaveLength(2);
				expect(screen.getAllByText("Path")).toHaveLength(2);
			});

			it("shows single endpoint group when only path available in dual mode", async () => {
				const noPortDeployment = {
					...mockDeploymentData,
					lb_port: 0,
				};

				axios.get.mockResolvedValueOnce({ data: noPortDeployment });

				renderComponent();

				await waitFor(() => {
					expect(screen.getByText("Deployment Detail")).toBeInTheDocument();
				});

				// Should show path URL
				expect(
					screen.getByRole("link", {
						name: "https://localhost/runtime/models/deployment-123",
					}),
				).toBeInTheDocument();
				// Port label should not be present since port is not available
				expect(screen.queryByText(/Port \(\d+\)/)).not.toBeInTheDocument();
			});
		});
	});
});
