import React from 'react';
import { render, screen, fireEvent, waitFor, within } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import DownloadStatusModal from '../DownloadStatus';
import ModelFilesContext from '../ModelFilesContext';

// Mock the Toast component
jest.mock('../../common/Toast', () => {
  return function MockToast({ open, message, severity, onClose }) {
    return open ? (
      <div data-testid="toast" data-severity={severity}>
        {message}
        <button onClick={onClose}>Close</button>
      </div>
    ) : null;
  };
});

// Test data factory
const createMockFile = (overrides = {}) => ({
  id: Math.random().toString(),
  name: 'test-file.bin',
  is_downloading: false,
  dl_requested_at: null,
  storage_location: null,
  download_percentage: 0,
  download_elapsed: '00:00:00',
  download_remaining: '00:00:00',
  download_throughput: '0 MB/s',
  ...overrides
});

// Mock context factory
const createMockContext = (overrides = {}) => ({
  statusModelFiles: [],
  clearCompletedDownloads: jest.fn(),
  cancelDownload: jest.fn().mockResolvedValue({ success: true }),
  cancelAllDownloads: jest.fn().mockResolvedValue({ success: true }),
  toast: { open: false, message: '', severity: 'info' },
  closeToast: jest.fn(),
  ...overrides
});

// Render helper with context
const renderWithContext = (contextValue = createMockContext(), props = {}) => {
  const closeModal = jest.fn();
  return {
    ...render(
      <ModelFilesContext.Provider value={contextValue}>
        <DownloadStatusModal closeModal={closeModal} {...props} />
      </ModelFilesContext.Provider>
    ),
    closeModal,
    contextValue
  };
};

describe('DownloadStatus - Rendering', () => {
  test('renders nothing when no downloads', () => {
    renderWithContext();
    expect(screen.queryByText('Download Status')).not.toBeInTheDocument();
  });

  test('renders floating window with downloads', () => {
    const mockFiles = [createMockFile({ is_downloading: true })];
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    expect(screen.getByText('Download Status')).toBeInTheDocument();
  });

  test('shows correct header in collapsed state', () => {
    const mockFiles = [
      createMockFile({ is_downloading: true, download_percentage: 50 })
    ];
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    const header = screen.getByText('Download Status');
    expect(header).toBeInTheDocument();
    
    // Should show progress bar in collapsed state
    const progressBar = screen.getByRole('progressbar');
    expect(progressBar).toBeInTheDocument();
  });

  test('shows progress bar when collapsed', () => {
    const mockFiles = [
      createMockFile({ is_downloading: true, download_percentage: 75 })
    ];
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    // In collapsed state, should show progress
    const progressBar = screen.getByRole('progressbar');
    expect(progressBar).toHaveAttribute('aria-valuenow', '75');
    expect(screen.getByText('75%')).toBeInTheDocument();
  });

  test('shows active/queued chips when collapsed', () => {
    const mockFiles = [
      createMockFile({ is_downloading: true }),
      createMockFile({ dl_requested_at: '2024-01-01' }) // queued
    ];
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    expect(screen.getByText('1 active')).toBeInTheDocument();
    expect(screen.getByText('1 queued')).toBeInTheDocument();
  });

  test('shows both active and completed sections when expanded', async () => {
    const user = userEvent.setup();
    const mockFiles = [
      createMockFile({ id: '1', is_downloading: true })
    ];
    
    const { rerender } = renderWithContext(
      createMockContext({ statusModelFiles: mockFiles })
    );
    
    // Click to expand
    await user.click(screen.getByText('Download Status'));
    
    // Should show active section
    expect(screen.getByText(/Active Downloads \(1\)/)).toBeInTheDocument();
    
    // Simulate a file completing by removing it from status
    rerender(
      <ModelFilesContext.Provider value={createMockContext({ statusModelFiles: [] })}>
        <DownloadStatusModal closeModal={jest.fn()} />
      </ModelFilesContext.Provider>
    );
    
    // Note: In real component, completed files tracking is more complex with useEffect
    // For this test, we're focusing on the rendering aspect
  });
});

describe('DownloadStatus - File State Detection', () => {
  test('correctly identifies queued files', async () => {
    const user = userEvent.setup();
    const queuedFile = createMockFile({
      name: 'queued.bin',
      dl_requested_at: '2024-01-01',
      is_downloading: false,
      download_percentage: 0
    });
    
    renderWithContext(createMockContext({ statusModelFiles: [queuedFile] }));
    
    // Expand to see details
    await user.click(screen.getByText('Download Status'));
    
    expect(screen.getByText('Queued')).toBeInTheDocument();
    expect(screen.getByText('Waiting in queue...')).toBeInTheDocument();
  });

  test('correctly identifies downloading files', async () => {
    const user = userEvent.setup();
    const downloadingFile = createMockFile({
      name: 'downloading.bin',
      is_downloading: true,
      download_percentage: 45
    });
    
    renderWithContext(createMockContext({ statusModelFiles: [downloadingFile] }));
    
    await user.click(screen.getByText('Download Status'));
    
    expect(screen.getByText('Downloading')).toBeInTheDocument();
    expect(screen.getByRole('progressbar')).toHaveAttribute('aria-valuenow', '45');
  });

  test('correctly identifies completed files', async () => {
    const user = userEvent.setup();
    const completedFile = createMockFile({
      name: 'completed.bin',
      is_downloading: false,
      dl_requested_at: null,
      storage_location: '/path/to/file',
      download_percentage: 100
    });
    
    renderWithContext(createMockContext({ statusModelFiles: [completedFile] }));
    
    await user.click(screen.getByText('Download Status'));
    
    // Completed files should not show in active section
    expect(screen.queryByText('completed.bin')).not.toBeInTheDocument();
  });
});

describe('DownloadStatus - User Interactions', () => {
  test('toggles expand/collapse on header click', async () => {
    const user = userEvent.setup();
    const mockFiles = [createMockFile({ is_downloading: true })];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    // Initially collapsed - no active downloads text
    expect(screen.queryByText(/Active Downloads/)).not.toBeInTheDocument();
    
    // Click to expand
    await user.click(screen.getByText('Download Status'));
    expect(screen.getByText(/Active Downloads/)).toBeInTheDocument();
    
    // Click to collapse
    await user.click(screen.getByText('Download Status'));
    expect(screen.queryByText(/Active Downloads/)).not.toBeInTheDocument();
  });

  test('toggles active section independently', async () => {
    const user = userEvent.setup();
    const mockFiles = [createMockFile({ is_downloading: true, name: 'active-file.bin' })];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    // Expand main view
    await user.click(screen.getByText('Download Status'));
    
    // Active section should be expanded by default
    expect(screen.getByText('active-file.bin')).toBeInTheDocument();
    
    // Click active section header to collapse
    await user.click(screen.getByText(/Active Downloads/));
    expect(screen.queryByText('active-file.bin')).not.toBeInTheDocument();
    
    // Header should still be visible
    expect(screen.getByText(/Active Downloads/)).toBeInTheDocument();
  });

  test('opens cancel dialog on cancel click', async () => {
    const user = userEvent.setup();
    const mockFile = createMockFile({ 
      id: 'file-1',
      name: 'test-file.bin',
      is_downloading: true 
    });
    
    renderWithContext(createMockContext({ statusModelFiles: [mockFile] }));
    
    // Expand and find cancel button
    await user.click(screen.getByText('Download Status'));
    
    const cancelButton = screen.getByTitle('Cancel download');
    await user.click(cancelButton);
    
    // Dialog should appear - use getAllByText to handle multiple elements
    const cancelDownloadElements = screen.getAllByText('Cancel Download');
    expect(cancelDownloadElements.length).toBeGreaterThan(0);
    expect(screen.getByText(/Are you sure you want to cancel the download of "test-file.bin"/)).toBeInTheDocument();
  });

  test('calls cancelDownload on confirm', async () => {
    const user = userEvent.setup();
    const mockCancelDownload = jest.fn().mockResolvedValue({ success: true });
    const mockFile = createMockFile({ 
      id: 'file-1',
      name: 'test-file.bin',
      is_downloading: true 
    });
    
    renderWithContext(createMockContext({ 
      statusModelFiles: [mockFile],
      cancelDownload: mockCancelDownload
    }));
    
    // Open dialog
    await user.click(screen.getByText('Download Status'));
    await user.click(screen.getByTitle('Cancel download'));
    
    // Confirm cancellation
    await user.click(screen.getByText('Cancel Download', { selector: 'button' }));
    
    expect(mockCancelDownload).toHaveBeenCalledWith('file-1', 'test-file.bin');
  });

  test('closes dialog on cancel', async () => {
    const user = userEvent.setup();
    const mockFile = createMockFile({ 
      id: 'file-1',
      name: 'test-file.bin',
      is_downloading: true 
    });
    
    renderWithContext(createMockContext({ statusModelFiles: [mockFile] }));
    
    // Open dialog
    await user.click(screen.getByText('Download Status'));
    await user.click(screen.getByTitle('Cancel download'));
    
    // Verify dialog is open
    expect(screen.getByRole('dialog')).toBeInTheDocument();
    
    // Cancel (keep downloading)
    await user.click(screen.getByText('Keep Downloading'));
    
    // Wait for dialog to close
    await waitFor(() => {
      expect(screen.queryByRole('dialog')).not.toBeInTheDocument();
    });
  });

  test('disables cancel button during operation', async () => {
    const user = userEvent.setup();
    const mockCancelDownload = jest.fn(() => new Promise(resolve => setTimeout(resolve, 100)));
    const mockFile = createMockFile({ 
      id: 'file-1',
      name: 'test-file.bin',
      is_downloading: true 
    });
    
    renderWithContext(createMockContext({ 
      statusModelFiles: [mockFile],
      cancelDownload: mockCancelDownload
    }));
    
    // Open dialog and confirm
    await user.click(screen.getByText('Download Status'));
    const cancelButton = screen.getByTitle('Cancel download');
    await user.click(cancelButton);
    await user.click(screen.getByText('Cancel Download', { selector: 'button' }));
    
    // Button should show loading state
    await waitFor(() => {
      const progressIndicator = screen.getByRole('progressbar');
      expect(progressIndicator).toBeInTheDocument();
    });
  });

  test('clears completed files', async () => {
    const user = userEvent.setup();
    const mockClearCompleted = jest.fn();
    
    // Start with an active file
    const { rerender } = renderWithContext(createMockContext({ 
      statusModelFiles: [createMockFile({ id: '1', is_downloading: true })],
      clearCompletedDownloads: mockClearCompleted
    }));
    
    // Simulate file completion by removing from status
    rerender(
      <ModelFilesContext.Provider value={createMockContext({ 
        statusModelFiles: [],
        clearCompletedDownloads: mockClearCompleted
      })}>
        <DownloadStatusModal closeModal={jest.fn()} />
      </ModelFilesContext.Provider>
    );
    
    // Note: Testing the clear functionality is complex due to the component's
    // internal state management. In a real scenario, we might need to refactor
    // the component to make this more testable.
  });
});

describe('DownloadStatus - Progress Calculations', () => {
  test('calculates total progress correctly', () => {
    const mockFiles = [
      createMockFile({ is_downloading: true, download_percentage: 50 }),
      createMockFile({ is_downloading: true, download_percentage: 75 }),
      createMockFile({ dl_requested_at: '2024-01-01' }) // queued = 0%
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    // Total progress = (50 + 75 + 0) / 3 = 41.67, rounded to 42
    expect(screen.getByText('42%')).toBeInTheDocument();
  });

  test('counts active files correctly', () => {
    const mockFiles = [
      createMockFile({ is_downloading: true }),
      createMockFile({ is_downloading: true }),
      createMockFile({ dl_requested_at: '2024-01-01' }) // queued
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    expect(screen.getByText('2 active')).toBeInTheDocument();
  });

  test('counts queued files correctly', () => {
    const mockFiles = [
      createMockFile({ dl_requested_at: '2024-01-01' }),
      createMockFile({ dl_requested_at: '2024-01-02' }),
      createMockFile({ is_downloading: true }) // active
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    expect(screen.getByText('2 queued')).toBeInTheDocument();
  });

  test('handles empty state correctly', () => {
    renderWithContext(createMockContext({ statusModelFiles: [] }));
    
    // Should not render anything
    expect(screen.queryByText('Download Status')).not.toBeInTheDocument();
  });
});

describe('DownloadStatus - Cancel All', () => {
  test('enables button when active downloads exist', async () => {
    const user = userEvent.setup();
    const mockFiles = [
      createMockFile({ is_downloading: true }),
      createMockFile({ dl_requested_at: '2024-01-01' })
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    await user.click(screen.getByText('Download Status'));
    
    const cancelAllButton = screen.getByText('Cancel All');
    expect(cancelAllButton).not.toBeDisabled();
  });

  test('disables button when no active downloads', async () => {
    const user = userEvent.setup();
    const mockFiles = [
      createMockFile({ 
        is_downloading: false,
        dl_requested_at: null,
        storage_location: '/path',
        download_percentage: 100
      })
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    await user.click(screen.getByText('Download Status'));
    
    // With only completed files, button should be disabled
    const cancelAllButton = screen.queryByText('Cancel All');
    expect(cancelAllButton).not.toBeInTheDocument(); // No active section
  });

  test('shows correct count in confirmation', async () => {
    const user = userEvent.setup();
    const mockFiles = [
      createMockFile({ is_downloading: true }),
      createMockFile({ is_downloading: true }),
      createMockFile({ dl_requested_at: '2024-01-01' })
    ];
    
    renderWithContext(createMockContext({ statusModelFiles: mockFiles }));
    
    await user.click(screen.getByText('Download Status'));
    await user.click(screen.getByText('Cancel All'));
    
    expect(screen.getByText(/Are you sure you want to cancel all active downloads \(3 files\)/)).toBeInTheDocument();
  });

  test('calls cancelAllDownloads on confirm', async () => {
    const user = userEvent.setup();
    const mockCancelAll = jest.fn().mockResolvedValue({ success: true });
    const mockFiles = [
      createMockFile({ 
        is_downloading: true,
        dl_requested_at: '2024-01-01',
        storage_location: null
      })
    ];
    
    renderWithContext(createMockContext({ 
      statusModelFiles: mockFiles,
      cancelAllDownloads: mockCancelAll
    }));
    
    await user.click(screen.getByText('Download Status'));
    
    // Find and click the Cancel All button
    const cancelAllButton = screen.getByRole('button', { name: /Cancel All/i });
    await user.click(cancelAllButton);
    
    // Wait for dialog to appear with the confirm text
    await waitFor(() => {
      expect(screen.getByText(/Are you sure you want to cancel all active downloads/)).toBeInTheDocument();
    });
    
    // Find the confirm button in the dialog - it has specific text
    const confirmButton = screen.getByRole('button', { name: 'Cancel All Downloads' });
    await user.click(confirmButton);
    
    expect(mockCancelAll).toHaveBeenCalled();
  });
});