import React from 'react';
import { render, screen, waitFor, fireEvent, act } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { ThemeProvider } from '@mui/material/styles';
import { BrowserRouter } from 'react-router-dom';
import AddModelModal from '../AddModelModal';
import ModelFilesContext from '../ModelFilesContext';
import axios from 'axios';
import theme from '../../../Theme';

// Mock axios
jest.mock('axios');

// Mock the BASE_URL
jest.mock('../../../const', () => ({
  BASE_URL: 'http://localhost:8000'
}));

describe('AddModelModal', () => {
  const mockOnClose = jest.fn();
  const mockContextValue = {
    updateStatusModelFiles: jest.fn(),
    refreshModelList: jest.fn(),
    startPolling: jest.fn()
  };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  const renderModal = (props = {}) => {
    return render(
      <ThemeProvider theme={theme}>
        <BrowserRouter>
          <ModelFilesContext.Provider value={mockContextValue}>
            <AddModelModal 
              open={true}
              onClose={mockOnClose}
              {...props}
            />
          </ModelFilesContext.Provider>
        </BrowserRouter>
      </ThemeProvider>
    );
  };

  it('renders button to add external inference endpoint', () => {
    renderModal();
    expect(screen.getByText('Add external inference endpoint')).toBeInTheDocument();
  });

  describe('Search functionality', () => {
    it('should show "Type to search for models..." message when typing without search results', async () => {
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      
      await userEvent.type(searchInput, 'test model');
      
      // Should show the typing message, not "No models found"
      expect(screen.getByText('Type to search for models...')).toBeInTheDocument();
      expect(screen.queryByText(/No models found/)).not.toBeInTheDocument();
    });

    it('should perform live search with debouncing', async () => {
      jest.useFakeTimers();
      
      const mockSearchResults = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/model1',
              version: 'v1.0'
            }
          }
        ]
      };
      
      axios.post.mockResolvedValueOnce({ data: mockSearchResults });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      
      // Type in the search box
      fireEvent.change(searchInput, { target: { value: 'test' } });
      
      // Verify no immediate API call (debouncing)
      expect(axios.post).not.toHaveBeenCalled();
      
      // Fast-forward timers to trigger debounced search
      act(() => {
        jest.advanceTimersByTime(800);
      });
      
      // Now the API should have been called
      await waitFor(() => {
        expect(axios.post).toHaveBeenCalledWith(
          'http://localhost:8000/models/search/',
          {
            query: 'test',
            exact: false,
            hubs_to_search: ['*']
          }
        );
      });
      
      jest.useRealTimers();
    });

    it('should show "No models found" only after search returns empty results', async () => {
      jest.useFakeTimers();
      
      axios.post.mockResolvedValueOnce({ data: { results: [] } });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      
      fireEvent.change(searchInput, { target: { value: 'nonexistent' } });
      
      // Initially should show typing message
      expect(screen.getByText('Type to search for models...')).toBeInTheDocument();
      
      // Trigger debounced search
      act(() => {
        jest.advanceTimersByTime(800);
      });
      
      // After search completes with no results
      await waitFor(() => {
        expect(screen.getByText('No models found for "nonexistent"')).toBeInTheDocument();
      });
      
      jest.useRealTimers();
    });

    it('should clear results when search input is cleared', async () => {
      jest.useFakeTimers();
      
      const mockSearchResults = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/model1',
              version: 'v1.0'
            }
          }
        ]
      };
      
      axios.post.mockResolvedValueOnce({ data: mockSearchResults });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      
      // Type and search
      fireEvent.change(searchInput, { target: { value: 'test' } });
      jest.advanceTimersByTime(800);
      
      await waitFor(() => {
        expect(screen.getByText('test/model1')).toBeInTheDocument();
      });
      
      // Clear the input
      fireEvent.change(searchInput, { target: { value: '' } });
      
      // Results should be cleared immediately
      expect(screen.queryByText('test/model1')).not.toBeInTheDocument();
      
      jest.useRealTimers();
    });

    it('should perform immediate search when Search button is clicked', async () => {
      const mockSearchResults = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/model1',
              version: 'v1.0'
            }
          }
        ]
      };
      
      axios.post.mockResolvedValueOnce({ data: mockSearchResults });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      const searchButton = screen.getByRole('button', { name: 'Search' });
      
      fireEvent.change(searchInput, { target: { value: 'test' } });
      
      // Click search button immediately
      fireEvent.click(searchButton);
      
      // API should be called immediately without waiting for debounce
      await waitFor(() => {
        expect(axios.post).toHaveBeenCalledWith(
          'http://localhost:8000/models/search/',
          {
            query: 'test',
            exact: false,
            hubs_to_search: ['*']
          }
        );
      });
    });

    it('should re-search when exact checkbox is toggled', async () => {
      jest.useFakeTimers();
      
      const mockSearchResults1 = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/model1',
              version: 'v1.0'
            }
          }
        ]
      };
      
      const mockSearchResults2 = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/exact-model',
              version: 'v2.0'
            }
          }
        ]
      };
      
      axios.post
        .mockResolvedValueOnce({ data: mockSearchResults1 })
        .mockResolvedValueOnce({ data: mockSearchResults2 });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      const exactCheckbox = screen.getByLabelText('Exact');
      
      // Initial search
      fireEvent.change(searchInput, { target: { value: 'test' } });
      act(() => {
        jest.advanceTimersByTime(800);
      });
      
      await waitFor(() => {
        expect(axios.post).toHaveBeenCalledTimes(1);
        expect(axios.post).toHaveBeenCalledWith(
          'http://localhost:8000/models/search/',
          {
            query: 'test',
            exact: false,
            hubs_to_search: ['*']
          }
        );
      });
      
      // Wait for results to appear (confirms hasSearched is true)
      await waitFor(() => {
        expect(screen.getByText('test/model1')).toBeInTheDocument();
      });
      
      // Toggle exact checkbox
      fireEvent.click(exactCheckbox);
      
      // Should trigger new search with exact = true
      await waitFor(() => {
        expect(axios.post).toHaveBeenCalledTimes(2);
        expect(axios.post).toHaveBeenLastCalledWith(
          'http://localhost:8000/models/search/',
          {
            query: 'test',
            exact: true,
            hubs_to_search: ['*']
          }
        );
      });
      
      jest.useRealTimers();
    });
  });

  describe('Modal behavior', () => {
    it('should clear all state when modal is closed', async () => {
      jest.useFakeTimers();
      
      const mockSearchResults = {
        results: [
          {
            model: {
              hub: 'huggingface',
              repo_modelId: 'test/model1',
              version: 'v1.0'
            }
          }
        ]
      };
      
      axios.post.mockResolvedValueOnce({ data: mockSearchResults });
      
      renderModal();
      const searchInput = screen.getByPlaceholderText('Search for models...');
      
      // Perform a search
      fireEvent.change(searchInput, { target: { value: 'test' } });
      act(() => {
        jest.advanceTimersByTime(800);
      });
      
      await waitFor(() => {
        expect(screen.getByText('test/model1')).toBeInTheDocument();
      });
      
      // Close modal
      const cancelButton = screen.getByRole('button', { name: 'Cancel' });
      fireEvent.click(cancelButton);
      
      expect(mockOnClose).toHaveBeenCalled();
      
      jest.useRealTimers();
    });
  });
});
