import React, { useState, useRef, useEffect } from 'react';
import PropTypes from 'prop-types';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Button,
  Box,
  Typography,
  CircularProgress,
  IconButton,
  Alert,
  Paper,
  Divider,
  Collapse,
  Link,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import CloseIcon from '@mui/icons-material/Close';
import SendIcon from '@mui/icons-material/Send';
import RefreshIcon from '@mui/icons-material/Refresh';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import ExpandLessIcon from '@mui/icons-material/ExpandLess';
import axios from 'axios';
import { buildModelRuntimeInfo } from '../../utils/routing';
import { getSessionAuthToken } from '../../utils/sessionToken';
import { useRoutingConfig } from '../../context/RoutingConfigContext';

const MessageContainer = styled(Paper, {
  shouldForwardProp: (prop) => prop !== 'isUser',
})(({ theme, isUser }) => ({
  padding: theme.spacing(1.5, 2),
  marginBottom: theme.spacing(1),
  maxWidth: '80%',
  alignSelf: isUser ? 'flex-end' : 'flex-start',
  backgroundColor: isUser 
    ? theme.palette.mode === 'dark' 
      ? 'rgba(144, 202, 249, 0.08)' // Very subtle blue tint for dark mode
      : 'rgba(25, 118, 210, 0.04)'   // Very subtle blue tint for light mode
    : 'transparent',
  border: `1px solid ${isUser ? theme.palette.primary.main : theme.palette.grey[600]}`,
  color: theme.palette.text.primary,
  borderRadius: theme.spacing(2),
  wordWrap: 'break-word',
  boxShadow: 'none',
}));

const ConversationContainer = styled(Box)(({ theme }) => ({
  height: '400px',
  overflowY: 'auto',
  padding: theme.spacing(2),
  display: 'flex',
  flexDirection: 'column',
  backgroundColor: theme.palette.background.default,
  borderRadius: theme.spacing(1),
  border: `1px solid ${theme.palette.divider}`,
}));

const ThinkingContent = styled(Box)(({ theme }) => ({
  marginTop: theme.spacing(1),
  padding: theme.spacing(1.5),
  backgroundColor: theme.palette.action.hover,
  borderRadius: theme.shape.borderRadius,
  fontStyle: 'italic',
  fontSize: '0.9em',
  color: theme.palette.text.secondary,
  borderLeft: `3px solid ${theme.palette.info.main}`,
}));

// Helper function to parse thinking content from response
const parseThinkingContent = (content) => {
  // Check for various thinking tag patterns
  const thinkingPatterns = [
    /<think>([\s\S]*?)<\/think>/i,
    /<thinking>([\s\S]*?)<\/thinking>/i,
    /<thought>([\s\S]*?)<\/thought>/i,
  ];
  
  for (const pattern of thinkingPatterns) {
    const match = content.match(pattern);
    if (match) {
      const thinkingContent = match[1].trim();
      const responseContent = content.replace(pattern, '').trim();
      return {
        hasThinking: true,
        thinking: thinkingContent,
        response: responseContent || '(No response after thinking)',
      };
    }
  }
  
  return {
    hasThinking: false,
    thinking: '',
    response: content,
  };
};

const TestConversationModal = ({ open, onClose, deployment }) => {
  const [messages, setMessages] = useState([]);
  const [inputValue, setInputValue] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [expandedThinking, setExpandedThinking] = useState({});
  const [expandedRaw, setExpandedRaw] = useState({});
  const conversationEndRef = useRef(null);

  // Get system routing mode for endpoint selection
  const { routingMode } = useRoutingConfig();

  // Auto-scroll to bottom when new messages are added
  useEffect(() => {
    conversationEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [messages]);

  // Reset state when modal opens
  useEffect(() => {
    if (open) {
      setMessages([]);
      setInputValue('');
      setError(null);
      setExpandedThinking({});
      setExpandedRaw({});
    }
  }, [open]);

  const sendMessage = async () => {
    if (!inputValue.trim() || loading) return;

    const userMessage = { role: 'user', content: inputValue.trim() };
    const updatedMessages = [...messages, userMessage];
    const cleanedMessages = updatedMessages.map(({ role, content }) => ({ role, content }));
    setMessages(updatedMessages);
    setInputValue('');
    setLoading(true);
    setError(null);

    try {
      const runtimeInfo = buildModelRuntimeInfo(deployment, routingMode);
      const endpoint = runtimeInfo.chatUrl;

      if (!endpoint) {
        throw new Error('Model endpoint is not available yet.');
      }

      // Use the sanitized model name from the deployment (m_name is already normalized by the backend)
      const modelName = deployment.m_name;
      
      // Build headers with Authorization token for cross-origin port-based deployments
      const authToken = getSessionAuthToken();
      const headers = {
        'Content-Type': 'application/json',
      };
      if (authToken) {
        headers['Authorization'] = `Bearer ${authToken}`;
      }

      const response = await axios.post(
        endpoint,
        {
          model: modelName,
          messages: cleanedMessages,
          max_tokens: 2048,
          temperature: 0.7,
        },
        {
          timeout: 30000, // 30 second timeout
          headers,
          // Allow credentials for CORS (required for cross-origin port-based deployments)
          withCredentials: true,
        }
      );

      if (response.data.choices && response.data.choices.length > 0) {
        const rawContent = response.data.choices[0].message.content;
        const parsed = parseThinkingContent(rawContent);

        const assistantMessage = {
          role: 'assistant',
          content: parsed.response,
          thinking: parsed.thinking,
          hasThinking: parsed.hasThinking,
          rawResponse: response.data, // Store the complete raw response
        };
        setMessages([...updatedMessages, assistantMessage]);
      } else {
        // Invalid response format - still capture the raw response
        const assistantMessage = {
          role: 'assistant',
          content: 'Invalid response format from model',
          hasThinking: false,
          rawResponse: response.data, // Store the complete raw response even for invalid format
        };
        setMessages([...updatedMessages, assistantMessage]);
        setError('Invalid response format: response.data.choices is missing or empty');
      }
    } catch (err) {
      console.error('Error calling model endpoint:', err);
      let errorMessage = 'Failed to get response from model';
      let rawErrorResponse = null;
      
      if (err.response) {
        errorMessage = `Error: ${err.response.status} - ${err.response.statusText}`;
        // Capture the raw response data even on error
        rawErrorResponse = {
          error: true,
          status: err.response.status,
          statusText: err.response.statusText,
          data: err.response.data,
          headers: err.response.headers,
        };
      } else if (err.request) {
        errorMessage = 'No response from model. Please check if the deployment is running.';
        // Capture request info if available
        rawErrorResponse = {
          error: true,
          message: 'No response received',
          request: err.request,
        };
      } else if (err.message) {
        errorMessage = err.message;
        rawErrorResponse = {
          error: true,
          message: err.message,
        };
      }
      
      setError(errorMessage);
      
      // Create an assistant message with error content and raw response
      const assistantMessage = {
        role: 'assistant',
        content: errorMessage,
        hasThinking: false,
        rawResponse: rawErrorResponse || { error: true, message: 'Unknown error occurred' },
      };
      setMessages([...updatedMessages, assistantMessage]);
    } finally {
      setLoading(false);
    }
  };

  const handleKeyPress = (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  const clearConversation = () => {
    setMessages([]);
    setError(null);
    setExpandedThinking({});
    setExpandedRaw({});
  };

  const toggleThinking = (messageIndex) => {
    setExpandedThinking((prev) => ({
      ...prev,
      [messageIndex]: !prev[messageIndex],
    }));
  };

  const toggleRawResponse = (messageIndex) => {
    setExpandedRaw((prev) => ({
      ...prev,
      [messageIndex]: !prev[messageIndex],
    }));
  };

  const runtimeInfo = buildModelRuntimeInfo(deployment, routingMode);
  const endpointUrl = runtimeInfo.chatUrl || 'Unavailable';

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        <Box display="flex" alignItems="center" justifyContent="space-between">
          <Typography variant="h6">
            Test Conversation - {deployment?.m_name}
          </Typography>
          <IconButton onClick={onClose} size="small">
            <CloseIcon />
          </IconButton>
        </Box>
        <Typography variant="caption" color="text.secondary">
          Endpoint: {endpointUrl} • Max tokens: 2048
        </Typography>
      </DialogTitle>
      <Divider />
      <DialogContent>
        <ConversationContainer>
          {messages.length === 0 && (
            <Typography
              variant="body2"
              color="text.secondary"
              align="center"
              sx={{ mt: 8 }}
            >
              Start a conversation with the model
            </Typography>
          )}
          {messages.map((message, index) => (
            <MessageContainer
              key={index}
              isUser={message.role === 'user'}
              elevation={1}
            >
              <Box>
                <Typography variant="body2" sx={{ whiteSpace: 'pre-wrap' }}>
                  {message.content}
                </Typography>
                {message.hasThinking && (
                  <>
                    <Link
                      component="button"
                      variant="caption"
                      onClick={() => toggleThinking(index)}
                      sx={{
                        mt: 1,
                        display: 'flex',
                        alignItems: 'center',
                        gap: 0.5,
                        textDecoration: 'none',
                        '&:hover': {
                          textDecoration: 'underline',
                        },
                      }}
                    >
                      {expandedThinking[index] ? <ExpandLessIcon fontSize="small" /> : <ExpandMoreIcon fontSize="small" />}
                      {expandedThinking[index] ? 'Hide' : 'Show'} thinking process
                    </Link>
                    <Collapse in={expandedThinking[index] || false}>
                      <ThinkingContent>
                        <Typography variant="caption" sx={{ whiteSpace: 'pre-wrap' }}>
                          {message.thinking}
                        </Typography>
                      </ThinkingContent>
                    </Collapse>
                  </>
                )}
                {message.rawResponse && message.role === 'assistant' && (
                  <>
                    <Link
                      component="button"
                      variant="caption"
                      onClick={() => toggleRawResponse(index)}
                      sx={{
                        mt: 1,
                        display: 'flex',
                        alignItems: 'center',
                        gap: 0.5,
                        textDecoration: 'none',
                        '&:hover': {
                          textDecoration: 'underline',
                        },
                      }}
                    >
                      {expandedRaw[index] ? <ExpandLessIcon fontSize="small" /> : <ExpandMoreIcon fontSize="small" />}
                      {expandedRaw[index] ? 'Hide' : 'View'} raw response
                    </Link>
                    <Collapse in={expandedRaw[index] || false}>
                      <ThinkingContent
                        sx={{
                          fontStyle: 'normal',
                          fontFamily: 'monospace',
                          fontSize: '0.75em',
                          maxHeight: '300px',
                          overflowY: 'auto',
                        }}
                      >
                        <pre style={{ margin: 0, whiteSpace: 'pre-wrap', wordBreak: 'break-word' }}>
                          {JSON.stringify(message.rawResponse, null, 2)}
                        </pre>
                      </ThinkingContent>
                    </Collapse>
                  </>
                )}
              </Box>
            </MessageContainer>
          ))}
          {loading && (
            <Box display="flex" alignItems="center" gap={1} sx={{ ml: 2 }}>
              <CircularProgress size={20} />
              <Typography variant="body2" color="text.secondary">
                Model is thinking...
              </Typography>
            </Box>
          )}
          <div ref={conversationEndRef} />
        </ConversationContainer>
        {error && (
          <Alert severity="error" sx={{ mt: 2 }}>
            {error}
          </Alert>
        )}
        <Box sx={{ mt: 2, display: 'flex', gap: 1 }}>
          <TextField
            fullWidth
            multiline
            maxRows={4}
            placeholder="Type your message..."
            value={inputValue}
            onChange={(e) => setInputValue(e.target.value)}
            onKeyPress={handleKeyPress}
            disabled={loading}
            variant="outlined"
            size="small"
          />
          <Button
            variant="contained"
            onClick={sendMessage}
            disabled={loading || !inputValue.trim()}
            endIcon={<SendIcon />}
          >
            Send
          </Button>
        </Box>
      </DialogContent>
      <DialogActions>
        <Button
          onClick={clearConversation}
          startIcon={<RefreshIcon />}
          disabled={messages.length === 0}
        >
          Clear
        </Button>
        <Button onClick={onClose}>Close</Button>
      </DialogActions>
    </Dialog>
  );
};

TestConversationModal.propTypes = {
  open: PropTypes.bool.isRequired,
  onClose: PropTypes.func.isRequired,
  deployment: PropTypes.shape({
    lb_port: PropTypes.number,
    engine_name: PropTypes.string,
    m_name: PropTypes.string,
  }).isRequired,
};

export default TestConversationModal;
