import React, { useState, useEffect, useRef } from 'react';
import { styled } from '@mui/material/styles';
import {
  Box,
  Typography,
  Card,
  CardContent,
  CardActions,
  Button,
  Grid,
  Chip,
  IconButton,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  LinearProgress,
  Alert,
  Tooltip,
  Rating,
  Divider,
  CircularProgress,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Switch,
  FormControlLabel,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
} from '@mui/material';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { buildModelRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';
import { useTour } from '../../contexts/TourContext';
import { useCluster } from '../../contexts/ClusterContext';
// import ModelFilesContext from './ModelFilesContext'; // Not currently used
import DownloadIcon from '@mui/icons-material/Download';
import CloudDownloadIcon from '@mui/icons-material/CloudDownload';
import PlayArrowIcon from '@mui/icons-material/PlayArrow';
import HelpOutlineIcon from '@mui/icons-material/HelpOutline';
import MemoryIcon from '@mui/icons-material/Memory';
import SpeedIcon from '@mui/icons-material/Speed';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import CancelIcon from '@mui/icons-material/Cancel';
import EditIcon from '@mui/icons-material/Edit';
import FilterListIcon from '@mui/icons-material/FilterList';
import WarningIcon from '@mui/icons-material/Warning';

// Development mode check
const isDevelopment = process.env.NODE_ENV === 'development';

const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(3),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const ModelCard = styled(Card)(({ theme }) => ({
  height: '100%',
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: theme.spacing(2),
  transition: 'all 0.3s ease',
  display: 'flex',
  flexDirection: 'column',
  '&:hover': {
    transform: 'translateY(-2px)',
    boxShadow: '0 8px 32px rgba(0, 192, 127, 0.15)',
    borderColor: theme.palette.primary.main,
  },
}));

const UseCaseChip = styled(Chip)(({ usecase }) => {
  const colors = {
    'General Use': { bg: 'rgba(33, 150, 243, 0.15)', border: '#2196f3' },
    Reasoning: { bg: 'rgba(156, 39, 176, 0.15)', border: '#9c27b0' },
    'Vision/MultiModal': { bg: 'rgba(255, 152, 0, 0.15)', border: '#ff9800' },
    Roleplaying: { bg: 'rgba(233, 30, 99, 0.15)', border: '#e91e63' },
  };
  const color = colors[usecase] || colors['General Use'];

  return {
    backgroundColor: color.bg,
    border: `1px solid ${color.border}`,
    color: color.border,
    fontWeight: 600,
  };
});

const PlatformChip = styled(Chip)(() => ({
  fontSize: '0.75rem',
  height: 24,
  '& .MuiChip-label': {
    padding: '0 8px',
  },
}));

const FilterContainer = styled(Box)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: theme.spacing(2),
  padding: theme.spacing(2),
  marginBottom: theme.spacing(3),
  display: 'flex',
  flexDirection: 'column',
  gap: theme.spacing(2),
}));

const FilterRow = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
  flexWrap: 'wrap',
}));

const VramButton = styled(Button)(() => ({
  minWidth: '120px',
  justifyContent: 'space-between',
  textTransform: 'none',
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.1)',
  },
}));

const NoviceModelView = () => {
  const autoDeployTriggered = useRef(new Set());
  // Utility functions for localStorage with 24h expiry
  const VARIANT_STORAGE_KEY = 'kz_selected_variants';
  const STORAGE_EXPIRY_HOURS = 24;

  const saveVariantSelection = (guideId, variant) => {
    try {
      const storage = JSON.parse(localStorage.getItem(VARIANT_STORAGE_KEY) || '{}');
      const expiryTime = Date.now() + (STORAGE_EXPIRY_HOURS * 60 * 60 * 1000);
      storage[guideId] = {
        variant,
        expires: expiryTime
      };
      localStorage.setItem(VARIANT_STORAGE_KEY, JSON.stringify(storage));
    } catch (error) {
      console.warn('Failed to save variant selection:', error);
    }
  };

  const loadVariantSelection = (guideId) => {
    try {
      const storage = JSON.parse(localStorage.getItem(VARIANT_STORAGE_KEY) || '{}');
      const stored = storage[guideId];
      if (stored && stored.expires > Date.now()) {
        return stored.variant;
      }
      // Clean up expired entry
      if (stored) {
        delete storage[guideId];
        localStorage.setItem(VARIANT_STORAGE_KEY, JSON.stringify(storage));
      }
    } catch (error) {
      console.warn('Failed to load variant selection:', error);
    }
    return null;
  };

  const { startTour } = useTour();
  const { clusterCapabilities, maxVram, memoryWarning, loading: clusterLoading, twoNodeInfo } = useCluster();
  const { routingMode } = useRoutingConfig();
  const [modelGuides, setModelGuides] = useState([]);
  const [models, setModels] = useState([]);
  const [loading, setLoading] = useState(true);

  // Use ref to always get current models array in callbacks
  const modelsRef = useRef([]);
  const modelGuidesRef = useRef([]);
  const [selectedModel, setSelectedModel] = useState(null);
  const [deployDialog, setDeployDialog] = useState(false);
  // Legacy state variables removed - replaced by unified modelStates Map
  const [variantDialog, setVariantDialog] = useState(false);
  const [selectedGuideForVariant, setSelectedGuideForVariant] = useState(null);

  // Use global model files context for progress tracking
  // const { } = useContext(ModelFilesContext); // Not currently used

  // Single unified state machine for all model states
  const [modelStates, setModelStates] = useState(new Map());

  // Filter states
  const [filters, setFilters] = useState({
    useCase: 'all',
    sizeCategory: 'all',
    maxVram: null,
    showUnsupported: false,
  });
  const [sortBy, setSortBy] = useState('score');
  const [editingVram, setEditingVram] = useState(false);
  const [vramInputValue, setVramInputValue] = useState('');

  // Force refresh trigger for deployment status checks
  const [deploymentCheckTrigger, setDeploymentCheckTrigger] = useState(0);

  // Track recently clicked deploy to immediately flip other models to download mode
  const [recentlyClickedDeploy, setRecentlyClickedDeploy] = useState(null);
  
  // Context selection state
  const [contextDialog, setContextDialog] = useState(false);
  const [selectedGuideForContext, setSelectedGuideForContext] = useState(null);
  const [selectedContextSizes, setSelectedContextSizes] = useState(new Map()); // Map of guide.base_model_id -> context size

  // State machine types and helpers
  const MODEL_STATES = {
    IDLE: 'idle',
    DOWNLOADING: 'downloading',
    DOWNLOADED: 'downloaded',
    DEPLOYING: 'deploying',
    INITIALIZING: 'initializing',
    RUNNING: 'running',
    STOPPING: 'stopping',
    FAILED: 'failed',
    RESETTING: 'resetting',
  };

  // Create initial model state
  const createModelState = (status = MODEL_STATES.IDLE) => ({
    status,
    progress: 0,
    downloadSpeed: 0,
    timeRemaining: null,
    deploymentInfo: null,
    shouldAutoDeploy: false,
    error: null,
  });

  // State machine reducer for transitions
  const updateModelState = (guideId, updates) => {
    setModelStates(prev => {
      const newStates = new Map(prev);
      const currentState = newStates.get(guideId) || createModelState();
      newStates.set(guideId, { ...currentState, ...updates });
      return newStates;
    });
  };

  // State query helpers
  const getModelState = guideId => {
    return modelStates.get(guideId) || createModelState();
  };

  const isModelDownloading = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.DOWNLOADING;
  };

  const isModelDeploying = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.DEPLOYING;
  };

  const isModelInitializing = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.INITIALIZING;
  };

  const isModelRunning = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.RUNNING;
  };

  const isModelStopping = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.STOPPING;
  };

  const isModelFailed = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.FAILED;
  };

  const isModelResetting = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.RESETTING;
  };

  const isModelDownloaded = guideId => {
    const state = getModelState(guideId);
    return state.status === MODEL_STATES.DOWNLOADED;
  };

  const getModelProgress = guideId => {
    const state = getModelState(guideId);
    return {
      percentage: state.progress,
      isDownloading: state.status === MODEL_STATES.DOWNLOADING,
      speed: state.downloadSpeed,
      timeRemaining: state.timeRemaining,
    };
  };

  const getModelDeploymentInfo = guideId => {
    const state = getModelState(guideId);
    return state.deploymentInfo;
  };

  useEffect(() => {
    fetchModelGuides();
    fetchModels();

    // Add window focus listener to refresh deployment status when user returns to tab
    const handleWindowFocus = () => {
      setDeploymentCheckTrigger(prev => prev + 1);
    };

    window.addEventListener('focus', handleWindowFocus);

    return () => {
      window.removeEventListener('focus', handleWindowFocus);
    };
  }, []);

  // Start tour and recover backend state after model guides are loaded and rendered
  useEffect(() => {
    if (modelGuides.length > 0) {
      // Trigger state recovery first
      recoverBackendState();

      // Wait for next render cycle to ensure DOM is updated, then start tour
      requestAnimationFrame(() => {
        setTimeout(() => {
          startTour('models');
        }, 500);
      });
    }
  }, [modelGuides.length, startTour]);

  // Keep refs in sync with state
  useEffect(() => {
    modelsRef.current = models;
  }, [models]);

  useEffect(() => {
    modelGuidesRef.current = modelGuides;
  }, [modelGuides]);

  // Legacy deployment info useEffect removed - deployment info now handled by state sync

  // Unified state sync function
  const syncAllModelStates = async () => {
    if (!modelGuides.length) {
      if (isDevelopment) console.log('No model guides loaded yet, skipping state sync');
      return;
    }

    try {
      // Get deployments and download status in parallel
      const [deploymentResponse, downloadStatusResponse] = await Promise.all([
        axios.get(`${BASE_URL}/serving/deployments`),
        axios.get(`${BASE_URL}/model_files/download_status/`),
      ]);

      const deployments = deploymentResponse.data || [];
      const downloadFiles = downloadStatusResponse.data || [];

      // Store deployments for other functions
      // setAllDeployments(deployments); // Not needed in unified state machine

      const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
      const currentGuides =
        modelGuidesRef.current.length > 0 ? modelGuidesRef.current : modelGuides;

      // Removed verbose development console debugging

      // Helper to pick the most relevant deployment for a model
      const selectRelevantDeployment = (modelDeployments) => {
        if (!Array.isArray(modelDeployments) || modelDeployments.length === 0) return null;

        // Prefer RUNNING/DEPLOYED (with a valid port if available)
        const runningOrDeployed = modelDeployments
          .filter(d => d && (d.status === 'RUNNING' || d.status === 'DEPLOYED'))
          .sort((a, b) => {
            const aHasEndpoint = buildModelRuntimeInfo(a, routingMode).baseUrl ? 1 : 0;
            const bHasEndpoint = buildModelRuntimeInfo(b, routingMode).baseUrl ? 1 : 0;
            return bHasEndpoint - aHasEndpoint;
          });
        if (runningOrDeployed.length > 0) return runningOrDeployed[0];

        // Next prefer INITIALIZING
        const initializing = modelDeployments.find(d => d && d.status === 'INITIALIZING');
        if (initializing) return initializing;

        // Then prefer FAILED/ERROR to surface issues to the user
        const failed = modelDeployments.find(d => d && (d.status === 'FAILED' || d.status === 'ERROR'));
        if (failed) return failed;

        // Fallback to the first entry
        return modelDeployments[0];
      };

      // Helper to get deployments for a guide, even if the model lookup fails
      const getDeploymentsForGuide = (guide, maybeModel, allDeployments) => {
        if (maybeModel && maybeModel.id) {
          return allDeployments.filter(d => d.m_id === maybeModel.id);
        }
        // Fallback: match by model name prefix saved when novice created the model
        // Normalize names to handle differences between guide names and deployment names
        const normalizeForMatching = (name) => {
          return name
            .toLowerCase()
            .replace(/\s+/g, '-')        // Replace spaces with dashes
            .replace(/[()]/g, '')        // Remove parentheses
            .replace(/[^\w-]/g, '');     // Remove other special characters except dashes
        };
        
        const normalizedGuideName = normalizeForMatching(guide?.name || '');
        const matchedDeployments = allDeployments.filter(d => {
          const normalizedDeploymentName = normalizeForMatching(d?.m_name || '');
          const matches = normalizedDeploymentName.startsWith(normalizedGuideName) ||
                         normalizedGuideName.startsWith(normalizedDeploymentName);
          return matches;
        });
        return matchedDeployments;
      };

      // Choose best (model, deployment) for a guide, preferring selected/recommended variant first
      const selectBestModelAndDeployment = (guide, currentModelsArr, allDeployments) => {
        const preferredVariant = getRelevantVariant(guide);
        if (!preferredVariant) return { model: null, deployment: null };

        const preferredModel = currentModelsArr.find(m => m.repo_modelId === preferredVariant.variant_repo) || null;
        const allVariantRepos = guide.variants.map(v => v.variant_repo);
        const otherModels = currentModelsArr.filter(
          m => allVariantRepos.includes(m.repo_modelId) && (!preferredModel || m.id !== preferredModel.id)
        );
        const candidates = [preferredModel, ...otherModels].filter(Boolean);

        const scoreDeployment = dep => {
          if (!dep) return -1;
          const hasEndpoint = buildModelRuntimeInfo(dep, routingMode).baseUrl ? 1 : 0;
          const hasError = !!(dep.last_error_code || dep.last_error_message);
          switch (dep.status) {
            case 'RUNNING':
            case 'DEPLOYED':
              return 100 + hasEndpoint;
            case 'INITIALIZING':
              return 60;
            case 'FAILED':
            case 'ERROR':
              return 20;
            case 'STOPPED':
              return hasError ? 15 : 5;
            default:
              return 0;
          }
        };

        let best = { model: null, deployment: null, score: -1 };
        candidates.forEach(m => {
          const deps = allDeployments.filter(d => d.m_id === m.id);
          if (deps.length === 0) return;
          const selected = selectRelevantDeployment(deps);
          const score = scoreDeployment(selected);
          if (score > best.score) best = { model: m, deployment: selected, score };
        });

        if (best.score < 0) {
          const deps = getDeploymentsForGuide(guide, preferredModel, allDeployments);
          const selected = selectRelevantDeployment(deps);
          const score = scoreDeployment(selected);
          if (score > best.score) best = { model: preferredModel, deployment: selected, score };
        }

        return { model: best.model, deployment: best.deployment };
      };

      // Update states for each guide
      setModelStates(prevStates => {
        const newStates = new Map(prevStates);

        currentGuides.forEach(guide => {
          const guideId = guide.base_model_id;
          const variant = getRelevantVariant(guide);
          if (!variant) return;

          // Choose best model/deployment with preference for selected variant (keeps novice aligned with dev view)
          const { model } = selectBestModelAndDeployment(guide, currentModels, deployments);
          const currentState = newStates.get(guideId) || createModelState();
          let newStatus = currentState.status;
          let deploymentInfo = null;
          let progress = 0;
          let downloadSpeed = 0;
          let timeRemaining = null;

          // Check deployment status
          {
            const modelDeployments = getDeploymentsForGuide(guide, model, deployments);

            // Select the most relevant deployment without relying on instance-level status,
            // which can vary across engines (llama.cpp, vLLM, MLX).
            const deployment = selectRelevantDeployment(modelDeployments);

            if (deployment) {
              const runtimeInfo = buildModelRuntimeInfo(deployment, routingMode);
              if (deployment.status === 'STOPPED') {
                // Distinguish between a clean stop vs. a failure that resulted in STOPPED
                const hasError = !!(deployment.last_error_code || deployment.last_error_message);
                if (hasError) {
                  // Treat STOPPED with error as a FAILED deployment and surface error details
                  newStatus = MODEL_STATES.FAILED;
                  deploymentInfo = {
                    ...deployment,
                    deployed: true, // mark true so UI renders message block
                    status: 'FAILED',
                    host_name: deployment.instances?.[0]?.host_name || window.location.hostname,
                    runtime_info: runtimeInfo,
                  };
                } else {
                  // Clean stop → files available, stay downloadable
                  newStatus = MODEL_STATES.DOWNLOADED;
                }
              } else if (deployment.status === 'DEPLOYING') {
                // Handle DEPLOYING deployments
                newStatus = MODEL_STATES.DEPLOYING;
              } else if (deployment.status === 'INITIALIZING') {
                // Handle INITIALIZING deployments first, regardless of instance status
                newStatus = MODEL_STATES.INITIALIZING;
              } else if (
                (deployment.status === 'DEPLOYED' || deployment.status === 'RUNNING')
              ) {
                newStatus = MODEL_STATES.RUNNING;

                // Build deployment info
                let host_name = window.location.hostname;
                if (deployment.instances?.[0]?.host_name) {
                  host_name = deployment.instances[0].host_name;
                }

                deploymentInfo = {
                  ...deployment,
                  deployed: true,
                  status: deployment.status,
                  host_name: host_name,
                  runtime_info: runtimeInfo,
                };
              } else if (deployment.status === 'FAILED' || deployment.status === 'ERROR') {
                newStatus = MODEL_STATES.FAILED;
                deploymentInfo = {
                  ...deployment,
                  deployed: true, // mark true so UI renders message block
                  status: 'FAILED',
                  host_name: deployment.instances?.[0]?.host_name || window.location.hostname,
                  runtime_info: runtimeInfo,
                };
              }
            }
          }

          // Check download progress (only if not already running, initializing, or failed)
          if (newStatus !== MODEL_STATES.RUNNING && newStatus !== MODEL_STATES.INITIALIZING && newStatus !== MODEL_STATES.FAILED && model) {
            const modelFiles = downloadFiles.filter(file => file.m_id === model.id);

            if (modelFiles.length > 0) {
              // Calculate progress
              const incompleteFiles = modelFiles.filter(
                file => (file.download_percentage || 0) < 100
              );
              if (incompleteFiles.length === 0) {
                progress = 100;
              } else {
                progress =
                  incompleteFiles.reduce((sum, file) => sum + (file.download_percentage || 0), 0) /
                  incompleteFiles.length;
              }

              // Check if actively downloading
              const isDownloading = modelFiles.some(
                file =>
                  file.is_downloading === true ||
                  (file.dl_requested_at !== null && file.dl_requested_at !== undefined)
              );

              if (isDownloading) {
                newStatus = MODEL_STATES.DOWNLOADING;

                // Calculate speed and time
                const activeDownloads = modelFiles.filter(
                  file => file.is_downloading && file.download_throughput
                );
                downloadSpeed = activeDownloads.reduce((sum, file) => {
                  const speed = parseFloat(file.download_throughput.replace(/[^0-9.]/g, '')) || 0;
                  return sum + speed;
                }, 0);

                // Find max time remaining
                if (activeDownloads.length > 0) {
                  const remainingTimes = activeDownloads
                    .filter(file => file.download_remaining)
                    .map(file => {
                      const parts = file.download_remaining.split(':').map(p => parseInt(p) || 0);
                      return parts.length === 2 ? parts[0] * 60 + parts[1] : 0;
                    })
                    .filter(seconds => seconds > 0);

                  if (remainingTimes.length > 0) {
                    const maxSeconds = Math.max(...remainingTimes);
                    const minutes = Math.floor(maxSeconds / 60);
                    const seconds = maxSeconds % 60;
                    timeRemaining = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                  }
                }
              } else if (
                progress === 100 &&
                currentState.shouldAutoDeploy &&
                newStatus !== MODEL_STATES.DEPLOYING &&
                !autoDeployTriggered.current.has(model.id)
              ) {
                // Download complete and should auto-deploy
                newStatus = MODEL_STATES.DEPLOYING;

                // Trigger deploy_after_download
                (async () => {
                  try {
                    await axios.post(
                      `${BASE_URL}/models/deploy_after_download/${model.id}`
                    );
                  } catch (error) {
                    console.error('Auto-deploy failed:', error);
                  }
                })();
                autoDeployTriggered.current.add(model.id);
              } else if (progress === 100) {
                newStatus = MODEL_STATES.DOWNLOADED;
              }
            } else if (model?.m_files?.some(f => 
              // Either: file is marked for download and has storage
              (f.download && f.storage_location) ||
              // Or: file has storage and wasn't explicitly marked as skip (for backward compat)
              (!Object.prototype.hasOwnProperty.call(f, 'download') && f.storage_location)
            )) {
              // Required files exist but no download progress
              if (currentState.shouldAutoDeploy && newStatus !== MODEL_STATES.DEPLOYING && !autoDeployTriggered.current.has(model.id)) {
                // Files are already present; honor auto-deploy intent immediately
                newStatus = MODEL_STATES.DEPLOYING;
                (async () => {
                  try {
                    await axios.post(
                      `${BASE_URL}/models/deploy_after_download/${model.id}`
                    );
                  } catch (error) {
                    console.error('Auto-deploy (no-download path) failed:', error);
                  }
                })();
                autoDeployTriggered.current.add(model.id);
              } else {
                newStatus = MODEL_STATES.DOWNLOADED;
              }
            } else if (currentState.status === MODEL_STATES.DOWNLOADING) {
              // Was downloading but no longer in download_status - download completed successfully
              if (currentState.shouldAutoDeploy && !autoDeployTriggered.current.has(model.id)) {
                // Download completed with auto-deploy flag - trigger deployment immediately
                newStatus = MODEL_STATES.DEPLOYING;

                // Trigger deploy_after_download
                (async () => {
                  try {
                    await axios.post(
                      `${BASE_URL}/models/deploy_after_download/${model.id}`
                    );
                  } catch (error) {
                    console.error('Auto-deploy API call failed:', error);
                  }
                })();
                autoDeployTriggered.current.add(model.id);
              } else {
                // Download completed but no auto-deploy requested
                newStatus = MODEL_STATES.DOWNLOADED;
              }
            }
          }

          // Don't downgrade from user-initiated states
          if (
            (currentState.status === MODEL_STATES.STOPPING &&
              newStatus !== MODEL_STATES.DOWNLOADED) ||
            (currentState.status === MODEL_STATES.RESETTING && newStatus !== MODEL_STATES.IDLE)
          ) {
            newStatus = currentState.status;
          }

          // Update state if changed
          if (
            newStatus !== currentState.status ||
            progress !== currentState.progress ||
            downloadSpeed !== currentState.downloadSpeed ||
            timeRemaining !== currentState.timeRemaining ||
            JSON.stringify(deploymentInfo) !== JSON.stringify(currentState.deploymentInfo)
          ) {
            newStates.set(guideId, {
              ...currentState,
              status: newStatus,
              progress,
              downloadSpeed,
              timeRemaining,
              deploymentInfo,
            });
          }
        });

        return newStates;
      });
    } catch (error) {
      console.error('Error syncing model states:', error);
    }
  };

  // Periodic state sync
  useEffect(() => {
    // Check if any models are in active states that need fast polling
    const hasActiveOperations = Array.from(modelStates.values()).some(
      state =>
        state.status === MODEL_STATES.DOWNLOADING ||
        state.status === MODEL_STATES.DEPLOYING ||
        state.status === MODEL_STATES.INITIALIZING ||
        state.status === MODEL_STATES.STOPPING ||
        state.status === MODEL_STATES.RESETTING
    );

    const pollInterval = hasActiveOperations ? 5000 : 15000;

    // Removed verbose polling interval console debugging

    const syncInterval = setInterval(syncAllModelStates, pollInterval);

    // Initial sync after a short delay
    const initialSyncTimeout = setTimeout(syncAllModelStates, 1000);

    return () => {
      clearInterval(syncInterval);
      clearTimeout(initialSyncTimeout);
    };
  }, [modelGuides.length, modelStates.size, deploymentCheckTrigger]);

  // Set initial VRAM filter based on cluster capabilities
  useEffect(() => {
    if (maxVram !== null && filters.maxVram === null) {
      setFilters(prev => ({ ...prev, maxVram }));
      setVramInputValue(maxVram.toString());
    }
  }, [maxVram, filters.maxVram]);

  const fetchModelGuides = async () => {
    try {
      const response = await axios.get(`${BASE_URL}/guide/`);
      const guides = response.data;
      
      // Restore saved variant selections from localStorage
      guides.forEach(guide => {
        const savedVariant = loadVariantSelection(guide.base_model_id);
        if (savedVariant) {
          guide.selectedVariant = savedVariant;
        }
      });
      
      setModelGuides(guides);
      modelGuidesRef.current = guides; // Keep ref in sync
    } catch (error) {
      console.error('Failed to fetch model guides:', error);
    }
  };

  const fetchModels = async () => {
    try {
      const response = await axios.get(`${BASE_URL}/models/`);
      setModels(response.data);
      modelsRef.current = response.data; // Keep ref in sync
      setLoading(false);
    } catch (error) {
      console.error('Failed to fetch models:', error);
      setLoading(false);
    }
  };

  // Legacy updateDeploymentInfo function removed - deployment info now handled by state sync

  const recoverBackendState = async () => {
    try {

      const response = await axios.get(`${BASE_URL}/models/pending_deployments`);
      const { pending_deployments, downloading_models } = response.data;

      // Removed verbose state recovery logging

      // Convert repo names to guide base_model_ids for state restoration
      const currentGuides =
        modelGuidesRef.current.length > 0 ? modelGuidesRef.current : modelGuides;

      // Create mapping from repo names to guide base_model_ids
      const repoToGuideId = {};
      currentGuides.forEach(guide => {
        guide.variants.forEach(variant => {
          repoToGuideId[variant.variant_repo] = guide.base_model_id;
        });
      });

      // Set initial states using the state machine
      setModelStates(prevStates => {
        const newStates = new Map(prevStates);

        // Set downloading states
        downloading_models.forEach(repoName => {
          const guideId =
            repoToGuideId[repoName] ||
            currentGuides.find(g => g.base_model_id === repoName)?.base_model_id;

          if (guideId) {
            newStates.set(guideId, {
              ...createModelState(MODEL_STATES.DOWNLOADING),
              shouldAutoDeploy: false, // Download-only initially
            });
            
          }
        });

        // Set deploying states (with shouldAutoDeploy flag)
        pending_deployments.forEach(repoName => {
          const guideId =
            repoToGuideId[repoName] ||
            currentGuides.find(g => g.base_model_id === repoName)?.base_model_id;

          if (guideId) {
            const currentState = newStates.get(guideId) || createModelState();
            newStates.set(guideId, {
              ...currentState,
              status:
                currentState.status === MODEL_STATES.DOWNLOADING
                  ? MODEL_STATES.DOWNLOADING
                  : MODEL_STATES.DEPLOYING,
              shouldAutoDeploy: true, // This model should auto-deploy after download
            });
            
          }
        });

        return newStates;
      });

      
    } catch (error) {
      console.error('STATE_RECOVERY: Failed to recover backend state:', error);
      // Don't throw - this is a best-effort recovery
    }
  };

  // Legacy isModelDownloaded function removed - replaced by state machine

  const isModelAvailable = guide => {
    // Check if the relevant variant has all required files downloaded
    const variant = getRelevantVariant(guide);
    // Use ref to get current models array
    const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
    const model = currentModels.find(m => m.repo_modelId === variant.variant_repo);

    if (!model || !model.m_files) {
      return false;
    }

    // For GGUF models, we only need the recommended file
    if (variant.variant_type === 'gguf' && variant.recommended_file) {
      const isAvailable = model.m_files.some(
        file =>
          file.name === variant.recommended_file &&
          file.storage_location &&
          !file.is_downloading &&
          file.dl_requested_at === null
      );

      return isAvailable;
    }

    // For non-GGUF models, we need all files to be downloaded
    // (This is a simplified check - in reality we might need to be more sophisticated)
    const downloadedFiles = model.m_files.filter(
      file => file.storage_location && !file.is_downloading && file.dl_requested_at === null
    );

    const isAvailable = downloadedFiles.length > 0;

    

    // Check if we have at least one substantial file downloaded
    // (This assumes that if any file is downloaded, the model is available)
    
    return isAvailable;
  };

  const hasInsufficientMemory = guide => {
    if (!filters.maxVram || !clusterCapabilities) return false;

    const variant = getRelevantVariant(guide);
    if (!variant || !variant.minimum_vram) return false;

    return variant.minimum_vram > filters.maxVram;
  };

  const renderDeploymentMessage = deploymentInfo => {
    if (!deploymentInfo || !deploymentInfo.deployed) return null;

    const { engine_name, status, last_error_code, last_error_message } = deploymentInfo;
    const runtimeInfo = deploymentInfo.runtime_info || buildModelRuntimeInfo(deploymentInfo, routingMode);
    const endpointLabel = runtimeInfo.displayEndpoint || 'unavailable';

    return (
      <Box
        sx={{
          mb: 1,
          p: 1,
          bgcolor: 'rgba(63, 81, 181, 0.15)', // Brighter blue background
          borderRadius: 1,
          border: '1px solid rgba(63, 81, 181, 0.4)', // Brighter blue border
        }}
      >
        <Typography variant="caption" sx={{ fontWeight: 500, color: '#c5cae9' }}>
          Model {status.toLowerCase()} at {endpointLabel}.{' '}
        </Typography>
        {status === 'FAILED' && (
          <Typography variant="caption" sx={{ display: 'block', color: '#ef9a9a' }}>
            {last_error_code ? `${last_error_code}: ` : ''}{last_error_message || 'Deployment failed'}
          </Typography>
        )}
        {engine_name === 'mlx' ? (
            <>
              <a href="/apps" style={{ color: 'inherit', textDecoration: 'underline' }}>
                Deploy an App
              </a>{' '}
              or{' '}
              <a
                href={runtimeInfo.docsUrl}
                target="_blank"
                rel="noopener noreferrer"
                style={{ color: 'inherit', textDecoration: 'underline' }}
              >
                view the API
              </a>
            </>
          ) : engine_name === 'llamacpp' ? (
            <a
              href={runtimeInfo.baseUrl}
              target="_blank"
              rel="noopener noreferrer"
              style={{ color: 'inherit', textDecoration: 'underline' }}
            >
              Try to connect directly
            </a>
          ) : engine_name === 'vllm' ? (
            <>
              <a href="/apps" style={{ color: 'inherit', textDecoration: 'underline' }}>
                Deploy an App
              </a>{' '}
              or{' '}
              <a
                href={runtimeInfo.docsUrl}
                target="_blank"
                rel="noopener noreferrer"
                style={{ color: 'inherit', textDecoration: 'underline' }}
              >
                view the API docs
              </a>
            </>
          ) : (
            <>
              <a href="/apps" style={{ color: 'inherit', textDecoration: 'underline' }}>
                Deploy an App
              </a>{' '}
              or{' '}
              <a
                href={runtimeInfo.docsUrl}
                target="_blank"
                rel="noopener noreferrer"
                style={{ color: 'inherit', textDecoration: 'underline' }}
              >
                view the API
              </a>
            </>
          )}
      </Box>
    );
  };

  // Legacy getModelProgress function removed - replaced by state machine

  // Legacy function removed - using state machine directly



  const getValidVariants = (guide) => {
    if (!clusterCapabilities) return guide.variants; // Show all variants if no cluster info
    
    const { available_platforms, has_llamacpp, supports_gguf, system_type } = clusterCapabilities;
    
    const filteredVariants = guide.variants.filter(variant => {
      const variantPlatforms = Array.isArray(variant.platform) ? variant.platform : [variant.platform];
      
      // For MLX variants on Apple systems, always include them if system is Apple
      if (variant.variant_type === 'mlx' && system_type === 'apple') {
        return true;
      }
      
      // Special case: if llamacpp is available and supports_gguf, allow GGUF variants
      if (variant.variant_type === 'gguf' && has_llamacpp && supports_gguf) {
        return true;
      }
      
      // For Mac users, filter out CPU-only variants when Apple Silicon alternatives are available
      if (system_type === 'apple') {
        const hasOnlyCpuPlatforms = variantPlatforms.every(p => 
          p === 'Fast CPU' || p === 'CPU'
        );
        const hasAnyAppleSiliconPlatform = variantPlatforms.some(p => 
          p.includes('Mac (Apple Silicon)') || p.includes('Apple Silicon')
        );
        
        // Only filter out if this variant is CPU-only AND has no Apple Silicon platforms
        if (hasOnlyCpuPlatforms && !hasAnyAppleSiliconPlatform) {
          const hasAppleSiliconAlternatives = guide.variants.some(v => {
            const vPlatforms = Array.isArray(v.platform) ? v.platform : [v.platform];
            return vPlatforms.some(p => 
              p.includes('Mac (Apple Silicon)') || p.includes('Apple Silicon')
            ) || v.variant_type === 'mlx';
          });
          
          // Don't show CPU-only variants if Apple Silicon alternatives exist
          if (hasAppleSiliconAlternatives) {
            return false;
          }
        }
      }
      
      // Check if variant platform is supported - be more flexible with platform matching
      const platformSupported = variantPlatforms.some(platform => {
        // Direct match
        if (available_platforms.includes(platform)) return true;
        
        // Flexible matching for platform names
        return available_platforms.some(availPlatform => {
          // Check for partial matches (case insensitive)
          const platLower = platform.toLowerCase();
          const availLower = availPlatform.toLowerCase();
          
          // Match if either contains the other, or both contain key terms
          if (platLower.includes(availLower) || availLower.includes(platLower)) return true;
          
          // Special cases for Apple Silicon variants
          if (platLower.includes('mac') && platLower.includes('apple') && 
              (availLower.includes('mac') || availLower.includes('apple'))) return true;
          
          // Special cases for GPU variants
          if (platLower.includes('gpu') && availLower.includes('gpu')) return true;
          
          // Special cases for CPU variants
          if ((platLower.includes('cpu') || platLower.includes('fast cpu')) && 
              (availLower.includes('cpu') || availLower.includes('fast cpu'))) return true;
          
          return false;
        });
      });
      
      return platformSupported;
    });
    
    return filteredVariants;
  };

  const getRelevantVariant = guide => {
    // If user has selected a specific variant, use that
    if (guide.selectedVariant) {
      return guide.selectedVariant;
    }

    // Use cluster capabilities if available, otherwise fall back to browser detection
    if (clusterCapabilities) {
      const { system_type, available_platforms, supports_gguf, total_node_vram, gpu_count } =
        clusterCapabilities;

      // Find variants that match available platforms
      const compatibleVariants = guide.variants.filter(variant => {
        // For MLX variants on Apple systems, always include them regardless of platform matching
        if (variant.variant_type === 'mlx' && system_type === 'apple') {
          return true;
        }
        
        // Handle both string and array formats for platform
        const variantPlatforms = Array.isArray(variant.platform) ? variant.platform : [variant.platform];
        return variantPlatforms.some(platform => {
          // Direct match
          if (available_platforms.includes(platform)) return true;
          
          // Flexible matching for platform names
          return available_platforms.some(availPlatform => {
            // Check for partial matches (case insensitive)
            const platLower = platform.toLowerCase();
            const availLower = availPlatform.toLowerCase();
            
            // Match if either contains the other, or both contain key terms
            if (platLower.includes(availLower) || availLower.includes(platLower)) return true;
            
            // Special cases for Apple Silicon variants
            if (platLower.includes('mac') && platLower.includes('apple') && 
                (availLower.includes('mac') || availLower.includes('apple'))) return true;
            
            // Special cases for GPU variants
            if (platLower.includes('gpu') && availLower.includes('gpu')) return true;
            
            // Special cases for CPU variants
            if ((platLower.includes('cpu') || platLower.includes('fast cpu')) && 
                (availLower.includes('cpu') || availLower.includes('fast cpu'))) return true;
            
            return false;
          });
        });
      });

      // If no compatible variants found, use browser fallback
      if (compatibleVariants.length === 0) {
        return getBrowserBasedVariant(guide);
      }

      // Prioritize variants based on system capabilities

      // Helper function to check if variant supports platform
      const variantSupportsplatform = (variant, platform) => {
        const variantPlatforms = Array.isArray(variant.platform)
          ? variant.platform
          : [variant.platform];
        return variantPlatforms.includes(platform);
      };

      // 1. Apple Silicon: prefer MLX, then GGUF if llamacpp available
      if (system_type === 'apple') {
        const mlxVariants = compatibleVariants.filter(v => v.variant_type === 'mlx');
        if (mlxVariants.length > 0) {
          // Prefer the highest quality MLX variant (lowest quality_penalty or 6-bit over 4-bit)
          const sortedMlxVariants = mlxVariants.sort((a, b) => {
            // Prefer 6-bit over 4-bit if both available
            if (a.variant_repo.includes('6bit') && b.variant_repo.includes('4bit')) return -1;
            if (a.variant_repo.includes('4bit') && b.variant_repo.includes('6bit')) return 1;
            // Otherwise prefer lower quality penalty
            return (a.quality_penalty || 0) - (b.quality_penalty || 0);
          });
          return sortedMlxVariants[0];
        }
        
        // If no MLX, try GGUF if llamacpp is available
        if (supports_gguf) {
          const ggufVariant = compatibleVariants.find(v => v.variant_type === 'gguf');
          if (ggufVariant) return ggufVariant;
        }
      }

      // 2. Intel OpenCL systems: prioritize Intel OpenCL variants for better performance
      if (available_platforms.includes('Intel OpenCL')) {
        const intelOpenclVariants = compatibleVariants.filter(v => variantSupportsplatform(v, 'Intel OpenCL'));
        if (intelOpenclVariants.length > 0) {
          // Prefer Intel OpenCL variants for better performance on Intel GPUs
          return intelOpenclVariants[0];
        }
      }

      // 3. GPU systems: prefer AWQ -> FP16 -> FP8, check VRAM requirements
      if (gpu_count > 0 && available_platforms.includes('GPU')) {
        const gpuVariants = compatibleVariants.filter(v => variantSupportsplatform(v, 'GPU'));

        // Sort by preference and VRAM requirements
        const suitableGpuVariants = gpuVariants.filter(
          v => !v.minimum_vram || v.minimum_vram <= total_node_vram
        );

        if (suitableGpuVariants.length > 0) {
          // Prefer AWQ for efficiency
          const awqVariant = suitableGpuVariants.find(v => v.variant_type === 'awq');
          if (awqVariant) return awqVariant;

          // Then FP16 for quality
          const fp16Variant = suitableGpuVariants.find(v => v.variant_type === 'fp16');
          if (fp16Variant) return fp16Variant;

          // Then FP8 for memory efficiency
          const fp8Variant = suitableGpuVariants.find(v => v.variant_type === 'fp8');
          if (fp8Variant) return fp8Variant;

          // Return any GPU variant that fits
          return suitableGpuVariants[0];
        }
      }

      // 4. CPU fallback: prefer GGUF if llamacpp available
      if (available_platforms.includes('Fast CPU')) {
        if (supports_gguf) {
          const ggufVariant = compatibleVariants.find(v => v.variant_type === 'gguf');
          if (ggufVariant) return ggufVariant;
        }

        // Return any CPU variant
        const cpuVariant = compatibleVariants.find(v => variantSupportsplatform(v, 'Fast CPU'));
        if (cpuVariant) return cpuVariant;
      }

      // Return first compatible variant
      return compatibleVariants[0];
    }

    // Fallback to browser-based detection if cluster capabilities not available
    return getBrowserBasedVariant(guide);
  };

  const getBrowserBasedVariant = guide => {
    // Browser-based detection fallback
    const hasGPU = navigator.gpu !== undefined; // Basic WebGPU detection
    const hasMac = navigator.platform.toLowerCase().includes('mac');
    const isAppleSilicon = hasMac && navigator.userAgent.includes('ARM'); // Rough M1/M2/M3 detection

    if (isAppleSilicon) {
      // Prefer Mac MLX variants for Apple Silicon
      return (
        guide.variants.find(v => v.platform === 'Mac (Apple Silicon)') ||
        guide.variants.find(v => v.platform === 'Apple Silicon') ||
        guide.variants[0]
      );
    } else if (hasGPU) {
      // Prefer GPU variants, prioritize AWQ over GPTQ
      const awqVariant = guide.variants.find(v => v.platform === 'GPU' && v.variant_type === 'awq');
      if (awqVariant) return awqVariant;
      return guide.variants.find(v => v.platform === 'GPU') || guide.variants[0];
    } else {
      // CPU variant
      return guide.variants.find(v => v.platform === 'Fast CPU') || guide.variants[0];
    }
  };

  const getFilteredAndSortedGuides = () => {
    let filteredGuides = modelGuides.filter(guide => {
      // Use case filter
      if (filters.useCase !== 'all') {
        const guideUseCases = Array.isArray(guide.use_case) ? guide.use_case : [guide.use_case];
        if (!guideUseCases.includes(filters.useCase)) {
          return false;
        }
      }

      // Size category filter
      if (filters.sizeCategory !== 'all' && guide.size_category !== filters.sizeCategory) {
        return false;
      }

      // Platform support filter (applies to all models, not just those with VRAM requirements)
      if (!filters.showUnsupported && clusterCapabilities) {
        const variant = getRelevantVariant(guide);
        if (!variant) return false;
        
        const variantPlatforms = Array.isArray(variant.platform) ? variant.platform : [variant.platform];
        const isSupported = variantPlatforms.some(platform => {
          // Direct match
          if (clusterCapabilities.available_platforms.includes(platform)) return true;
          
          // Flexible matching for platform names
          return clusterCapabilities.available_platforms.some(availPlatform => {
            // Check for partial matches (case insensitive)
            const platLower = platform.toLowerCase();
            const availLower = availPlatform.toLowerCase();
            
            // Match if either contains the other, or both contain key terms
            if (platLower.includes(availLower) || availLower.includes(platLower)) return true;
            
            // Special cases for Apple Silicon variants
            if (platLower.includes('mac') && platLower.includes('apple') && 
                (availLower.includes('mac') || availLower.includes('apple'))) return true;
            
            // Special cases for GPU variants
            if (platLower.includes('gpu') && availLower.includes('gpu')) return true;
            
            // Special cases for CPU variants
            if ((platLower.includes('cpu') || platLower.includes('fast cpu')) && 
                (availLower.includes('cpu') || availLower.includes('fast cpu'))) return true;
            
            return false;
          });
        });
        
        if (!isSupported) return false;
      }

      // VRAM filter - only exclude if showUnsupported is false
      if (filters.maxVram !== null && !filters.showUnsupported) {
        const variant = getRelevantVariant(guide);
        if (!variant) return false;

        // Check VRAM requirements
        if (variant.minimum_vram && variant.minimum_vram > filters.maxVram) {
          return false;
        }
      }

      return true;
    });

    // Sort the filtered guides
    filteredGuides.sort((a, b) => {
      switch (sortBy) {
        case 'name':
          return a.name.localeCompare(b.name);
        case 'sizeUp':
          return getSizeOrder(a.size_category) - getSizeOrder(b.size_category);
        case 'sizeDown':
          return getSizeOrder(b.size_category) - getSizeOrder(a.size_category);
        case 'score':
          return (b.score_average || 0) - (a.score_average || 0);
        default:
          return 0;
      }
    });

    return filteredGuides;
  };

  const getSizeOrder = size => {
    const sizes = { small: 1, medium: 2, large: 3, very_large: 4 };
    return sizes[size] || 0;
  };

  const handleVramEdit = () => {
    setEditingVram(true);
    setVramInputValue(filters.maxVram?.toString() || '');
  };

  const handleVramSave = () => {
    const newVram = parseInt(vramInputValue) || 0;
    if (newVram >= 0) {
      setFilters(prev => ({ ...prev, maxVram: newVram }));
    }
    setEditingVram(false);
  };

  const handleVramCancel = () => {
    setVramInputValue(filters.maxVram?.toString() || '');
    setEditingVram(false);
  };

  const handleVramKeyDown = e => {
    if (e.key === 'Enter') {
      handleVramSave();
    } else if (e.key === 'Escape') {
      handleVramCancel();
    }
  };

  const handleDownloadModel = async (guide, autoDeploy = false) => {
    const variant = getRelevantVariant(guide);
    const guideId = guide.base_model_id;

    // If this is a deploy request, set the recently clicked deploy timer
    if (autoDeploy) {
      setRecentlyClickedDeploy(guideId);
      // Clear the timer after 5 seconds
      setTimeout(() => {
        setRecentlyClickedDeploy(null);
      }, 5000);
    }

    // Set initial state
    updateModelState(guideId, {
      status: MODEL_STATES.DOWNLOADING,
      shouldAutoDeploy: autoDeploy,
      progress: 0,
      downloadSpeed: 0,
      timeRemaining: null,
    });

    try {
      // Create model if it doesn't exist
      const modelData = {
        repo_modelId: variant.variant_repo,
        name: `${guide.name} (${variant.variant_type.toUpperCase()})`,
        modelfamily: guide.producer,
        description: guide.description,
        hub: 'HubsHf',
      };

      await axios.post(`${BASE_URL}/models/`, modelData);

      // Use download_and_deploy endpoint
      const contextSize = getSelectedContextSize(guide);
      
      // Debug logging
      console.log('Selected context for download & deploy:', {
        guide: guide.name,
        contextSize: contextSize,
        userSelected: selectedContextSizes.has(guide.base_model_id),
        selectedValue: selectedContextSizes.get(guide.base_model_id)
      });
      
      const downloadRequest = {
        model: variant.variant_repo,
        version: null,
        hub: 'HubsHf',
        files_to_download: variant.recommended_file ? [variant.recommended_file] : null,
        deploy_after_download: autoDeploy,
        novice_selected_context: contextSize,
      };

      

      const response = await axios.post(`${BASE_URL}/models/download_and_deploy`, downloadRequest);

      // Check if files were queued for download
      if (response.data.files && response.data.files.length === 0) {
        // No files to download - model already available
        if (isModelAvailable(guide)) {
          updateModelState(guideId, {
            status: autoDeploy ? MODEL_STATES.DEPLOYING : MODEL_STATES.DOWNLOADED,
            progress: 100,
          });
        } else {
          // Reset to idle if model not available and no download needed
          updateModelState(guideId, { status: MODEL_STATES.IDLE });
        }
      }

      // Refresh models list (state sync will handle the rest)
      await fetchModels();
    } catch (error) {
      console.error(`Failed to ${autoDeploy ? 'download and deploy' : 'download'} model:`, error);
      updateModelState(guideId, {
        status: MODEL_STATES.FAILED,
        error: error.message,
      });
    }
  };

  // Convenience wrappers
  const handleDownloadAndDeploy = guide => handleDownloadModel(guide, true);
  const handleDownloadOnly = guide => handleDownloadModel(guide, false);

  // Legacy pollForDeploymentCompletion function removed - replaced by unified state sync

  const handleStopModel = async guide => {
    const guideId = guide.base_model_id;

    // Set stopping state immediately
    updateModelState(guideId, { status: MODEL_STATES.STOPPING });

    try {
      const variant = getRelevantVariant(guide);
      const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
      const model = currentModels.find(m => m.repo_modelId === variant.variant_repo);

      if (!model) {
        console.error('Model not found for stopping:', guide.name);
        updateModelState(guideId, { status: MODEL_STATES.IDLE });
        return;
      }

      // Get the deployment ID for this model
      const deploymentResponse = await axios.get(`${BASE_URL}/serving/deployments`);
      const deployments = deploymentResponse.data || [];

      // Find active deployment of this model
      const activeDeployment = deployments.find(
        deployment =>
          deployment.m_id === model.id &&
          (deployment.status === 'DEPLOYED' || deployment.status === 'RUNNING')
      );

      if (!activeDeployment) {
        updateModelState(guideId, { status: MODEL_STATES.DOWNLOADED });
        return;
      }

      

      // Stop the deployment with force=true
      await axios.delete(`${BASE_URL}/serving/deployment/${activeDeployment.id}`, {
        params: { force: true },
      });

      

      // State sync will detect the stopped deployment and update state accordingly
    } catch (error) {
      console.error('Error stopping model:', error);
      updateModelState(guideId, {
        status: MODEL_STATES.FAILED,
        error: error.message,
      });
    }
  };

  const handleResetFailedModel = async guide => {
    const guideId = guide.base_model_id;

    // Set resetting state immediately
    updateModelState(guideId, { status: MODEL_STATES.RESETTING });

    try {
      const variant = getRelevantVariant(guide);
      const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
      const model = currentModels.find(m => m.repo_modelId === variant.variant_repo);

      if (!model) {
      
        // Reset to idle if no model found
        updateModelState(guideId, { status: MODEL_STATES.IDLE });
        return;
      }

      // Get deployments and find any deployment for this model
      const deploymentResponse = await axios.get(`${BASE_URL}/serving/deployments`);
      const deployments = deploymentResponse.data || [];
      const deployment = deployments.find(dep => dep.m_id === model.id);

      if (deployment) {
        

        // Force remove the failed deployment
        await axios.delete(`${BASE_URL}/serving/deployment/${deployment.id}`, {
          params: { force: true },
        });

        
      }

      // Determine final state based on model availability
      if (isModelAvailable(guide)) {
        updateModelState(guideId, { status: MODEL_STATES.DOWNLOADED });
      } else {
        updateModelState(guideId, { status: MODEL_STATES.IDLE });
      }
    } catch (error) {
      console.error('Error resetting failed model:', error);
      updateModelState(guideId, {
        status: MODEL_STATES.FAILED,
        error: error.message,
      });
    }
  };

  const handleVariantSwitch = guide => {
    setSelectedGuideForVariant(guide);
    setVariantDialog(true);
  };

  const handleVariantSelection = async (guide, selectedVariant) => {
    setVariantDialog(false);

    // Update the guide to use the selected variant as the primary one
    // This is a UI-only change - we'll use the selected variant for operations
    guide.selectedVariant = selectedVariant;

    // Save selection to localStorage with 24h expiry
    saveVariantSelection(guide.base_model_id, selectedVariant);

    // Reset the state for this guide to reflect the new variant's status
    const guideId = guide.base_model_id;
    updateModelState(guideId, { status: MODEL_STATES.IDLE, progress: 0, downloadSpeed: 0, timeRemaining: null });

    // Force re-render to update the recommended variant display
    setModelGuides(prev => [...prev]);

    // Trigger an immediate state sync to get the correct status for the new variant
    setTimeout(() => {
      syncAllModelStates();
    }, 100);
  };

  const cancelDownload = async guide => {
    const guideId = guide.base_model_id;

    try {
      // Find the model ID for this guide by looking at its variant
      const variant = getRelevantVariant(guide);
      const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
      const model = currentModels.find(m => m.repo_modelId === variant.variant_repo);

        if (model && model.id) {
        // Cancel downloads for this specific model only
        await axios.delete(`${BASE_URL}/model_files/downloads/cancel_all?model_id=${model.id}`);
      } else {
          // Fallback to cancel all downloads if we can't find the specific model
          await axios.delete(`${BASE_URL}/model_files/downloads/cancel_all`);
      }

      // Reset to idle state
      updateModelState(guideId, {
        status: MODEL_STATES.IDLE,
        progress: 0,
        downloadSpeed: 0,
        timeRemaining: null,
      });
    } catch (error) {
      console.error('Failed to cancel download:', error);
      updateModelState(guideId, {
        status: MODEL_STATES.FAILED,
        error: error.message,
      });
    }
  };

  const handleDeployModel = guide => {
    setSelectedModel(guide);
    setDeployDialog(true);
  };

  const confirmDeploy = async () => {
    const guideId = selectedModel.base_model_id;

    // Set deploying state immediately
    updateModelState(guideId, { status: MODEL_STATES.DEPLOYING });

    // Close dialog immediately to show spinner on card
    setDeployDialog(false);
    setSelectedModel(null);

    try {
      const variant = getRelevantVariant(selectedModel);
      const currentModels = modelsRef.current.length > 0 ? modelsRef.current : models;
      const model = currentModels.find(m => m.repo_modelId === variant.variant_repo);

      if (!model) {
        console.error('Model not found. Please download it first.');
        updateModelState(guideId, { status: MODEL_STATES.IDLE });
        return;
      }

      // For already-downloaded models in Novice mode, we just need to deploy directly
      // The deploy_after_download endpoint will handle everything including config creation
      const contextSize = getSelectedContextSize(selectedModel);
      
      // Debug logging
      console.log('Selected context for deployment:', {
        guide: selectedModel.name,
        contextSize: contextSize,
        userSelected: selectedContextSizes.has(selectedModel.base_model_id),
        selectedValue: selectedContextSizes.get(selectedModel.base_model_id)
      });
      
      // Get the list of already-downloaded files
      let filesToSpecify = [];
      if (model.m_files) {
        // For GGUF models, use the recommended file if it exists and is downloaded
        if (variant.variant_type === 'gguf' && variant.recommended_file) {
          const recommendedFile = model.m_files.find(file => 
            file.name === variant.recommended_file && 
            file.storage_location && 
            !file.is_downloading
          );
          if (recommendedFile) {
            filesToSpecify = [variant.recommended_file];
          }
        } else {
          // For non-GGUF models, get all downloaded files
          // Note: dl_requested_at should be null for completed downloads
          const downloadedFiles = model.m_files.filter(
            file => file.storage_location && !file.is_downloading
          );
          filesToSpecify = downloadedFiles.map(f => f.name);
        }
      }
      
      console.log('Files to specify for deployment:', filesToSpecify);
      
      // For already-downloaded models, pass context and files directly to deploy endpoint
      const deployRequest = {
        novice_selected_context: contextSize,
        files_to_deploy: filesToSpecify.length ? filesToSpecify : null
      };

      if (isDevelopment) {
        console.log('Deploying already-downloaded model with context:', deployRequest);
      }

      // Call deploy_after_download with context and files as parameters
      await axios.post(`${BASE_URL}/models/deploy_after_download/${model.id}`, deployRequest);

      
    } catch (error) {
      console.error('Failed to deploy model:', error);
      updateModelState(guideId, {
        status: MODEL_STATES.FAILED,
        error: error.message,
      });
    }
  };

  const getQualityRating = quality => {
    const ratings = {
      'State-of-the-art': 5,
      'Near State-of-the-art': 4.5,
      Excellent: 4,
      'Very Good': 3.5,
      Good: 3,
      Fair: 2,
      Poor: 1,
    };
    return ratings[quality] || 3;
  };

  const getSizeSpeedLabel = speedRating => {
    const labels = {
      very_fast: { label: 'Very Fast', color: '#4caf50' },
      fast: { label: 'Fast', color: '#8bc34a' },
      medium: { label: 'Moderate', color: '#ff9800' },
      slow: { label: 'Slower', color: '#f44336' },
    };
    return labels[speedRating] || { label: 'Unknown', color: '#666' };
  };

  const getSizeCategoryLabel = sizeCategory => {
    const labels = {
      'tiny': 'Tiny (≤2B)',
      'small': 'Small (≤8B)',
      'medium': 'Medium (8B-32B)', 
      'large': 'Large (33B-72B)',
      'very_large': 'Very Large (>72B)'
    };
    return labels[sizeCategory] || sizeCategory;
  };

  const getUseCaseTooltip = () => (
    <Box sx={{ maxWidth: 400, p: 1 }}>
      <Typography variant="subtitle2" sx={{ fontWeight: 600, mb: 1 }}>
        Choose based on your primary use case:
      </Typography>
      <Box sx={{ mb: 1 }}>
        <Typography variant="body2" sx={{ fontWeight: 500, color: '#2196f3' }}>
          General Use
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary' }}>
          Versatile models for writing, coding, analysis, and everyday tasks. Good balance of
          capabilities.
        </Typography>
      </Box>
      <Box sx={{ mb: 1 }}>
        <Typography variant="body2" sx={{ fontWeight: 500, color: '#9c27b0' }}>
          Reasoning
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary' }}>
          Specialized for complex problem-solving, math, logic, and multi-step analysis.
        </Typography>
      </Box>
      <Box sx={{ mb: 1 }}>
        <Typography variant="body2" sx={{ fontWeight: 500, color: '#ff9800' }}>
          Vision/MultiModal
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary' }}>
          Can understand and analyze images, charts, and visual content alongside text.
        </Typography>
      </Box>
      <Box>
        <Typography variant="body2" sx={{ fontWeight: 500, color: '#e91e63' }}>
          Roleplaying
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary' }}>
          Optimized for creative writing, character roleplay, and storytelling.
        </Typography>
      </Box>
    </Box>
  );


  // Context selection helpers
  const getSelectedContextSize = (guide) => {
    return selectedContextSizes.get(guide.base_model_id) || 8192; // Default to 8k
  };

  const handleContextClick = (guide) => {
    setSelectedGuideForContext(guide);
    setContextDialog(true);
  };

  const handleContextSelection = (contextSize) => {
    if (selectedGuideForContext) {
      const newSizes = new Map(selectedContextSizes);
      newSizes.set(selectedGuideForContext.base_model_id, contextSize);
      setSelectedContextSizes(newSizes);
    }
    setContextDialog(false);
    setSelectedGuideForContext(null);
  };

  const getContextOptions = (maxContext) => {
    const options = [];
    let size = 1024;
    while (size <= maxContext) {
      options.push(size);
      size *= 2;
    }
    // If no options were generated (maxContext too small), at least return some defaults
    if (options.length === 0) {
      console.warn('No context options generated for maxContext:', maxContext);
      return [1024, 2048, 4096, 8192]; // Default options
    }
    return options;
  };

  const formatContextSize = (size) => {
    if (size >= 1024) {
      return `${size / 1024}k`;
    }
    return `${size}`;
  };

  const getSizeTooltip = () => (
    <Box sx={{ maxWidth: 500, p: 1 }}>
      <Typography variant="subtitle2" sx={{ fontWeight: 600, mb: 1 }}>
        Model Size vs Performance
      </Typography>
      <Typography variant="body2" sx={{ mb: 2 }}>
        Larger models generally produce better results but require more memory and run slower.
      </Typography>

      <TableContainer component={Paper} sx={{ mb: 2, bgcolor: 'rgba(0,0,0,0.02)' }}>
        <Table size="small">
          <TableHead>
            <TableRow>
              <TableCell sx={{ fontWeight: 600 }}>Size</TableCell>
              <TableCell sx={{ fontWeight: 600 }}>Capability</TableCell>
              <TableCell sx={{ fontWeight: 600 }}>Best For</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            <TableRow>
              <TableCell>Small (≤8B)</TableCell>
              <TableCell>⭐⭐ Limited</TableCell>
              <TableCell>Simple tasks, fast responses</TableCell>
            </TableRow>
            <TableRow>
              <TableCell>Medium (8B-32B)</TableCell>
              <TableCell>⭐⭐⭐⭐ Strong</TableCell>
              <TableCell>Most tasks, good balance</TableCell>
            </TableRow>
            <TableRow>
              <TableCell>Large (33B-72B)</TableCell>
              <TableCell>⭐⭐⭐⭐⭐ Excellent</TableCell>
              <TableCell>Complex work, high quality</TableCell>
            </TableRow>
            <TableRow>
              <TableCell>Very Large (&gt;72B)</TableCell>
              <TableCell>⭐⭐⭐⭐⭐ State-of-art</TableCell>
              <TableCell>Research, expert analysis</TableCell>
            </TableRow>
          </TableBody>
        </Table>
      </TableContainer>

      <Box>
        <Typography variant="body2" sx={{ fontWeight: 500, mb: 0.5 }}>
          Key Trade-offs:
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block' }}>
          • Small models: More errors, limited reasoning, but very fast
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block' }}>
          • Medium models: Leading medium models can approach state-of-the-art
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block' }}>
          • Large models: Near state-of-the-art performance across most tasks
        </Typography>
        <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block' }}>
          • Very Large models: State-of-the-art capabilities, leading in evaluations
        </Typography>
      </Box>
    </Box>
  );

  if (loading || clusterLoading) {
    return (
      <ContentContainer>
        <LinearProgress />
      </ContentContainer>
    );
  }

  return (
    <ContentContainer>
      <SectionTitle variant="h4" data-tour="recommended-models">
        Recommended Models
      </SectionTitle>

      <Alert severity="info" sx={{ mb: 3 }}>
        These models are a curated selection by the Kamiwaza team as some of the most worthy
        candidates. They offer some of the best performance for their size, and we&apos;ve tried to
        select optimal quality quantizations.
      </Alert>

      {/* Filter Controls */}
      <FilterContainer>
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
          <FilterListIcon color="primary" />
          <Typography variant="h6" sx={{ color: 'primary.main' }}>
            Filters
          </Typography>
        </Box>

        <FilterRow>
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
            <FormControl variant="outlined" size="small" sx={{ minWidth: 150 }}>
              <InputLabel>Use Case</InputLabel>
              <Select
                value={filters.useCase}
                onChange={e => setFilters(prev => ({ ...prev, useCase: e.target.value }))}
                label="Use Case"
              >
                <MenuItem value="all">All Use Cases</MenuItem>
                <MenuItem value="General Use">General Use</MenuItem>
                <MenuItem value="Reasoning">Reasoning</MenuItem>
                <MenuItem value="Vision/MultiModal">Vision/MultiModal</MenuItem>
                <MenuItem value="Roleplaying">Roleplaying</MenuItem>
              </Select>
            </FormControl>
            <Tooltip title={getUseCaseTooltip()} arrow placement="top">
              <HelpOutlineIcon sx={{ fontSize: 16, color: 'text.secondary', cursor: 'help' }} />
            </Tooltip>
          </Box>

          <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
            <FormControl variant="outlined" size="small" sx={{ minWidth: 150 }}>
              <InputLabel>Size</InputLabel>
              <Select
                value={filters.sizeCategory}
                onChange={e => setFilters(prev => ({ ...prev, sizeCategory: e.target.value }))}
                label="Size"
              >
                <MenuItem value="all">All Sizes</MenuItem>
                <MenuItem value="small">Small (≤8B)</MenuItem>
                <MenuItem value="medium">Medium (8B-32B)</MenuItem>
                <MenuItem value="large">Large (33B-72B)</MenuItem>
                <MenuItem value="very_large">Very Large (&gt;72B)</MenuItem>
              </Select>
            </FormControl>
            <Tooltip title={getSizeTooltip()} arrow placement="top">
              <HelpOutlineIcon sx={{ fontSize: 16, color: 'text.secondary', cursor: 'help' }} />
            </Tooltip>
          </Box>

          <FormControl variant="outlined" size="small" sx={{ minWidth: 120 }}>
            <InputLabel>Sort By</InputLabel>
            <Select value={sortBy} onChange={e => setSortBy(e.target.value)} label="Sort By">
              <MenuItem value="score">Quality Score</MenuItem>
              <MenuItem value="name">Name</MenuItem>
              <MenuItem value="sizeUp">Size (Small → Large)</MenuItem>
              <MenuItem value="sizeDown">Size (Large → Small)</MenuItem>
            </Select>
          </FormControl>

          {/* VRAM Filter */}
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
            <Typography variant="body2" sx={{ color: 'text.secondary' }}>
              Max VRAM for Models:
            </Typography>
            {editingVram ? (
              <TextField
                size="small"
                value={vramInputValue}
                onChange={e => setVramInputValue(e.target.value)}
                onKeyDown={handleVramKeyDown}
                onBlur={handleVramSave}
                sx={{ width: 80 }}
                inputProps={{
                  style: { textAlign: 'center' },
                  min: 0,
                  type: 'number',
                }}
                autoFocus
              />
            ) : (
              <VramButton
                variant="outlined"
                size="small"
                onClick={handleVramEdit}
                endIcon={<EditIcon />}
              >
                {filters.maxVram || 0}GB
              </VramButton>
            )}
            {clusterCapabilities && clusterCapabilities.total_node_vram && (
              <Typography variant="caption" sx={{ color: 'text.secondary', ml: 0.5 }}>
                {clusterCapabilities.total_node_vram}GB detected
              </Typography>
            )}
            {twoNodeInfo && (
              <Tooltip
                title={`Remote GPU node ${twoNodeInfo.remote_host || ''} ${
                  twoNodeInfo.remote_user ? `via ${twoNodeInfo.remote_user}` : ''
                }`}
                arrow
              >
                <Chip
                  size="small"
                  color="primary"
                  icon={<MemoryIcon fontSize="inherit" />}
                  label={`Paired GPU node: ${twoNodeInfo.remote_host}`}
                  sx={{ ml: 1, fontSize: '0.7rem' }}
                />
              </Tooltip>
            )}
            {memoryWarning && (
              <Alert severity="warning" sx={{ ml: 2, py: 0.5, px: 1 }}>
                <Typography variant="caption">{memoryWarning}</Typography>
              </Alert>
            )}
          </Box>

          <FormControlLabel
            control={
              <Switch
                checked={filters.showUnsupported}
                onChange={e => setFilters(prev => ({ ...prev, showUnsupported: e.target.checked }))}
                size="small"
              />
            }
            label="Show Unsupported"
            sx={{ color: 'text.secondary' }}
          />
        </FilterRow>
      </FilterContainer>

      <Grid container spacing={3}>
        {getFilteredAndSortedGuides().map((guide, index) => {
          const variant = getRelevantVariant(guide);
          const isDownloaded = isModelDownloaded(guide.base_model_id);
          const isAvailable = isModelAvailable(guide);
          const isDownloading = isModelDownloading(guide.base_model_id);
          const isDeploying = isModelDeploying(guide.base_model_id);
          const isInitializing = isModelInitializing(guide.base_model_id);
          const isRunning = isModelRunning(guide.base_model_id);
          const isStopping = isModelStopping(guide.base_model_id);
          const isFailed = isModelFailed(guide.base_model_id);
          const isResetting = isModelResetting(guide.base_model_id);
          const speedInfo = getSizeSpeedLabel(variant.speed_rating);
          const modelDeploymentInfo = getModelDeploymentInfo(guide.base_model_id);

          // Check if any other model is deploying or initializing (for one-at-a-time deployment)
          const otherDeploymentInProgress = Array.from(modelStates.keys()).some(
            id => id !== guide.base_model_id && (isModelDeploying(id) || isModelInitializing(id))
          );

          // Also check if another model was recently clicked for deploy (5s window)
          const otherRecentlyClickedDeploy =
            recentlyClickedDeploy && recentlyClickedDeploy !== guide.base_model_id;

          return (
            <Grid item xs={12} md={6} lg={4} key={guide.id}>
              <ModelCard data-tour={index === 0 ? 'model-card' : undefined}>
                <CardContent sx={{ flexGrow: 1 }}>
                  <Box
                    sx={{
                      display: 'flex',
                      justifyContent: 'space-between',
                      alignItems: 'flex-start',
                      mb: 1,
                    }}
                  >
                    <Typography variant="h6" component="h3" sx={{ fontWeight: 600, flex: 1 }}>
                      {guide.name}
                    </Typography>
                    {isDownloaded && <CheckCircleIcon color="success" sx={{ ml: 1 }} />}
                  </Box>

                  <Box sx={{ display: 'flex', gap: 1, mb: 2, flexWrap: 'wrap' }}>
                    {(() => {
                      const useCases = Array.isArray(guide.use_case)
                        ? guide.use_case
                        : [guide.use_case];
                      return useCases.map((useCase, index) => (
                        <UseCaseChip
                          key={index}
                          label={useCase}
                          size="small"
                          usecase={useCase}
                          data-tour={index === 0 ? 'use-case-chip' : undefined}
                        />
                      ));
                    })()}
                    <Chip
                      label={`(${formatContextSize(getSelectedContextSize(guide))} / ${guide.context_length} context)`}
                      size="small"
                      variant="outlined"
                      sx={{ cursor: 'pointer' }}
                      onClick={() => handleContextClick(guide)}
                      data-tour={index === 0 ? 'context-info' : undefined}
                    />
                    <Chip
                      label={getSizeCategoryLabel(guide.size_category)}
                      size="small"
                      variant="outlined"
                      color="primary"
                    />
                  </Box>

                  {/* Deployment Info or Selected Variant Info */}
                  {(isRunning || isInitializing) && modelDeploymentInfo ? (
                    renderDeploymentMessage(modelDeploymentInfo)
                  ) : (
                    <Box
                      sx={{
                        mb: 1,
                        p: 1,
                        bgcolor: 'rgba(0, 192, 127, 0.05)',
                        borderRadius: 1,
                        border: '1px solid rgba(0, 192, 127, 0.2)',
                      }}
                      data-tour={index === 0 ? 'auto-config' : undefined}
                    >
                      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                        <Typography variant="caption" sx={{ fontWeight: 500, color: 'primary.main' }}>
                          {guide.selectedVariant ? 'Selected' : 'Recommended'} variant: {variant.variant_type.toUpperCase()} for {Array.isArray(variant.platform) ? variant.platform.join(', ') : variant.platform}
                        </Typography>
                        {getValidVariants(guide).length > 1 && (
                          <IconButton
                            size="small"
                            onClick={() => handleVariantSwitch(guide)}
                            sx={{ ml: 1 }}
                          >
                            <EditIcon fontSize="small" />
                          </IconButton>
                        )}
                      </Box>
                    </Box>
                  )}

                  {/* Progress Bar */}
                  {(() => {
                    const progress = getModelProgress(guide.base_model_id);
                    return (
                      progress.isDownloading && (
                        <Box sx={{ mb: 2 }}>
                          <Box
                            sx={{
                              display: 'flex',
                              justifyContent: 'space-between',
                              alignItems: 'center',
                              mb: 1,
                            }}
                          >
                            <Typography variant="body2" sx={{ fontWeight: 500 }}>
                              Downloading... {Math.round(progress.percentage)}%
                            </Typography>
                            {progress.speed > 0 && (
                              <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                                {progress.speed.toFixed(1)} MB/s
                                {progress.timeRemaining ? ` (${progress.timeRemaining} est)` : ''}
                              </Typography>
                            )}
                          </Box>
                          <LinearProgress
                            variant="determinate"
                            value={progress.percentage}
                            sx={{
                              height: 6,
                              borderRadius: 3,
                              bgcolor: 'rgba(0, 192, 127, 0.1)',
                              '& .MuiLinearProgress-bar': {
                                bgcolor: 'primary.main',
                              },
                            }}
                          />
                        </Box>
                      )
                    );
                  })()}

                  {/* Insufficient Memory Warning */}
                  {hasInsufficientMemory(guide) && (
                    <Alert
                      severity="error"
                      sx={{
                        mb: 2,
                        '& .MuiAlert-message': {
                          display: 'flex',
                          alignItems: 'center',
                          gap: 1,
                        },
                      }}
                    >
                      <Typography variant="body2" sx={{ fontWeight: 500, color: 'error.main' }}>
                        ⚠️ Insufficient memory estimated
                      </Typography>
                      <Typography variant="caption" sx={{ color: 'error.main' }}>
                        (Requires {variant.minimum_vram}GB, available {filters.maxVram}GB)
                      </Typography>
                    </Alert>
                  )}

                  {/* Failed Deployment Help Message */}
                  {isFailed && (
                    <Alert severity="warning" sx={{ mb: 2 }}>
                      <Typography variant="body2" sx={{ color: 'warning.main' }}>
                        Deployments failing?{' '}
                        <Typography
                          component="a"
                          href="https://docs.kamiwaza.ai/models/overview"
                          target="_blank"
                          rel="noopener noreferrer"
                          sx={{
                            color: 'primary.main',
                            textDecoration: 'none',
                            '&:hover': { textDecoration: 'underline' },
                          }}
                        >
                          Read more here
                        </Typography>
                      </Typography>
                    </Alert>
                  )}

                  <Typography variant="body2" sx={{ color: 'text.secondary', mb: 2 }}>
                    {guide.description}
                  </Typography>

                  <Divider sx={{ my: 2 }} />

                  {/* Quality and Performance - Merged into one line */}
                  <Box sx={{ mb: 2 }}>
                    <Box
                      sx={{ display: 'flex', alignItems: 'center', gap: 2 }}
                      data-tour={index === 0 ? 'quality-rating' : undefined}
                    >
                      <Box sx={{ display: 'flex', alignItems: 'center' }}>
                        <Typography variant="body2" sx={{ fontWeight: 500, mr: 1 }}>
                          Quality:
                        </Typography>
                        <Rating
                          value={getQualityRating(guide.quality_overall)}
                          precision={0.5}
                          size="small"
                          readOnly
                        />
                        <Typography variant="caption" sx={{ ml: 1, color: 'text.secondary' }}>
                          ({guide.quality_overall})
                        </Typography>
                      </Box>

                      <Box sx={{ display: 'flex', alignItems: 'center' }}>
                        <SpeedIcon sx={{ fontSize: 16, mr: 0.5, color: speedInfo.color }} />
                        <Typography variant="body2" sx={{ fontWeight: 500, mr: 1 }}>
                          Speed:
                        </Typography>
                        <Typography variant="body2" sx={{ color: speedInfo.color }}>
                          {speedInfo.label}
                        </Typography>
                      </Box>
                    </Box>
                  </Box>

                  {/* Variant-specific notes */}
                  {variant.notes && (
                    <Box sx={{ mb: 2 }}>
                      <Typography
                        variant="caption"
                        sx={{ fontStyle: 'italic', color: 'text.secondary' }}
                      >
                        {variant.notes}
                      </Typography>
                    </Box>
                  )}

                  {/* Platform Compatibility - Hidden when deployed */}
                  {!(isRunning && modelDeploymentInfo) && (
                    <Box
                      sx={{ mb: 2 }}
                      data-tour={index === 0 ? 'platform-compatibility' : undefined}
                    >
                      <Typography variant="body2" sx={{ fontWeight: 500, mb: 1 }}>
                        Available for:
                      </Typography>
                      <Box sx={{ display: 'flex', gap: 0.5, flexWrap: 'wrap' }}>
                        {[
                          ...new Set(
                            guide.variants.flatMap(v =>
                              Array.isArray(v.platform) ? v.platform : [v.platform]
                            )
                          ),
                        ].map(platform => (
                          <PlatformChip
                            key={platform}
                            label={platform}
                            size="small"
                            color="primary"
                            variant="outlined"
                          />
                        ))}
                      </Box>
                    </Box>
                  )}

                  {/* VRAM Requirements */}
                  {variant.minimum_vram > 0 && (
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                      <MemoryIcon sx={{ fontSize: 16, color: 'text.secondary' }} />
                      <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                        {variant.minimum_vram}GB min / {variant.recommended_vram}GB recommended VRAM
                      </Typography>
                    </Box>
                  )}

                  {/* Kamiwaza Notes */}
                  {guide.kamiwaza_notes && (
                    <Box sx={{ mt: 2, p: 1.5, bgcolor: 'rgba(0, 192, 127, 0.1)', borderRadius: 1 }}>
                      <Typography variant="caption" sx={{ fontStyle: 'italic' }}>
                        {guide.kamiwaza_notes}
                      </Typography>
                    </Box>
                  )}
                </CardContent>

                <CardActions sx={{ p: 2, pt: 0 }}>
                  {isStopping ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="error"
                      startIcon={<CircularProgress size={20} />}
                      disabled
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Stopping...
                    </Button>
                  ) : isRunning ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="error"
                      startIcon={<CancelIcon />}
                      onClick={() => handleStopModel(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Running - Stop
                    </Button>
                  ) : isInitializing && !isDownloading ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="primary"
                      startIcon={<CircularProgress size={20} />}
                      disabled
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Initializing...
                    </Button>
                  ) : isDeploying && !isDownloading ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="primary"
                      startIcon={<CircularProgress size={20} />}
                      disabled
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Deploying...
                    </Button>
                  ) : isDownloading && (isDeploying || isInitializing) ? (
                    <Box sx={{ display: 'flex', gap: 1 }}>
                      <Button
                        fullWidth
                        variant="outlined"
                        startIcon={<CircularProgress size={20} />}
                        disabled
                        data-tour={index === 0 ? 'deploy-button' : undefined}
                      >
                        Downloading and Deploying
                      </Button>
                      <Button
                        variant="outlined"
                        size="small"
                        onClick={() => cancelDownload(guide)}
                        sx={{ minWidth: 'auto', px: 1 }}
                      >
                        Cancel
                      </Button>
                    </Box>
                  ) : isDownloading ? (
                    <Box sx={{ display: 'flex', gap: 1 }}>
                      <Button
                        fullWidth
                        variant="outlined"
                        startIcon={<CircularProgress size={20} />}
                        disabled
                        data-tour={index === 0 ? 'deploy-button' : undefined}
                      >
                        Downloading...
                      </Button>
                      <Button
                        variant="outlined"
                        size="small"
                        onClick={() => cancelDownload(guide)}
                        sx={{ minWidth: 'auto', px: 1 }}
                      >
                        Cancel
                      </Button>
                    </Box>
                  ) : !isAvailable && !otherDeploymentInProgress && !otherRecentlyClickedDeploy ? (
                    <Button
                      fullWidth
                      variant="contained"
                      startIcon={<CloudDownloadIcon />}
                      onClick={() => handleDownloadAndDeploy(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Download & Deploy
                    </Button>
                  ) : !isAvailable && (otherDeploymentInProgress || otherRecentlyClickedDeploy) ? (
                    <Button
                      fullWidth
                      variant="outlined"
                      startIcon={<DownloadIcon />}
                      onClick={() => handleDownloadOnly(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Download
                    </Button>
                  ) : isDownloaded && !isAvailable ? (
                    <Button
                      fullWidth
                      variant="outlined"
                      startIcon={<CloudDownloadIcon />}
                      onClick={() => handleDownloadOnly(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Download Required Files
                    </Button>
                  ) : isResetting ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="warning"
                      startIcon={<CircularProgress size={20} />}
                      disabled
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Resetting...
                    </Button>
                  ) : isFailed ? (
                    <Button
                      fullWidth
                      variant="contained"
                      color="warning"
                      startIcon={<WarningIcon />}
                      onClick={() => handleResetFailedModel(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Failed - Reset
                    </Button>
                  ) : (
                    <Button
                      fullWidth
                      variant="contained"
                      color="success"
                      startIcon={<PlayArrowIcon />}
                      onClick={() => handleDeployModel(guide)}
                      data-tour={index === 0 ? 'deploy-button' : undefined}
                    >
                      Deploy Model
                    </Button>
                  )}
                </CardActions>
              </ModelCard>
            </Grid>
          );
        })}
      </Grid>

      {/* Variant Selection Dialog */}
      <Dialog
        open={variantDialog}
        onClose={() => setVariantDialog(false)}
        maxWidth="md"
        fullWidth
        disablePortal={true}
        aria-labelledby="variant-dialog-title"
      >
        <DialogTitle id="variant-dialog-title">
          Select Model Variant for {selectedGuideForVariant?.name}
        </DialogTitle>
        <DialogContent>
          <Typography variant="body2" sx={{ mb: 2, color: 'text.secondary' }}>
            Choose the model variant that best fits your hardware and requirements:
          </Typography>

          {selectedGuideForVariant && (
            <TableContainer component={Paper} sx={{ mt: 2 }}>
              <Table>
                <TableHead>
                  <TableRow>
                    <TableCell>Type</TableCell>
                    <TableCell>Platform</TableCell>
                    <TableCell>Min VRAM</TableCell>
                    <TableCell>Rec VRAM</TableCell>
                    <TableCell>Speed</TableCell>
                    <TableCell>Action</TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {getValidVariants(selectedGuideForVariant).map((variant, index) => {
                    const isRecommended = JSON.stringify(variant) === JSON.stringify(getRelevantVariant(selectedGuideForVariant));
                    const platforms = Array.isArray(variant.platform) ? variant.platform : [variant.platform];
                    
                    // For Apple Silicon systems, don't show CPU platforms when Apple Silicon platforms exist
                    let displayPlatforms = platforms;
                    if (clusterCapabilities?.system_type === 'apple') {
                      const hasAppleSiliconPlatform = platforms.some(p => 
                        p.includes('Mac (Apple Silicon)') || p.includes('Apple Silicon')
                      );
                      if (hasAppleSiliconPlatform) {
                        displayPlatforms = platforms.filter(p => p !== 'Fast CPU' && p !== 'CPU');
                      }
                    }
                    
                    return (
                      <TableRow
                        key={index}
                        sx={
                          isRecommended
                            ? {
                                position: 'relative',
                                bgcolor: 'rgba(0, 192, 127, 0.08)',
                                border: '2px solid',
                                borderColor: 'primary.main',
                                '& td': { borderBottom: 'none' },
                              }
                            : {}
                        }
                      >
                        <TableCell sx={{ position: 'relative' }}>
                          {variant.variant_type.toUpperCase()}
                          {isRecommended && (
                            <Box
                              sx={{
                                position: 'absolute',
                                top: -2,
                                right: -2,
                                bgcolor: 'primary.main',
                                color: 'white',
                                px: 1,
                                py: 0.5,
                                borderRadius: '0 0 0 8px',
                                fontSize: '0.75rem',
                                fontWeight: 500,
                                lineHeight: 1,
                              }}
                            >
                              Recommended
                            </Box>
                          )}
                        </TableCell>
                        <TableCell>
                          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 0.5 }}>
                            {displayPlatforms.map((platform, platformIndex) => (
                              <Chip
                                key={platformIndex}
                                label={platform}
                                size="small"
                                variant="outlined"
                                sx={{
                                  height: '20px',
                                  fontSize: '0.75rem',
                                  borderRadius: '10px',
                                }}
                              />
                            ))}
                          </Box>
                        </TableCell>
                        <TableCell>{variant.minimum_vram}GB</TableCell>
                        <TableCell>{variant.recommended_vram}GB</TableCell>
                        <TableCell>{getSizeSpeedLabel(variant.speed_rating).label}</TableCell>
                        <TableCell>
                          <Button
                            size="small"
                            variant={isRecommended ? 'contained' : 'outlined'}
                            color={isRecommended ? 'primary' : 'inherit'}
                            onClick={() => handleVariantSelection(selectedGuideForVariant, variant)}
                          >
                            {isRecommended ? 'Use This' : 'Select'}
                          </Button>
                        </TableCell>
                      </TableRow>
                    );
                  })}
                </TableBody>
              </Table>
            </TableContainer>
          )}
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setVariantDialog(false)}>Cancel</Button>
        </DialogActions>
      </Dialog>

      {/* Context Selection Dialog */}
      {/* Debug console output removed */}
      <Dialog
        open={contextDialog}
        onClose={() => setContextDialog(false)}
        maxWidth="sm"
        fullWidth
        disablePortal={true}
        aria-labelledby="context-dialog-title"
      >
        <DialogTitle id="context-dialog-title">
          Select Context Size for {selectedGuideForContext?.name}
        </DialogTitle>
        <DialogContent>
          <Typography variant="body2" sx={{ mb: 2, color: 'text.secondary' }}>
            Choose the context size (memory) for this model. Larger context uses more VRAM but allows the model to remember more of your conversation.
          </Typography>
          
          {selectedGuideForContext && (
            <Box sx={{ mt: 2 }}>
              <Typography variant="body2" sx={{ mb: 2, fontWeight: 500 }}>
                Available options (maximum: {selectedGuideForContext.context_length}):
              </Typography>
              
              <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1 }}>
                {getContextOptions((() => {
                  const contextStr = selectedGuideForContext.context_length;
                  // Parse context like "32k" to 32768, "128k" to 131072, etc.
                  const match = contextStr.match(/(\d+)([kK]?)/);
                  if (match) {
                    const num = parseInt(match[1]);
                    const multiplier = match[2].toLowerCase() === 'k' ? 1024 : 1;
                    const maxContext = num * multiplier;
                    return maxContext;
                  }
                  console.warn('Failed to parse context:', contextStr);
                  return 8192; // Default fallback
                })()).map((size) => {
                  const isSelected = getSelectedContextSize(selectedGuideForContext) === size;
                  return (
                    <Button
                      key={size}
                      variant={isSelected ? "contained" : "outlined"}
                      size="small"
                      onClick={() => handleContextSelection(size)}
                      sx={{
                        minWidth: '80px',
                        backgroundColor: isSelected ? 'primary.main' : 'transparent',
                        '&:hover': {
                          backgroundColor: isSelected ? 'primary.dark' : 'primary.light',
                        }
                      }}
                    >
                      {formatContextSize(size)}
                    </Button>
                  );
                })}
              </Box>
              
              <Alert severity="info" sx={{ mt: 2 }}>
                <Typography variant="caption">
                  <strong>Recommendation:</strong> Start with 8k-16k context unless you need to work with very long documents. 
                  Higher context sizes use significantly more VRAM.
                </Typography>
              </Alert>
              
              {/* Extended explanation tooltip content */}
              <Box sx={{ mt: 3, p: 2, bgcolor: 'background.default', borderRadius: 1 }}>
                <Typography variant="subtitle2" sx={{ fontWeight: 600, mb: 1, color: 'primary.main' }}>
                  Understanding Context Length
                </Typography>
                <Typography variant="body2" sx={{ mb: 2, color: 'text.secondary' }}>
                  Context determines how much text the model can &quot;remember&quot; in a single conversation.
                </Typography>
                
                <Typography variant="body2" sx={{ fontWeight: 500, mb: 1 }}>
                  Why it matters:
                </Typography>
                <Box sx={{ mb: 2, pl: 2 }}>
                  <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block', mb: 0.5 }}>
                    • Longer context = model remembers more of your conversation
                  </Typography>
                  <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block', mb: 0.5 }}>
                    • Can work with larger documents and maintain coherence
                  </Typography>
                  <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block' }}>
                    • Enables multi-turn conversations with better continuity
                  </Typography>
                </Box>
                
                <Typography variant="body2" sx={{ fontWeight: 500, mb: 1 }}>
                  VRAM Usage Impact:
                </Typography>
                <Typography variant="caption" sx={{ color: 'text.secondary', display: 'block', pl: 2, mb: 2 }}>
                  Longer context uses significantly more VRAM. A 32k context may use 2-4x more memory than 8k context.
                </Typography>
                
                <Typography variant="caption" sx={{ color: 'warning.main', fontStyle: 'italic' }}>
                  💡 Tip: Start with shorter context (8k-16k) unless you specifically need to work with very long documents.
                </Typography>
              </Box>
            </Box>
          )}
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setContextDialog(false)}>Close</Button>
        </DialogActions>
      </Dialog>

      {/* Deploy Confirmation Dialog */}
      <Dialog
        open={deployDialog}
        onClose={() => setDeployDialog(false)}
        disablePortal={true}
        aria-labelledby="deploy-dialog-title"
        aria-describedby="deploy-dialog-description"
      >
        <DialogTitle id="deploy-dialog-title">Deploy {selectedModel?.name}?</DialogTitle>
        <DialogContent>
          <Typography id="deploy-dialog-description" variant="body1" gutterBottom>
            This will automatically configure and deploy the model with optimal settings for your
            hardware.
          </Typography>
          <Typography variant="body2" sx={{ color: 'text.secondary', mt: 2 }}>
            The model will be available for use through the API and chat interfaces once deployment
            is complete.
          </Typography>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeployDialog(false)}>Cancel</Button>
          <Button onClick={confirmDeploy} variant="contained" color="primary">
            Deploy Now
          </Button>
        </DialogActions>
      </Dialog>
    </ContentContainer>
  );
};

export default NoviceModelView;
