import React, { useState, useEffect, useContext } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  IconButton,
  Tooltip,
  Checkbox,
  Link,
} from '@mui/material';
import { Link as RouterLink, useNavigate } from 'react-router-dom';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import CloseIcon from '@mui/icons-material/Close';
import DownloadIcon from '@mui/icons-material/Download';
import DeleteIcon from '@mui/icons-material/Delete';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import axios from 'axios';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import DownloadModal from './DownloadModal';
import ModelFilesContext from './ModelFilesContext';
import { BASE_URL } from '../../const';
import { buildModelRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';
import Modal from 'react-modal';
import ConfirmModal from '../common/ConfirmModal';
import DeployModal from './DeployModal';
import ModelDeploymentDetailModal from './ModelDeploymentDetail';

Modal.setAppElement('#root');

// Utility functions
const formatBytes = (bytes, decimals = 2) => {
  if (bytes === 0) return '0 B';
  if (!bytes) return '-';

  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];

  const i = Math.floor(Math.log(bytes) / Math.log(k));

  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
};

const getModelType = (files, model) => {
  // Check if model has library tag from HuggingFace
  if (model && model.library) {
    // Map common library names to display names
    const libraryMap = {
      pytorch: 'PyTorch',
      tensorflow: 'TensorFlow',
      gguf: 'GGUF',
      mlx: 'MLX',
      safetensors: 'SafeTensors',
      transformers: 'Transformers',
    };
    return libraryMap[model.library.toLowerCase()] || model.library;
  }

  // Check if model has tags that indicate type
  if (model && model.tags && Array.isArray(model.tags)) {
    const typeIndicatingTags = ['pytorch', 'tensorflow', 'gguf', 'mlx', 'safetensors'];
    const foundTag = model.tags.find(tag => typeIndicatingTags.includes(tag.toLowerCase()));
    if (foundTag) {
      return foundTag.charAt(0).toUpperCase() + foundTag.slice(1);
    }
  }

  // Fall back to file extension detection
  if (!files || files.length === 0) return 'Unknown';

  const types = new Set();

  files.forEach(file => {
    if (file.name) {
      if (file.name.endsWith('.gguf')) {
        types.add('GGUF');
      } else if (file.name.endsWith('.mlx')) {
        types.add('MLX');
      } else if (file.name.endsWith('.safetensors')) {
        types.add('SafeTensors');
      } else if (file.name.endsWith('.bin') || file.name.endsWith('.pth')) {
        types.add('PyTorch');
      }
    }
  });

  return types.size > 0 ? Array.from(types).join(', ') : 'Unknown';
};

const calculateTotalSize = files => {
  if (!files || files.length === 0) return 0;

  return files.reduce((total, file) => {
    // Only count files that have been downloaded (have storage_location)
    if (file.storage_location && file.size) {
      return total + (parseInt(file.size) || 0);
    }
    return total;
  }, 0);
};

const parseExternalEndpointConfig = rawConfig => {
  if (!rawConfig) return null;
  if (typeof rawConfig === 'object') return rawConfig;
  if (typeof rawConfig === 'string') {
    try {
      return JSON.parse(rawConfig);
    } catch (err) {
      console.warn('Unable to parse external endpoint config:', err);
      return null;
    }
  }
  return null;
};

const getExternalModelMetadata = model => {
  if (!model) return { externalConfig: null, isExternal: false };

  const defaultConfig = model.default_config?.config || model.default_config;
  const rawExternal = defaultConfig?.external_endpoint;
  const externalConfig = parseExternalEndpointConfig(rawExternal);

  const isExternal = Boolean(externalConfig) ||
    model.hub === 'external' ||
    model.purpose === 'external' ||
    model.modelfamily === 'external';

  return { externalConfig, isExternal };
};

const getExternalServiceLabel = service => {
  if (!service) return 'External endpoint';
  const serviceMap = {
    aws_bedrock: 'AWS Bedrock',
  };
  return serviceMap[service] || service;
};

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 600,
  marginTop: theme.spacing(4),
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflowX: 'auto',
  width: '100%',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
  minWidth: '800px',
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 2),
  whiteSpace: 'nowrap',
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
  whiteSpace: 'nowrap',
  overflow: 'hidden',
  textOverflow: 'ellipsis',
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  '&.empty-row': {
    height: 100,
    '& td': {
      textAlign: 'center',
      color: theme.palette.text.secondary,
    },
  },
}));

const ActionButton = styled(Button)(({ theme, color = 'primary' }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.8, 2),
  borderRadius: theme.shape.borderRadius,
  transition: 'all 0.2s',
  backgroundColor: color === 'primary' ? theme.palette.primary.main : theme.palette.error.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: color === 'primary' ? theme.palette.primary.dark : theme.palette.error.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
  '&:disabled': {
    backgroundColor: theme.palette.action.disabledBackground,
    color: theme.palette.action.disabled,
    cursor: 'not-allowed',
    transform: 'none',
    boxShadow: 'none',
  },
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const ModelLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  transition: 'all 0.2s',
  display: 'inline-block',
  borderBottom: '1px solid transparent',
  '&:hover': {
    color: theme.palette.primary.light,
    borderBottomColor: theme.palette.primary.light,
    transform: 'translateY(-1px)',
  },
}));

const ExternalLinkIcon = styled('span')(({ theme }) => ({
  display: 'inline-block',
  width: '12px',
  height: '12px',
  marginLeft: '4px',
  backgroundImage: `url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" fill="%2300c07f" viewBox="0 0 24 24"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path><polyline points="15 3 21 3 21 9"></polyline><line x1="10" y1="14" x2="21" y2="3"></line></svg>')`,
  backgroundSize: 'contain',
  backgroundRepeat: 'no-repeat',
}));

const StyledCloseButton = styled(IconButton)(({ theme }) => ({
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  padding: theme.spacing(0.5),
  '&:hover': {
    color: theme.palette.primary.main,
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
}));

const ModelList = ({
  models: propModels,
  search = false,
  clearSearchResults,
  setModelFiles,
  selectedModels = [],
  onSelectionChange,
  refreshTrigger = 0,
}) => {
  const [models, setModels] = useState(propModels || []);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedModel, setSelectedModel] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [confirmDelete, setConfirmDelete] = useState(false);
  const [deploymentRuntimes, setDeploymentRuntimes] = useState({});
  const [showDeployModal, setShowDeployModal] = useState(false);
  const [deployModel, setDeployModel] = useState(null);
  const [selectedDeploymentId, setSelectedDeploymentId] = useState(null);
  const contextValue = useContext(ModelFilesContext);
  const { modelListRefreshTrigger, statusModelFiles = [] } = contextValue || {};
  const navigate = useNavigate();
  const { routingMode } = useRoutingConfig();

  // Check if a model has active downloads
  const isModelDownloading = modelId => {
    if (!statusModelFiles || statusModelFiles.length === 0) return false;

    return statusModelFiles.some(
      file =>
        file.m_id === modelId &&
        (file.is_downloading || (file.dl_requested_at && !file.storage_location))
    );
  };

  // Fetch deployment counts for models
  const fetchDeploymentRuntimes = async () => {
    try {
      const response = await axios.get(`${BASE_URL}/serving/deployments`);
      console.log('Deployments response:', response.data);
      if (response.status === 200 && response.data) {
        // Collect runtime info by model ID, skipping entries without available routing info
        const runtimesByModel = {};
        response.data.forEach(deployment => {
          console.log('Processing deployment:', {
            id: deployment.id,
            m_id: deployment.m_id,
            status: deployment.status,
            lb_port: deployment.lb_port,
            serve_path: deployment.serve_path,
          });

          const runtimeInfo = buildModelRuntimeInfo(deployment, routingMode);
          const hasRuntime = Boolean(runtimeInfo.baseUrl);

          // Check if deployment has model info, is deployed or initializing, and exposes a runtime endpoint
          if (
            deployment.m_id &&
            (deployment.status === 'DEPLOYED' || deployment.status === 'INITIALIZING') &&
            hasRuntime
          ) {
            if (!runtimesByModel[deployment.m_id]) {
              runtimesByModel[deployment.m_id] = [];
            }
            runtimesByModel[deployment.m_id].push({
              id: deployment.id,
              runtimeInfo,
            });
          }
        });
        console.log('Runtimes by model:', runtimesByModel);
        setDeploymentRuntimes(runtimesByModel);
      }
    } catch (err) {
      console.error('Error fetching deployment ports:', err);
      // Don't set error state as this is a non-critical enhancement
    }
  };

  useEffect(() => {
    const fetchModels = async () => {
      setLoading(true);
      try {
        const uri = `${BASE_URL}/models/?load_files=true`;
        const response = await axios.get(uri);
        if (response.status === 200) {
          setModels(response.data);
        } else {
          throw new Error(`Failed to fetch models from ${uri}`);
        }
      } catch (err) {
        console.error('Error fetching models:', err);
        setError(err);
      } finally {
        setLoading(false);
      }
    };

    if (!propModels) {
      fetchModels();
    } else {
      setModels(propModels);
    }

    // Fetch deployment counts if not in search mode
    if (!search) {
      fetchDeploymentRuntimes();
    }
  }, [propModels, modelListRefreshTrigger, search, refreshTrigger]);

  const handleDownloadClick = async model => {
    setLoading(true);
    try {
      const response = await axios.post(`${BASE_URL}/model_files/search/`, {
        hub: model.hub,
        model: model.repo_modelId,
        version: model.version || '',
      });
      setSelectedModel({ ...model, files: response.data });
      setShowModal(true);
    } catch (err) {
      console.error('Error fetching files:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = model => {
    setSelectedModel(model);
    setConfirmDelete(true);
  };

  const handleConfirmDelete = async () => {
    setLoading(true);
    try {
      const response = await axios.delete(`${BASE_URL}/models/${selectedModel.id}`);
      if (response.status === 200) {
        setModels(models.filter(model => model.id !== selectedModel.id));
        setConfirmDelete(false);
        // Refresh deployment ports after successful deletion
    if (!search) {
      fetchDeploymentRuntimes();
    }
      } else {
        throw new Error(`Failed to delete model with id ${selectedModel.id}`);
      }
    } catch (err) {
      console.error('Error deleting model:', err);
      // Check if it's a 405 error (Method Not Allowed)
      if (err.response && err.response.status === 405) {
        // Extract the error message from the response
        const errorMessage =
          err.response.data?.detail ||
          'Cannot delete model with active deployments. Please stop all deployments first.';
        setError({ message: errorMessage });
      } else {
        setError(err);
      }
      setConfirmDelete(false); // Close the confirm dialog on error
    } finally {
      setLoading(false);
    }
  };


  const handleDeployClick = model => {
    setDeployModel(model);
    setShowDeployModal(true);
  };

  const handleDeployModalClose = () => {
    setShowDeployModal(false);
    setDeployModel(null);
  };

  const handleDeployWithConfig = async (model, config) => {
    try {
      console.log('Deploying model with config:', model.name, config?.name);

      // Create payload for deployment
      const deployPayload = {
        m_id: model.id,
        m_config_id: config.id,
      };

      let externalEndpointConfig = null;
      const rawExternalEndpoint = config?.config?.external_endpoint;
      if (rawExternalEndpoint) {
        if (typeof rawExternalEndpoint === 'string') {
          try {
            externalEndpointConfig = JSON.parse(rawExternalEndpoint);
          } catch (parseError) {
            console.warn('Unable to parse external endpoint config:', parseError);
          }
        } else if (typeof rawExternalEndpoint === 'object') {
          externalEndpointConfig = rawExternalEndpoint;
        }
      }

      const externalService = externalEndpointConfig?.service;
      const configEngineName = config?.config?.engine_name;
      if (externalService && externalService === 'aws_bedrock') {
        deployPayload.engine_name = 'aws_bedrock';
      } else if (configEngineName) {
        deployPayload.engine_name = configEngineName;
      }

      // Deploy the model with the config
      const deployResponse = await axios.post(`${BASE_URL}/serving/deploy_model`, deployPayload);
      console.log('Deployment response:', deployResponse.data);

      // Show success message (assuming there's a toast or notification system)
      // For now, just close the modal and refresh
      setShowDeployModal(false);

      // Refresh deployment counts after a short delay
      setTimeout(() => {
            fetchDeploymentRuntimes();
      }, 2000);
    } catch (error) {
      console.error('Error deploying model:', error);
      // TODO: Show error message to user
      alert(`Failed to deploy model: ${error.response?.data?.detail || error.message}`);
    }
  };

  const handleAddConfigFromDeploy = model => {
    navigate(`/models/${model.id}`);
  };

  const handleSelectAll = event => {
    if (!onSelectionChange) return;

    if (event.target.checked) {
      const allModelIds = models.filter(model => model.id).map(model => model.id);
      onSelectionChange(allModelIds);
    } else {
      onSelectionChange([]);
    }
  };

  const handleSelectModel = modelId => {
    if (!onSelectionChange) return;

    const newSelection = selectedModels.includes(modelId)
      ? selectedModels.filter(id => id !== modelId)
      : [...selectedModels, modelId];

    onSelectionChange(newSelection);
  };

  const isAllSelected =
    models.length > 0 &&
    models.filter(model => model.id).every(model => selectedModels.includes(model.id));

  return (
    <Box sx={{ mb: 4 }}>
      <Box sx={{ display: 'flex', alignItems: 'center' }}>
        <SubTitle variant="h5">
          {!search ? 'Your Models' : 'Model Search Results'}
          <Tooltip title="Models you have downloaded or uploaded to your Kamiwaza instance">
            <InfoIcon />
          </Tooltip>
        </SubTitle>
        {search && models.length > 0 && (
          <StyledCloseButton onClick={clearSearchResults} aria-label="clear results">
            <CloseIcon />
          </StyledCloseButton>
        )}
      </Box>

      {loading && <Spinner />}
      {error && <ErrorComponent message={error?.message || 'An error occurred'} />}

      {models.length === 0 && !loading ? (
        <Typography sx={{ ml: 2, color: 'text.secondary' }}>No models available.</Typography>
      ) : (
        <StyledTableContainer component={Paper}>
          <StyledTable>
            <TableHead>
              <TableRow>
                {!search && onSelectionChange && (
                  <TableHeaderCell padding="checkbox">
                    <Checkbox
                      checked={isAllSelected}
                      indeterminate={
                        selectedModels.length > 0 &&
                        selectedModels.length < models.filter(m => m.id).length
                      }
                      onChange={handleSelectAll}
                      sx={{ color: 'primary.main' }}
                    />
                  </TableHeaderCell>
                )}
                <TableHeaderCell>Model Name</TableHeaderCell>
                <TableHeaderCell>Provider</TableHeaderCell>
                <TableHeaderCell>Size</TableHeaderCell>
                <TableHeaderCell>Type</TableHeaderCell>
                {!search && <TableHeaderCell>Deployment IDs</TableHeaderCell>}
                <TableHeaderCell>Actions</TableHeaderCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {models.map(model => {
                const { externalConfig, isExternal } = getExternalModelMetadata(model);
                const totalSize = calculateTotalSize(model.m_files);
                const modelType = getModelType(model.m_files, model);
                const displayModelType = isExternal ? 'External Endpoint' : modelType;
                const displaySize = isExternal ? '—' : formatBytes(totalSize);
                const isDownloading = isModelDownloading(model.id);

                return (
                  <StyledTableRow key={model.id || model.repo_modelId}>
                    {!search && onSelectionChange && (
                      <TableBodyCell padding="checkbox">
                        <Checkbox
                          checked={model.id ? selectedModels.includes(model.id) : false}
                          onChange={() => model.id && handleSelectModel(model.id)}
                          disabled={!model.id}
                          sx={{ color: 'primary.main' }}
                        />
                      </TableBodyCell>
                    )}
                    <TableBodyCell>
                      {model.id ? (
                        <ModelLink
                          component={RouterLink}
                          to={`/models/${model.id}`}
                          title={model.name}
                        >
                          {model.name?.substring(0, 60) || 'Unnamed Model'}
                        </ModelLink>
                      ) : (
                        <span title={model.repo_modelId}>
                          {model.repo_modelId?.substring(0, 60) || 'Unknown Model'}
                        </span>
                      )}
                      {model.source_repository && !isExternal && (
                        <a
                          href={model.source_repository}
                          target="_blank"
                          rel="noopener noreferrer"
                          style={{
                            marginLeft: '8px',
                            color: '#00c07f',
                            textDecoration: 'none',
                          }}
                        >
                          <ExternalLinkIcon />
                        </a>
                      )}
                    </TableBodyCell>
                    <TableBodyCell>
                      {(() => {
                        if (isExternal) {
                          return getExternalServiceLabel(externalConfig?.service);
                        }
                        const provider = model.repo_modelId?.split('/')[0];
                        const hub = model.hub === 'HubsHf' ? 'huggingface.co' : model.hub;
                        if (provider && hub === 'huggingface.co') {
                          return (
                            <Tooltip title="View provider's HuggingFace profile">
                              <a
                                href={`https://${hub}/${provider}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                style={{
                                  color: '#00c07f',
                                  textDecoration: 'none',
                                }}
                              >
                                {provider}
                              </a>
                            </Tooltip>
                          );
                        }
                        return provider || '-';
                      })()}
                    </TableBodyCell>
                    <TableBodyCell>{displaySize}</TableBodyCell>
                    <TableBodyCell>{displayModelType}</TableBodyCell>
                    {!search && (
                      <TableBodyCell>
                        {deploymentRuntimes[model.id] ? (
                          <Box sx={{ display: 'flex', gap: 0.5, flexWrap: 'wrap' }}>
                            {deploymentRuntimes[model.id].map((deployment, index) => (
                              <React.Fragment key={deployment.id}>
                                <Link
                                  component="button"
                                  onClick={() => setSelectedDeploymentId(deployment.id)}
                                  sx={{
                                    color: 'primary.main',
                                    textDecoration: 'none',
                                    cursor: 'pointer',
                                    background: 'none',
                                    border: 'none',
                                    padding: 0,
                                    font: 'inherit',
                                    '&:hover': {
                                      textDecoration: 'underline',
                                    },
                                  }}
                                >
                                  {deployment.id}
                                </Link>
                                {index < deploymentRuntimes[model.id].length - 1 && ','}
                              </React.Fragment>
                            ))}
                          </Box>
                        ) : (
                          <Typography variant="body2" color="text.secondary">
                            -
                          </Typography>
                        )}
                      </TableBodyCell>
                    )}
                    <TableBodyCell>
                      <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                        {!isExternal && (
                          <Tooltip
                            title={
                              isDownloading
                                ? 'Download in progress'
                                : search
                                  ? 'Select model files to download'
                                  : 'Re-download model files'
                            }
                          >
                            <span>
                              <ActionButton
                                size="small"
                                startIcon={<DownloadIcon />}
                                onClick={() => handleDownloadClick(model)}
                                sx={{ minWidth: 'auto', px: 1.5 }}
                                disabled={isDownloading}
                              >
                                {search ? 'Select' : 'Re-download'}
                              </ActionButton>
                            </span>
                          </Tooltip>
                        )}
                        {!search && model.id && (
                          <>
                            <Tooltip
                              title={
                                isDownloading
                                  ? 'Cannot deploy while downloading'
                                  : 'Deploy this model'
                              }
                            >
                              <span>
                                <ActionButton
                                  size="small"
                                  startIcon={<RocketLaunchIcon />}
                                  onClick={() => handleDeployClick(model)}
                                  sx={{ minWidth: 'auto', px: 1.5 }}
                                  disabled={isDownloading}
                                >
                                  Deploy
                                </ActionButton>
                              </span>
                            </Tooltip>
                            <Tooltip
                              title={
                                deploymentRuntimes[model.id] && deploymentRuntimes[model.id].length > 0
                                  ? 'Cannot delete model with active deployments. Stop all deployments first.'
                                : 'Delete this model'
                              }
                            >
                              <span>
                                <ActionButton
                                  size="small"
                                  color="error"
                                  startIcon={<DeleteIcon />}
                                  onClick={() => handleDeleteClick(model)}
                                  sx={{ minWidth: 'auto', px: 1.5 }}
                                    disabled={
                                    deploymentRuntimes[model.id] &&
                                    deploymentRuntimes[model.id].length > 0
                                  }
                                >
                                  Delete
                                </ActionButton>
                              </span>
                            </Tooltip>
                          </>
                        )}
                      </Box>
                    </TableBodyCell>
                  </StyledTableRow>
                );
              })}
            </TableBody>
          </StyledTable>
        </StyledTableContainer>
      )}

      {showModal && selectedModel && (
        <DownloadModal
          model={selectedModel}
          closeModal={() => setShowModal(false)}
          showModal={showModal}
          setModelFiles={setModelFiles}
        />
      )}

      {confirmDelete && selectedModel && (
        <ConfirmModal
          title="Confirm Delete"
          message="DELETING THIS MODEL WILL REMOVE ALL IT'S FILES. Do you want to delete it and all it's files?"
          showModal={confirmDelete}
          onConfirm={handleConfirmDelete}
          onCancel={() => setConfirmDelete(false)}
        />
      )}

      <DeployModal
        open={showDeployModal}
        onClose={handleDeployModalClose}
        model={deployModel}
        onDeploy={handleDeployWithConfig}
        onAddConfig={handleAddConfigFromDeploy}
      />

      {selectedDeploymentId && (
        <ModelDeploymentDetailModal
          deploymentId={selectedDeploymentId}
          isOpen={!!selectedDeploymentId}
          onClose={() => setSelectedDeploymentId(null)}
          onDeploymentStopped={() => {
            setSelectedDeploymentId(null);
            fetchDeploymentRuntimes(); // Refresh runtime data after stopping
          }}
        />
      )}
    </Box>
  );
};

ModelList.propTypes = {
  models: PropTypes.arrayOf(PropTypes.object),
  search: PropTypes.bool,
  clearSearchResults: PropTypes.func,
  setModelFiles: PropTypes.func,
  selectedModels: PropTypes.arrayOf(PropTypes.string),
  onSelectionChange: PropTypes.func,
  refreshTrigger: PropTypes.number,
};

export default ModelList;
