// ModelIndex.js
import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import { Box, Typography, Tab, Tabs, Paper, Button, ButtonGroup } from '@mui/material';
import AddIcon from '@mui/icons-material/Add';
import DeleteIcon from '@mui/icons-material/Delete';
import axios from 'axios';
import ModelList from './ModelList';
import ModelDeploymentList from './ModelDeploymentList';
import NoviceModelView from './NoviceModelView';
import AddModelModal from './AddModelModal';
import ConfirmModal from '../common/ConfirmModal';
import { BASE_URL } from '../../const';

// Styled components for visual enhancements
const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem', // Reduced from 1.8rem
  fontWeight: 700,
  marginBottom: theme.spacing(1.5), // Reduced spacing
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px', // Reduced width
    height: '2px', // Reduced height
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const TabsContainer = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: '12px 12px 0 0',
  boxShadow: 'none',
  borderBottom: '1px solid rgba(255, 255, 255, 0.08)',
  marginBottom: theme.spacing(3),
  marginTop: theme.spacing(3),
}));

const StyledTabs = styled(Tabs)(({ theme }) => ({
  '& .MuiTabs-indicator': {
    backgroundColor: theme.palette.primary.main,
    height: 3,
  },
}));

const StyledTab = styled(Tab)(({ theme }) => ({
  textTransform: 'none',
  fontSize: '0.875rem', // Reduced from 1rem
  fontWeight: 500,
  padding: theme.spacing(1.5, 3), // Reduced from 2, 4
  color: theme.palette.text.secondary,
  '&.Mui-selected': {
    color: theme.palette.primary.main,
    fontWeight: 600,
  },
  '&:hover': {
    color: theme.palette.text.primary,
    opacity: 1,
  },
}));

const TabPanel = styled(Box)(({ theme }) => ({
  padding: theme.spacing(3, 0),
}));

const ControlsContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  marginBottom: theme.spacing(3),
  flexWrap: 'wrap',
  gap: theme.spacing(2),
}));

const ActionButton = styled(Button)(({ theme }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.75, 1.75), // Reduced from 1, 2
  borderRadius: theme.shape.borderRadius,
  fontSize: '0.875rem', // Set explicit size
  transition: 'all 0.2s',
  '&:hover': {
    transform: 'translateY(-1px)',
    boxShadow: '0 2px 4px rgba(0, 0, 0, 0.15)',
  },
}));

function TabPanelContent(props) {
  const { children, value, index, ...other } = props;

  return (
    <TabPanel
      role="tabpanel"
      hidden={value !== index}
      id={`models-tabpanel-${index}`}
      aria-labelledby={`models-tab-${index}`}
      {...other}
    >
      {value === index && children}
    </TabPanel>
  );
}

TabPanelContent.propTypes = {
  children: PropTypes.node,
  value: PropTypes.number.isRequired,
  index: PropTypes.number.isRequired,
};

const ModelIndex = () => {
  const [advancedMode, setAdvancedMode] = useState(false);
  const [tabValue, setTabValue] = useState(0);
  const [selectedModels, setSelectedModels] = useState([]);
  const [selectedDeployments, setSelectedDeployments] = useState([]);
  const [showAddModelModal, setShowAddModelModal] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  useEffect(() => {
    // Check if advanced mode is enabled in localStorage
    const savedMode = localStorage.getItem('advancedDeveloperMode');
    setAdvancedMode(savedMode === 'true');
  }, []);

  // Show novice view if not in advanced mode
  if (!advancedMode) {
    return <NoviceModelView />;
  }

  const handleTabChange = (event, newValue) => {
    setTabValue(newValue);
  };

  const handleAddModel = () => {
    setShowAddModelModal(true);
  };

  const handleDeleteModels = () => {
    setShowDeleteConfirm(true);
  };

  const handleConfirmDelete = async () => {
    setIsDeleting(true);
    try {
      // Delete each selected model
      const deletePromises = selectedModels.map(modelId =>
        axios.delete(`${BASE_URL}/models/${modelId}`)
      );

      await Promise.all(deletePromises);

      // Clear selection and refresh the list
      setSelectedModels([]);
      setRefreshTrigger(prev => prev + 1);
      setShowDeleteConfirm(false);
    } catch (error) {
      console.error('Error deleting models:', error);
      alert('Failed to delete some models. Please try again.');
    } finally {
      setIsDeleting(false);
    }
  };

  const handleUndeployModels = async () => {
    if (selectedDeployments.length === 0) return;

    const confirmed = window.confirm(
      `Are you sure you want to stop ${selectedDeployments.length} deployment${selectedDeployments.length > 1 ? 's' : ''}?`
    );
    if (!confirmed) return;

    try {
      // Stop each selected deployment
      const stopPromises = selectedDeployments.map(deploymentId =>
        axios.delete(`${BASE_URL}/serving/deployment/${deploymentId}`, { params: { force: true } })
      );

      await Promise.all(stopPromises);

      // Clear selection and refresh the list
      setSelectedDeployments([]);
      setRefreshTrigger(prev => prev + 1);
    } catch (error) {
      console.error('Error stopping deployments:', error);
      alert('Failed to stop some deployments. Please try again.');
    }
  };

  // Show advanced view with tabs
  return (
    <ContentContainer>
      <SectionTitle variant="h4">Models</SectionTitle>

      <TabsContainer>
        <StyledTabs value={tabValue} onChange={handleTabChange} aria-label="models tabs">
          <StyledTab label="Models" />
          <StyledTab label="Deployments" />
        </StyledTabs>
      </TabsContainer>

      <TabPanelContent value={tabValue} index={0}>
        <ControlsContainer sx={{ justifyContent: 'flex-start' }}>
          <ButtonGroup variant="contained">
            <ActionButton color="primary" startIcon={<AddIcon />} onClick={handleAddModel}>
              Add Model
            </ActionButton>
            <ActionButton
              color="error"
              startIcon={<DeleteIcon />}
              onClick={handleDeleteModels}
              disabled={selectedModels.length === 0}
            >
              Delete Selected
            </ActionButton>
          </ButtonGroup>
        </ControlsContainer>
        <ModelList
          selectedModels={selectedModels}
          onSelectionChange={setSelectedModels}
          refreshTrigger={refreshTrigger}
        />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={1}>
        <ControlsContainer sx={{ justifyContent: 'flex-start' }}>
          <ActionButton
            variant="contained"
            color="error"
            startIcon={<DeleteIcon />}
            onClick={handleUndeployModels}
            disabled={selectedDeployments.length === 0}
          >
            Delete Selected
          </ActionButton>
        </ControlsContainer>
        <ModelDeploymentList
          selectedDeployments={selectedDeployments}
          onSelectionChange={setSelectedDeployments}
          refreshDeployments={refreshTrigger}
        />
      </TabPanelContent>

      <AddModelModal open={showAddModelModal} onClose={() => setShowAddModelModal(false)} />

      <ConfirmModal
        title="Confirm Delete"
        message={`Are you sure you want to delete ${selectedModels.length} selected model${selectedModels.length > 1 ? 's' : ''}? This will remove all associated files and cannot be undone.`}
        showModal={showDeleteConfirm}
        onConfirm={handleConfirmDelete}
        onCancel={() => setShowDeleteConfirm(false)}
        confirmDisabled={isDeleting}
      />
    </ContentContainer>
  );
};

export default ModelIndex;
