import InfoOutlinedIcon from "@mui/icons-material/InfoOutlined";
import SearchIcon from "@mui/icons-material/Search";
import WarningAmberIcon from "@mui/icons-material/WarningAmber";
import {
	Alert,
	Box,
	Button,
	Checkbox,
	FormControlLabel,
	Paper,
	TextField,
	Tooltip,
	Typography,
} from "@mui/material";
import { styled } from "@mui/material/styles";
import axios from "axios";
import React, { useContext, useState } from "react";
import { BASE_URL } from "../../const";
import ErrorComponent from "../common/ErrorComponent";
import Spinner from "../common/Spinner";
import Toast from "../common/Toast";
import ModelFilesContext from "./ModelFilesContext";
import ModelList from "./ModelList";

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
	color: theme.palette.primary.main,
	fontSize: "1.5rem",
	fontWeight: 600,
	marginTop: theme.spacing(4),
	marginBottom: theme.spacing(2),
	paddingBottom: theme.spacing(1),
	position: "relative",
	"&:after": {
		content: '""',
		position: "absolute",
		left: 0,
		bottom: 0,
		width: "60px",
		height: "2px",
		background: "linear-gradient(90deg, #00c07f, transparent)",
	},
}));

const SearchContainer = styled(Box)(({ theme }) => ({
	display: "flex",
	alignItems: "center",
	gap: theme.spacing(2),
	marginBottom: theme.spacing(4),
	flexWrap: "wrap",
}));

const SearchInput = styled(TextField)(({ theme }) => ({
	backgroundColor: "rgba(255, 255, 255, 0.03)",
	borderRadius: theme.shape.borderRadius,
	"& .MuiOutlinedInput-root": {
		"& fieldset": {
			borderColor: "rgba(255, 255, 255, 0.1)",
		},
		"&:hover fieldset": {
			borderColor: theme.palette.primary.main,
		},
		"&.Mui-focused fieldset": {
			borderColor: theme.palette.primary.main,
		},
	},
	"& .MuiInputBase-input": {
		padding: theme.spacing(1.5, 2),
	},
}));

const ActionButton = styled(Button)(({ theme }) => ({
	textTransform: "none",
	fontWeight: 500,
	padding: theme.spacing(0.8, 2),
	borderRadius: theme.shape.borderRadius,
	transition: "all 0.2s",
	backgroundColor: theme.palette.primary.main,
	"&:hover": {
		backgroundColor: theme.palette.primary.dark,
		transform: "translateY(-2px)",
		boxShadow: "0 4px 8px rgba(0, 0, 0, 0.2)",
	},
}));

const StyledCheckbox = styled(FormControlLabel)(({ theme }) => ({
	"& .MuiFormControlLabel-label": {
		fontSize: "0.9rem",
	},
	"& .MuiCheckbox-root": {
		color: theme.palette.primary.main,
		"&.Mui-checked": {
			color: theme.palette.primary.main,
		},
	},
	margin: 0, // Remove default margin
	alignSelf: "center", // Ensure vertical alignment
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
	fontSize: "1.1rem",
	color: theme.palette.text.secondary,
	marginLeft: theme.spacing(1),
	cursor: "pointer",
	transition: "color 0.2s",
	"&:hover": {
		color: theme.palette.primary.main,
	},
}));

const ModelHubSearch = () => {
	const { setModelFiles } = useContext(ModelFilesContext);
	const [query, setQuery] = useState("");
	const [isExact, setIsExact] = useState(false);
	const [results, setResults] = useState([]);
	const [loading, setLoading] = useState(false);
	const [error, setError] = useState(null);
	const [hubError, setHubError] = useState(null);
	const [hasSearched, setHasSearched] = useState(false);
	const [toast, setToast] = useState({
		open: false,
		message: "",
		severity: "info",
	});
	const clearResults = () => {
		setResults([]);
		setHubError(null);
		setHasSearched(false);
	};
	const closeToast = () => setToast({ ...toast, open: false });

	const handleSearch = async (event) => {
		event.preventDefault();
		setLoading(true);
		setHubError(null);
		setHasSearched(true);

		if (query.trim() !== "") {
			try {
				const response = await axios.post(`${BASE_URL}/models/search/`, {
					query,
					exact: isExact,
					hubs_to_search: ["*"],
				});
				// Assuming the API returns an object with a 'results' array
				setResults(response.data.results.map((item) => item.model));

				// Track hub error for persistent display
				if (response.data.hub_error) {
					setHubError(response.data.hub_error);
				}
				setLoading(false);
			} catch (err) {
				console.error("Search error:", err);
				setError(err);
				setLoading(false);
			}
		} else {
			setLoading(false); // Ensure loading is set to false if query is empty
			setHasSearched(false);
		}
	};

	return (
		<Box sx={{ mb: 4 }}>
			<SubTitle variant="h5">
				Model Hub Search
				<Tooltip title="Search for models from Hugging Face and other supported model hubs">
					<InfoIcon />
				</Tooltip>
			</SubTitle>

			<Paper
				elevation={0}
				sx={{
					bgcolor: "rgba(255, 255, 255, 0.03)",
					borderRadius: 2,
					p: 3,
					mb: 3,
				}}
			>
				<form onSubmit={handleSearch}>
					<SearchContainer>
						<SearchInput
							placeholder="Search for models..."
							value={query}
							onChange={(e) => setQuery(e.target.value)}
							fullWidth
							variant="outlined"
							InputProps={{
								startAdornment: (
									<SearchIcon sx={{ mr: 1, color: "text.secondary" }} />
								),
							}}
						/>

						<ActionButton
							type="submit"
							variant="contained"
							startIcon={<SearchIcon />}
							disabled={loading}
						>
							Search
						</ActionButton>

						<StyledCheckbox
							control={
								<Checkbox
									checked={isExact}
									onChange={(e) => setIsExact(e.target.checked)}
								/>
							}
							label="Exact"
						/>
					</SearchContainer>
				</form>
			</Paper>

			{loading && <Spinner />}
			{error && <ErrorComponent message={error.message} />}

			{/* Show hub error warning banner */}
			{hubError && !loading && (
				<Alert
					severity="warning"
					icon={<WarningAmberIcon />}
					sx={{ mb: 2 }}
					onClose={() => setHubError(null)}
				>
					<strong>HuggingFace Hub temporarily unavailable:</strong> {hubError}
					{results.length > 0
						? " Showing local results only."
						: " Please try again later or search for models you've already downloaded."}
				</Alert>
			)}

			{/* Show no results message when searched but nothing found */}
			{hasSearched &&
				!loading &&
				!error &&
				results.length === 0 &&
				!hubError && (
					<Alert severity="info" sx={{ mb: 2 }}>
						No models found matching &quot;{query}&quot;. Try a different search
						term.
					</Alert>
				)}

			{results.length > 0 && !loading && (
				<ModelList
					models={results}
					search={true}
					setModelFiles={setModelFiles}
					clearSearchResults={clearResults}
				/>
			)}

			<Toast
				open={toast.open}
				message={toast.message}
				severity={toast.severity}
				onClose={closeToast}
				autoHideDuration={6000}
			/>
		</Box>
	);
};

export default ModelHubSearch;
