// ModelFilesContext.js
import React from 'react';
import { createContext, useState, useEffect, useCallback, useRef } from 'react';
import axios from 'axios';
import { BASE_URL } from '../../const';

const ModelFilesContext = createContext();

export const ModelFilesProvider = ({ children }) => {
  const [statusModelFiles, setStatusModelFiles] = useState([]);
  const [showDownloadStatus, setShowDownloadStatus] = useState(false);
  const [toast, setToast] = useState({ open: false, message: '', severity: 'info' });
  const [modelListRefreshTrigger, setModelListRefreshTrigger] = useState(0);
  const [isPolling, setIsPolling] = useState(false);
  const [hasActiveDownloads, setHasActiveDownloads] = useState(false);
  const [fileMetadata, setFileMetadata] = useState(new Map()); // Store file metadata separately

  // Show toast notification
  const showToast = useCallback((message, severity = 'info') => {
    setToast({ open: true, message, severity });
  }, []);

  // Close toast notification
  const closeToast = useCallback(() => {
    setToast(prev => ({ ...prev, open: false }));
  }, []);

  // Update statusModelFiles - merge with file metadata to preserve names
  const updateStatusModelFiles = useCallback(
    (newData, isInitialData = false) => {
      // If this is initial data with names, store the metadata
      if (isInitialData) {
        const newMetadata = new Map(fileMetadata);
        newData.forEach(file => {
          if (file.name && file.name !== 'Unnamed file') {
            newMetadata.set(file.id, {
              name: file.name,
              size: file.size,
              m_id: file.m_id,
            });
          }
        });
        setFileMetadata(newMetadata);

        // For initial data, use the names directly from the incoming data
        setStatusModelFiles(
          newData.map(file => ({
            ...file,
            name: file.name || 'Unnamed file',
          }))
        );
      } else {
        // For updates, merge with stored metadata
        setStatusModelFiles(
          newData.map(file => ({
            ...file,
            name: fileMetadata.get(file.id)?.name || file.name || 'Unnamed file',
          }))
        );
      }
    },
    [fileMetadata]
  );

  // Function to fetch download status
  const fetchDownloadStatus = useCallback(async () => {
    try {
      const response = await axios.get(`${BASE_URL}/model_files/download_status/`);
      // Check if any files have names we should capture
      const hasNames = response.data.some(file => file.name && file.name !== 'Unnamed file');
      updateStatusModelFiles(response.data, hasNames);

      // Check if there are any active downloads
      const activeDownloads = response.data.some(
        file => file.is_downloading || (file.dl_requested_at && !file.storage_location)
      );

      setHasActiveDownloads(activeDownloads);
      setShowDownloadStatus(response.data.length > 0);

      return activeDownloads;
    } catch (error) {
      console.error('Error fetching download status:', error);
      const status = error.response?.status;
      if (status === 503) {
        showToast('Service temporarily unavailable. Retrying...', 'warning');
      } else if (status && status !== 401 && status !== 403) {
        showToast('Failed to fetch download status', 'error');
      }
      return false;
    }
  }, [updateStatusModelFiles, showToast]);

  // Initial check on mount
  useEffect(() => {
    fetchDownloadStatus().then(hasActive => {
      if (hasActive) {
        setIsPolling(true);
      }
    });
  }, []); // Empty dependency array - only run once on mount

  // Store fetchDownloadStatus in a ref to avoid effect dependencies
  const fetchDownloadStatusRef = useRef();
  fetchDownloadStatusRef.current = fetchDownloadStatus;

  // Polling effect - only runs when isPolling is true
  useEffect(() => {
    if (!isPolling) return;

    const intervalId = setInterval(async () => {
      const hasActive = await fetchDownloadStatusRef.current();
      if (!hasActive) {
        // No more active downloads, stop polling
        setIsPolling(false);
      }
    }, 2000);

    return () => clearInterval(intervalId);
  }, [isPolling]); // Only depend on isPolling

  // Cancel a specific download
  const cancelDownload = async (fileId, fileName) => {
    try {
      const response = await axios.delete(`${BASE_URL}/model_files/${fileId}/download`);

      if (response.data.result) {
        // Remove the cancelled file from the status list immediately
        setStatusModelFiles(prev => prev.filter(item => item.id !== fileId));

        // Trigger model list refresh to update file states after a short delay
        setTimeout(() => {
          refreshModelList();
        }, 500);

        // Show success toast
        showToast(`Download cancelled for ${fileName}`, 'success');
        console.log(`Download cancelled for ${fileName}`);
        return { success: true, message: response.data.message };
      } else {
        console.error('Failed to cancel download:', response.data.message);
        showToast(`Failed to cancel download: ${response.data.message}`, 'error');
        return { success: false, message: response.data.message };
      }
    } catch (error) {
      console.error('Error cancelling download:', error);
      const errorMessage = error.response?.data?.detail || 'Failed to cancel download';
      showToast(`Error: ${errorMessage}`, 'error');
      return { success: false, message: errorMessage };
    }
  };

  // Cancel all active downloads
  const cancelAllDownloads = async () => {
    try {
      const response = await axios.delete(`${BASE_URL}/model_files/downloads/cancel_all`);

      if (response.data.result) {
        // Remove cancelled files from the status list
        const cancelledFileIds = response.data.cancelled_files || [];
        setStatusModelFiles(prev => prev.filter(item => !cancelledFileIds.includes(item.id)));

        // Trigger model list refresh to update file states after a short delay
        setTimeout(() => {
          refreshModelList();
        }, 500);

        // Show success toast
        const { cancelled_count, failed_count, total_downloads } = response.data;
        if (failed_count === 0) {
          showToast(`Successfully cancelled all ${cancelled_count} downloads`, 'success');
        } else if (cancelled_count === 0) {
          showToast(`Failed to cancel any of the ${total_downloads} downloads`, 'error');
        } else {
          showToast(`Cancelled ${cancelled_count} downloads, ${failed_count} failed`, 'warning');
        }

        console.log(`Bulk cancellation result:`, response.data);
        return { success: true, data: response.data };
      } else {
        console.error('Failed to cancel all downloads:', response.data.message);
        showToast(`Failed to cancel downloads: ${response.data.message}`, 'error');
        return { success: false, message: response.data.message };
      }
    } catch (error) {
      console.error('Error cancelling all downloads:', error);
      const errorMessage = error.response?.data?.detail || 'Failed to cancel all downloads';
      showToast(`Error: ${errorMessage}`, 'error');
      return { success: false, message: errorMessage };
    }
  };

  // Method to clear completed downloads, to be called by user action
  // This method now checks if the download is completed based on the presence of 'storage_location' and absence of 'dl_requested_at'
  const clearCompletedDownloads = () => {
    //setStatusModelFiles(prev => prev.filter(item => !(item.storage_location && !item.dl_requested_at)));
    setStatusModelFiles(prev => prev.filter(item => !!item.dl_requested_at));

    // Also clear metadata for completed files
    setFileMetadata(prev => {
      const newMetadata = new Map(prev);
      statusModelFiles.forEach(file => {
        if (file.storage_location && !file.dl_requested_at) {
          newMetadata.delete(file.id);
        }
      });
      return newMetadata;
    });
  };

  // Method to trigger a refresh of the model list
  const refreshModelList = useCallback(() => {
    setModelListRefreshTrigger(prev => prev + 1);
  }, []);

  // Method to start polling (e.g., when a download is initiated)
  const startPolling = () => {
    if (!isPolling) {
      setIsPolling(true);
    }
  };

  const value = {
    statusModelFiles,
    updateStatusModelFiles,
    clearCompletedDownloads,
    cancelDownload,
    cancelAllDownloads,
    showDownloadStatus,
    setShowDownloadStatus,
    toast,
    showToast,
    closeToast,
    refreshModelList,
    modelListRefreshTrigger,
    startPolling,
    hasActiveDownloads,
  };

  return <ModelFilesContext.Provider value={value}>{children}</ModelFilesContext.Provider>;
};

export default ModelFilesContext;
