import React, { useState, useEffect, useCallback, useContext, useMemo } from 'react';
import { styled } from '@mui/material/styles';
import axios from 'axios';
import { useParams } from 'react-router-dom';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Tooltip,
  Grid,
  Checkbox,
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import ModelConfigList from './ModelConfigList';
import ModelDeploymentList from './ModelDeploymentList';
import AddModelConfigForm from './AddModelConfigForm';
import ModelFilesContext from './ModelFilesContext';
import { BASE_URL } from '../../const';

// Styled components
const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem', // Reduced from 1.8rem
  fontWeight: 700,
  marginBottom: theme.spacing(1.5), // Reduced spacing
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px', // Reduced width
    height: '2px', // Reduced height
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  padding: theme.spacing(3, 2),
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.875rem', // Reduced from 0.95rem
  padding: theme.spacing(1.25, 1.75), // Reduced from 1.5, 2
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.5, 1.75), // Reduced from 1.8, 2
  fontSize: '0.8125rem', // Reduced from 0.9rem (13px)
}));

const ModelDetail = () => {
  const [model, setModel] = useState(null);
  const [modelConfigs, setModelConfigs] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [refreshDeployments, setRefreshDeployments] = useState(false);
  const { model_id } = useParams();

  // Get download status from context
  const { statusModelFiles } = useContext(ModelFilesContext) || { statusModelFiles: [] };

  // Check if the current model has files that are downloading
  const isModelDownloading = useMemo(() => {
    if (!model || !statusModelFiles.length) return false;

    return statusModelFiles.some(
      file =>
        file.m_id === model.id &&
        (file.is_downloading || (file.dl_requested_at && !file.storage_location))
    );
  }, [model, statusModelFiles]);

  useEffect(() => {
    const fetchModelDetail = async () => {
      setLoading(true);
      try {
        // Add timestamp to bypass any potential caching
        const response = await axios.get(`${BASE_URL}/models/${model_id}`, {
          params: { _t: Date.now() },
        });
        setModel(response.data);
        setLoading(false);
      } catch (err) {
        if (err.response && err.response.status === 404) {
          setError(new Error('Model not found'));
        } else {
          setError(err);
        }
        setLoading(false);
      }
    };

    const fetchModelConfigs = async () => {
      setLoading(true);
      try {
        const response = await axios.get(`${BASE_URL}/models/${model_id}/configs`);
        const configData = response.data || [];

        // Ensure configData is an array before spreading
        const configArray = Array.isArray(configData) ? configData : [];
        
        // Sort configs: default first, then alphabetically by name
        const sortedConfigs = [...configArray].sort((a, b) => {
          // Default configs come first
          if (a.default && !b.default) return -1;
          if (!a.default && b.default) return 1;

          // Then sort alphabetically by name
          const nameA = a.name || 'Unnamed';
          const nameB = b.name || 'Unnamed';
          return nameA.localeCompare(nameB);
        });

        setModelConfigs(sortedConfigs);
        setLoading(false);
      } catch (err) {
        setError(err);
        setLoading(false);
      }
    };

    fetchModelDetail();
    fetchModelConfigs();
  }, [model_id]);



  const handleAddConfig = () => {
    setShowModal(true);
  };

  const closeModal = () => {
    setShowModal(false);
  };

  const triggerRefreshDeployments = useCallback(() => {
    setRefreshDeployments(prevState => !prevState);
  }, []);

  if (loading) return <Spinner />;

  return (
    <ContentContainer>
      {error ? (
        <ErrorComponent message={error.message} />
      ) : model ? (
        <Grid container spacing={4}>
          <Grid item xs={12} md={6}>
            <SectionTitle variant="h4">Model Details</SectionTitle>
            <StyledTableContainer component={Paper}>
              <StyledTable>
                <TableBody>
                  <TableRow>
                    <TableBodyCell>Name</TableBodyCell>
                    <TableBodyCell>{model.name}</TableBodyCell>
                  </TableRow>
                  {model.family && (
                    <TableRow>
                      <TableBodyCell>Family</TableBodyCell>
                      <TableBodyCell>{model.family}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.purpose && (
                    <TableRow>
                      <TableBodyCell>Purpose</TableBodyCell>
                      <TableBodyCell>{model.purpose}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.version && (
                    <TableRow>
                      <TableBodyCell>Version</TableBodyCell>
                      <TableBodyCell>{model.version}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.author && (
                    <TableRow>
                      <TableBodyCell>Author</TableBodyCell>
                      <TableBodyCell>{model.author}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.source_repository && (
                    <TableRow>
                      <TableBodyCell>Source Repository</TableBodyCell>
                      <TableBodyCell>{model.source_repository}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.description && (
                    <TableRow>
                      <TableBodyCell>Description</TableBodyCell>
                      <TableBodyCell>{model.description}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.sha_repository && (
                    <TableRow>
                      <TableBodyCell>SHA Repository</TableBodyCell>
                      <TableBodyCell>{model.sha_repository}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.hub && (
                    <TableRow>
                      <TableBodyCell>Hub</TableBodyCell>
                      <TableBodyCell>{model.hub}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.private !== null && (
                    <TableRow>
                      <TableBodyCell>Private</TableBodyCell>
                      <TableBodyCell>{model.private ? 'Yes' : 'No'}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.repo_modelId && (
                    <TableRow>
                      <TableBodyCell>
                        Repository Model ID
                        <Tooltip title="The unique identifier for this model in the source repository">
                          <InfoOutlinedIcon
                            sx={{
                              fontSize: '0.9rem',
                              ml: 0.5,
                              color: 'text.secondary',
                              verticalAlign: 'middle',
                            }}
                          />
                        </Tooltip>
                      </TableBodyCell>
                      <TableBodyCell>{model.repo_modelId}</TableBodyCell>
                    </TableRow>
                  )}
                  {model.modelfamily && (
                    <TableRow>
                      <TableBodyCell>
                        Model Family
                        <Tooltip title="The model architecture family (e.g., GPT, BERT, LLaMA)">
                          <InfoOutlinedIcon
                            sx={{
                              fontSize: '0.9rem',
                              ml: 0.5,
                              color: 'text.secondary',
                              verticalAlign: 'middle',
                            }}
                          />
                        </Tooltip>
                      </TableBodyCell>
                      <TableBodyCell>{model.modelfamily}</TableBodyCell>
                    </TableRow>
                  )}
                  {(() => {
                    // Find the first file with a storage location
                    const fileWithStorage = model.m_files?.find(file => file.storage_location);
                    if (fileWithStorage) {
                      return (
                        <>
                          <TableRow>
                            <TableBodyCell>Storage Location</TableBodyCell>
                            <TableBodyCell
                              sx={{
                                wordBreak: 'break-all',
                                userSelect: 'text',
                                cursor: 'text',
                              }}
                            >
                              {fileWithStorage.storage_location}
                            </TableBodyCell>
                          </TableRow>
                          <TableRow>
                            <TableBodyCell>Storage Type</TableBodyCell>
                            <TableBodyCell>{fileWithStorage.storage_type || 'file'}</TableBodyCell>
                          </TableRow>
                          {fileWithStorage.storage_host && (
                            <TableRow>
                              <TableBodyCell>Storage Host</TableBodyCell>
                              <TableBodyCell>{fileWithStorage.storage_host}</TableBodyCell>
                            </TableRow>
                          )}
                        </>
                      );
                    }
                    return null;
                  })()}
                  {model.created_timestamp && (
                    <TableRow>
                      <TableBodyCell>Created</TableBodyCell>
                      <TableBodyCell>
                        {new Date(model.created_timestamp).toLocaleString('en-US', {
                          month: 'short',
                          day: 'numeric',
                          year: 'numeric',
                          hour: 'numeric',
                          minute: '2-digit',
                          hour12: true,
                        })}
                      </TableBodyCell>
                    </TableRow>
                  )}
                  {model.modified_timestamp && (
                    <TableRow>
                      <TableBodyCell>Last Modified</TableBodyCell>
                      <TableBodyCell>
                        {new Date(model.modified_timestamp).toLocaleString('en-US', {
                          month: 'short',
                          day: 'numeric',
                          year: 'numeric',
                          hour: 'numeric',
                          minute: '2-digit',
                          hour12: true,
                        })}
                      </TableBodyCell>
                    </TableRow>
                  )}
                </TableBody>
              </StyledTable>
            </StyledTableContainer>

            {model.m_files && model.m_files.length > 0 && (
              <>
                <SectionTitle variant="h5" sx={{ mt: 4 }}>
                  Model Files
                </SectionTitle>
                <StyledTableContainer component={Paper}>
                  <StyledTable style={{ tableLayout: 'fixed', width: '100%' }}>
                    <TableHead>
                      <TableRow>
                        <TableHeaderCell style={{ width: '45%' }}>File Name</TableHeaderCell>
                        <TableHeaderCell style={{ width: '25%' }}>Size</TableHeaderCell>
                        <TableHeaderCell style={{ width: '30%' }}>Status</TableHeaderCell>
                      </TableRow>
                    </TableHead>
                    <TableBody>
                      {model.m_files.map(file => {
                        // Find matching status from context
                        const statusFile = statusModelFiles.find(sf => sf.id === file.id);
                        const mergedFile = statusFile ? { ...file, ...statusFile } : file;
                        // Only show as downloading/queued if file is in active status
                        const hasActiveStatus = !!statusFile;

                        let fileSize = mergedFile.size;
                        let unit = 'B';
                        if (fileSize >= 1024) {
                          fileSize /= 1024;
                          unit = 'K';
                        }
                        if (fileSize >= 1024) {
                          fileSize /= 1024;
                          unit = 'M';
                        }
                        if (fileSize >= 1024) {
                          fileSize /= 1024;
                          unit = 'G';
                        }

                        // Determine if file is available (downloaded)
                        // If file is in active status, trust the status completely
                        // Widget considers file completed if storage_location exists AND dl_requested_at is null
                        const isAvailable = hasActiveStatus
                          ? (mergedFile.storage_location !== null && mergedFile.dl_requested_at === null)
                          : (mergedFile.storage_location !== null || mergedFile.dl_requested_at !== null);

                        // Build status text for downloading/queued files
                        let statusText = '';
                        if (hasActiveStatus && mergedFile.is_downloading) {
                          statusText = 'downloading';
                        } else if (
                          hasActiveStatus &&
                          mergedFile.dl_requested_at !== null &&
                          !mergedFile.is_downloading &&
                          !isAvailable
                        ) {
                          statusText = 'queued';
                        }

                        return (
                          <TableRow key={file.id}>
                            <TableBodyCell style={{ overflow: 'hidden', textOverflow: 'ellipsis' }}>
                              {file.name}
                            </TableBodyCell>
                            <TableBodyCell>
                              {fileSize.toFixed(1)}
                              {unit}
                            </TableBodyCell>
                            <TableBodyCell>
                              <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                <Checkbox
                                  checked={isAvailable}
                                  disabled
                                  size="small"
                                  sx={{ p: 0 }}
                                />
                                {statusText && (
                                  <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                                    {statusText}
                                  </Typography>
                                )}
                              </Box>
                            </TableBodyCell>
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </StyledTable>
                </StyledTableContainer>
              </>
            )}
          </Grid>

          <Grid item xs={12} md={6}>
            <ModelConfigList
              model_id={model_id}
              handleAddConfig={handleAddConfig}
              triggerRefreshDeployments={triggerRefreshDeployments}
              setParentModelConfigs={setModelConfigs}
              modelConfigs={modelConfigs}
              isModelDownloading={isModelDownloading}
            />

            <AddModelConfigForm
              showModal={showModal}
              closeModal={closeModal}
              m_id={model_id}
              setParentModelConfigs={setModelConfigs}
            />

            <Box sx={{ mt: 4 }}>
              <Typography
                variant="h5"
                sx={{
                  color: 'primary.main',
                  fontWeight: 600,
                  mb: 1.5,
                  fontSize: '1.25rem', // Reduced from 1.5rem
                }}
              >
                Model Deployments
                <Tooltip title="Active deployments of this model">
                  <InfoOutlinedIcon
                    sx={{
                      fontSize: '1rem', // Reduced from 1.1rem
                      ml: 0.75,
                      color: 'text.secondary',
                      cursor: 'pointer',
                      '&:hover': { color: 'primary.main' },
                    }}
                  />
                </Tooltip>
              </Typography>
              <ModelDeploymentList model_id={model_id} refreshDeployments={refreshDeployments} />
            </Box>
          </Grid>
        </Grid>
      ) : (
        <Typography variant="body1" color="text.secondary">
          Model not found
        </Typography>
      )}
    </ContentContainer>
  );
};

export default ModelDetail;
