import React, { useState, useEffect, useCallback } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Checkbox,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  DialogContentText,
  Chip,
  Modal,
  IconButton,
} from '@mui/material';
import StopIcon from '@mui/icons-material/Stop';
import DescriptionIcon from '@mui/icons-material/Description';
import CloseIcon from '@mui/icons-material/Close';
import CircularProgress from '@mui/material/CircularProgress';
import axios from 'axios';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';
import { buildModelRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';
import ModelDeploymentDetailModal from './ModelDeploymentDetail';
import TestConversationModal from './TestConversationModal';
import ContainerLogViewer from './ContainerLogViewer';

// Styled components - SubTitle removed as it's no longer needed

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'auto',
  maxWidth: '100%',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  minWidth: 1200, // Ensure table has minimum width for scrolling
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.875rem', // Reduced from 0.95rem
  padding: theme.spacing(1.25, 1.75), // Reduced from 1.5, 2
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.5, 1.75), // Reduced from 1.8, 2
  fontSize: '0.8125rem', // Reduced from 0.9rem (13px)
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  // Style for empty tables
  '&.empty-row': {
    height: 100,
    '& td': {
      textAlign: 'center',
      color: theme.palette.text.secondary,
    },
  },
}));

const ActionButton = styled(Button)(({ theme }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.5, 1.5), // Reduced from 0.8, 2
  borderRadius: theme.shape.borderRadius,
  fontSize: '0.8125rem', // 13px for compact buttons
  transition: 'all 0.2s',
  '&:hover': {
    transform: 'translateY(-1px)',
    boxShadow: '0 2px 4px rgba(0, 0, 0, 0.15)',
  },
  '&.MuiButton-containedPrimary': {
    backgroundColor: theme.palette.primary.main,
    color: theme.palette.primary.contrastText,
    '&:hover': {
      backgroundColor: theme.palette.primary.dark,
    },
  },
  '&.MuiButton-containedError': {
    backgroundColor: theme.palette.error.main,
    color: theme.palette.error.contrastText,
    '&:hover': {
      backgroundColor: theme.palette.error.dark,
    },
  },
}));

const LogsModalContainer = styled(Box)(({ theme }) => ({
  position: 'absolute',
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  width: 'auto',
  maxWidth: '95vw',
  minWidth: '800px',
  backgroundColor: theme.palette.background.paper,
  boxShadow: theme.shadows[5],
  padding: theme.spacing(3),
  borderRadius: 12,
  maxHeight: '95vh',
  overflow: 'auto',
  display: 'flex',
  flexDirection: 'column',
}));

const LogsModalHeader = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'flex-start',
  marginBottom: theme.spacing(2),
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 500,
  fontSize: '0.6875rem', // Reduced from 0.75rem (11px)
  height: '22px', // Reduced from 24px
  ...(status === 'DEPLOYED' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'INITIALIZING' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    color: '#ff9800',
    borderColor: '#ff9800',
  }),
  ...(status === 'ERROR' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    color: '#f44336',
    borderColor: '#f44336',
  }),
  ...(status === 'FAILED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    color: '#f44336',
    borderColor: '#f44336',
  }),
  ...(status === 'STOPPED' && {
    backgroundColor: 'rgba(158, 158, 158, 0.1)',
    color: '#9e9e9e',
    borderColor: '#9e9e9e',
  }),
}));

const ErrorBadge = ({ deployment }) => {
  if (!deployment.last_error_code && !deployment.last_error_message) return null;
  const label = deployment.last_error_code || 'Error';
  return (
    <Chip
      label={label}
      size="small"
      color="error"
      variant="outlined"
      sx={{ ml: 1, height: '20px', fontSize: '0.6875rem' }}
      title={deployment.last_error_message || label}
    />
  );
};

ErrorBadge.propTypes = {
  deployment: PropTypes.shape({
    last_error_code: PropTypes.string,
    last_error_message: PropTypes.string,
  }).isRequired,
};

/**
 * Component to list model deployments with options to view details and open API documentation.
 * @param {Object} props - Component props.
 * @param {string} [props.m_id] - Optional model ID to filter deployments.
 * @param {boolean} props.refreshDeployments - Trigger to refresh deployments list.
 */
const ModelDeploymentList = ({
  model_id,
  refreshDeployments,
  selectedDeployments = [],
  onSelectionChange,
}) => {
  // Get routing mode from context to handle dual mode properly
  const { routingMode } = useRoutingConfig();

  const [modelDeployments, setModelDeployments] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedDeploymentId, setSelectedDeploymentId] = useState(null);
  const [stoppingDeploymentId, setStoppingDeploymentId] = useState(null);
  const [showStopConfirm, setShowStopConfirm] = useState({ open: false, deployment: null });
  const [testDeployment, setTestDeployment] = useState(null);
  const [showLogsDeploymentId, setShowLogsDeploymentId] = useState(null);

  // Auto-refresh functionality with adaptive intervals
  const [refreshInterval, setRefreshInterval] = useState(null);
  const FAST_REFRESH_DELAY = 5000; // 5 seconds for initializing deployments
  const SLOW_REFRESH_DELAY = 60000; // 60 seconds for stable deployments

  // Define fetchModelDeployments as a useCallback hook to avoid re-creation on every render
  const fetchModelDeployments = useCallback(
    async (silent = false) => {
      if (!silent) {
        setLoading(true);
      }
      try {
        let url = `${BASE_URL}/serving/deployments`;
        if (model_id) {
          url += `?model_id=${model_id}`;
        }
        const response = await axios.get(url);
        setModelDeployments(response.data);
        if (!silent) {
          setLoading(false);
        }
      } catch (err) {
        console.error('Error fetching deployments:', err);
        if (!silent) {
          setError(err);
          setLoading(false);
        }
      }
    },
    [model_id]
  );

  useEffect(() => {
    fetchModelDeployments();
  }, [fetchModelDeployments, refreshDeployments]);

  // Auto-refresh effect with adaptive intervals
  useEffect(() => {
    // Don't auto-refresh if any modal is open (to avoid disrupting user interactions)
    const isModalOpen = selectedDeploymentId || showStopConfirm.open;

    if (!isModalOpen) {
      // Determine refresh interval based on deployment statuses
      const hasInitializingDeployment = modelDeployments.some(
        deployment => deployment.status === 'INITIALIZING'
      );
      const refreshDelay = hasInitializingDeployment ? FAST_REFRESH_DELAY : SLOW_REFRESH_DELAY;

      const interval = setInterval(() => {
        fetchModelDeployments(true); // Silent refresh
      }, refreshDelay);

      setRefreshInterval(interval);

      return () => {
        clearInterval(interval);
        setRefreshInterval(null);
      };
    } else if (refreshInterval) {
      // Clear interval if modal is open
      clearInterval(refreshInterval);
      setRefreshInterval(null);
    }
  }, [
    selectedDeploymentId,
    showStopConfirm.open,
    modelDeployments,
    FAST_REFRESH_DELAY,
    SLOW_REFRESH_DELAY,
    fetchModelDeployments,
  ]);

  // Cleanup interval on unmount
  useEffect(() => {
    return () => {
      if (refreshInterval) {
        clearInterval(refreshInterval);
      }
    };
  }, [refreshInterval]);

  const handleOpenDetails = deploymentId => {
    setSelectedDeploymentId(deploymentId);
  };

  // Function to handle refreshing the list of deployments
  const handleRefreshDeployments = () => {
    fetchModelDeployments();
  };

  const handleStopDeployment = deployment => {
    setShowStopConfirm({ open: true, deployment });
  };

  const handleConfirmStop = async () => {
    const deployment = showStopConfirm.deployment;
    if (!deployment) return;

    setStoppingDeploymentId(deployment.id);

    try {
      await axios.delete(`${BASE_URL}/serving/deployment/${deployment.id}`, {
        params: { force: true },
      });

      // Refresh the deployment list
      fetchModelDeployments();

      // Close the confirmation dialog
      setShowStopConfirm({ open: false, deployment: null });
    } catch (error) {
      console.error('Error stopping deployment:', error);
      alert('Failed to stop deployment. Please try again.');
    } finally {
      setStoppingDeploymentId(null);
    }
  };

  const handleCancelStop = () => {
    setShowStopConfirm({ open: false, deployment: null });
  };

  const handleSelectAll = event => {
    if (!onSelectionChange) return;

    if (event.target.checked) {
      const allDeploymentIds = modelDeployments
        .filter(deployment => deployment.status !== 'STOPPED')
        .map(deployment => deployment.id);
      onSelectionChange(allDeploymentIds);
    } else {
      onSelectionChange([]);
    }
  };

  const handleSelectDeployment = deploymentId => {
    if (!onSelectionChange) return;

    const newSelection = selectedDeployments.includes(deploymentId)
      ? selectedDeployments.filter(id => id !== deploymentId)
      : [...selectedDeployments, deploymentId];

    onSelectionChange(newSelection);
  };

  const handleTestDeployment = deployment => {
    setTestDeployment(deployment);
  };

  const handleCloseTestModal = () => {
    setTestDeployment(null);
  };

  const handleOpenLogs = (deploymentId) => {
    setShowLogsDeploymentId(deploymentId);
  };

  const handleCloseLogs = () => {
    setShowLogsDeploymentId(null);
  };

  const activeDeployments = modelDeployments.filter(deployment => deployment.status !== 'STOPPED');
  const isAllSelected =
    activeDeployments.length > 0 &&
    activeDeployments.every(deployment => selectedDeployments.includes(deployment.id));

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error.message} />;

  return (
    <Box sx={{ mb: 4 }}>
      <StyledTableContainer component={Paper}>
        <StyledTable>
          <TableHead>
            <TableRow>
              {onSelectionChange && (
                <TableHeaderCell padding="checkbox">
                  <Checkbox
                    checked={isAllSelected}
                    indeterminate={
                      selectedDeployments.length > 0 &&
                      selectedDeployments.length < activeDeployments.length
                    }
                    onChange={handleSelectAll}
                    sx={{ color: 'primary.main' }}
                  />
                </TableHeaderCell>
              )}
              <TableHeaderCell>Model Name</TableHeaderCell>
              <TableHeaderCell>Status</TableHeaderCell>
              <TableHeaderCell>Engine</TableHeaderCell>
              <TableHeaderCell>Config</TableHeaderCell>
              <TableHeaderCell>Base URL</TableHeaderCell>
              <TableHeaderCell>API Endpoints</TableHeaderCell>
              <TableHeaderCell>Actions</TableHeaderCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {modelDeployments.length === 0 ? (
              <StyledTableRow className="empty-row">
                <TableBodyCell colSpan={onSelectionChange ? 7 : 6}>
                  No deployed models found
                </TableBodyCell>
              </StyledTableRow>
            ) : (
              modelDeployments
                .filter(deployment => deployment.status !== 'STOPPED')
                .map(deployment => {
                  const runtimeInfo = buildModelRuntimeInfo(deployment, routingMode);
                  const isDualMode = routingMode === 'dual';
                  const hasPort = Boolean(runtimeInfo.portEndpoints?.baseUrl);
                  const hasPath = Boolean(runtimeInfo.pathEndpoints?.baseUrl);
                  const runtimeAvailable = hasPort || hasPath;
                  // Show port route unless we're in path-only mode
                  const showPort = isDualMode || routingMode !== 'path';
                  // Show path route in path mode, dual mode, or as fallback when no port available
                  const needsPathFallback = showPort && !hasPort;
                  const showPath = isDualMode || routingMode === 'path' || needsPathFallback;

                  // Build list of routes to display
                  const routes = [];
                  if (showPort && hasPort) {
                    routes.push({ url: runtimeInfo.portEndpoints.baseUrl, type: 'port', endpoints: runtimeInfo.portEndpoints });
                  }
                  if (showPath && hasPath) {
                    routes.push({ url: runtimeInfo.pathEndpoints.baseUrl, type: 'path', endpoints: runtimeInfo.pathEndpoints });
                  }

                  const publicPath = runtimeInfo.publicPath || '';
                  const formatEndpointLabel = url => {
                    if (!url) return 'unavailable';
                    try {
                      const { pathname } = new URL(url);
                      if (!pathname) return 'unavailable';
                      if (publicPath && pathname.startsWith(publicPath)) {
                        const suffix = pathname.slice(publicPath.length) || '/';
                        return suffix.startsWith('/') ? suffix : `/${suffix}`;
                      }
                      return pathname;
                    } catch (error) {
                      return url;
                    }
                  };

                  return (
                    <StyledTableRow key={deployment.id}>
                    {onSelectionChange && (
                      <TableBodyCell padding="checkbox">
                        <Checkbox
                          checked={selectedDeployments.includes(deployment.id)}
                          onChange={() => handleSelectDeployment(deployment.id)}
                          sx={{ color: 'primary.main' }}
                        />
                      </TableBodyCell>
                    )}
                    <TableBodyCell>
                      <Button
                        component="a"
                        href={`/models/${deployment.m_id}`}
                        sx={{
                          textTransform: 'none',
                          color: 'primary.main',
                          fontWeight: 500,
                          textDecoration: 'none',
                          padding: 0,
                          minWidth: 'auto',
                          '&:hover': {
                            textDecoration: 'underline',
                            backgroundColor: 'transparent',
                          },
                        }}
                      >
                        {deployment.m_name}
                      </Button>
                    </TableBodyCell>
                    <TableBodyCell>
                      <StatusChip
                        label={deployment.status}
                        size="small"
                        variant="outlined"
                        status={deployment.status}
                      />
                      <ErrorBadge deployment={deployment} />
                    </TableBodyCell>
                    <TableBodyCell>
                      {deployment.engine_name || 'Unknown'}
                    </TableBodyCell>
                    <TableBodyCell>{deployment.m_config_name || 'Default Config'}</TableBodyCell>
                    <TableBodyCell>
                      {runtimeAvailable ? (
                        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 0.5 }}>
                          {routes.map((route) => (
                            <Box key={route.type} sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                              {isDualMode && routes.length > 1 && (
                                <Typography
                                  variant="caption"
                                  sx={{
                                    color: route.type === 'port' ? 'secondary.main' : 'primary.main',
                                    fontWeight: 500,
                                    minWidth: '32px',
                                  }}
                                >
                                  {route.type === 'port' ? 'Port' : 'Path'}
                                </Typography>
                              )}
                              <Typography
                                component="a"
                                href={route.url}
                                target="_blank"
                                rel="noopener noreferrer"
                                sx={{
                                  fontFamily: 'monospace',
                                  fontSize: '0.8rem',
                                  color: 'primary.main',
                                  textDecoration: 'none',
                                  '&:hover': { textDecoration: 'underline' },
                                }}
                              >
                                {route.url}
                              </Typography>
                            </Box>
                          ))}
                        </Box>
                      ) : (
                        <Typography variant="caption" color="text.secondary">
                          unavailable
                        </Typography>
                      )}
                    </TableBodyCell>
                    <TableBodyCell>
                      {deployment.status === 'FAILED' ? (
                        <Typography variant="caption" color="error">
                          Failed
                        </Typography>
                      ) : runtimeAvailable ? (
                        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 0.5 }}>
                          <Typography
                            component="a"
                            href={runtimeInfo.chatUrl}
                            target="_blank"
                            rel="noopener noreferrer"
                            variant="caption"
                            sx={{
                              fontFamily: 'monospace',
                              fontSize: '0.75rem',
                              color: 'primary.main',
                              textDecoration: 'none',
                              '&:hover': { textDecoration: 'underline' },
                            }}
                          >
                            {formatEndpointLabel(runtimeInfo.chatUrl)}
                          </Typography>
                          {(deployment.engine_name === 'vllm' ||
                            deployment.engine_name === 'VLLM') && (
                            <Typography
                              component="a"
                              href={runtimeInfo.docsUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              variant="caption"
                              sx={{
                                fontFamily: 'monospace',
                                fontSize: '0.75rem',
                                color: 'primary.main',
                                textDecoration: 'none',
                                '&:hover': { textDecoration: 'underline' },
                              }}
                            >
                              {formatEndpointLabel(runtimeInfo.docsUrl)}
                            </Typography>
                          )}
                          {(deployment.engine_name === 'llamacpp' ||
                            deployment.engine_name === 'LlamaCpp' ||
                            deployment.engine_name === 'llama.cpp') && (
                            <>
                            <Typography
                              component="a"
                              href={runtimeInfo.propsUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              variant="caption"
                              sx={{
                                fontFamily: 'monospace',
                                fontSize: '0.75rem',
                                color: 'primary.main',
                                textDecoration: 'none',
                                '&:hover': { textDecoration: 'underline' },
                              }}
                            >
                              {formatEndpointLabel(runtimeInfo.propsUrl)}
                            </Typography>
                            <Typography
                              component="a"
                              href={runtimeInfo.healthUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              variant="caption"
                              sx={{
                                fontFamily: 'monospace',
                                fontSize: '0.75rem',
                                color: 'primary.main',
                                textDecoration: 'none',
                                '&:hover': { textDecoration: 'underline' },
                              }}
                            >
                              {formatEndpointLabel(runtimeInfo.healthUrl)}
                            </Typography>
                            <Typography
                              component="a"
                              href={runtimeInfo.modelsUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              variant="caption"
                              sx={{
                                fontFamily: 'monospace',
                                fontSize: '0.75rem',
                                color: 'primary.main',
                                textDecoration: 'none',
                                '&:hover': { textDecoration: 'underline' },
                              }}
                            >
                              {formatEndpointLabel(runtimeInfo.modelsUrl)}
                            </Typography>
                          </>
                          )}
                          {(deployment.engine_name === 'mlx' ||
                            deployment.engine_name === 'MLX') && (
                            <>
                              <Typography
                                component="a"
                                href={runtimeInfo.docsUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                variant="caption"
                                sx={{
                                  fontFamily: 'monospace',
                                  fontSize: '0.75rem',
                                  color: 'primary.main',
                                  textDecoration: 'none',
                                  '&:hover': { textDecoration: 'underline' },
                                }}
                              >
                                /docs
                              </Typography>
                              <Typography
                                component="a"
                                href={runtimeInfo.modelsUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                variant="caption"
                                sx={{
                                  fontFamily: 'monospace',
                                  fontSize: '0.75rem',
                                  color: 'primary.main',
                                  textDecoration: 'none',
                                  '&:hover': { textDecoration: 'underline' },
                                }}
                              >
                                /v1/models
                              </Typography>
                            </>
                          )}
                        </Box>
                      ) : (
                        <span>&nbsp;</span>
                      )}
                    </TableBodyCell>
                    <TableBodyCell>
                      <Box
                        sx={{
                          display: 'flex',
                          gap: 0.5,
                          flexWrap: 'nowrap',
                          minWidth: 'max-content',
                        }}
                      >
                        {runtimeAvailable && (
                          <>
                            <ActionButton
                              onClick={() => handleTestDeployment(deployment)}
                              size="small"
                              variant="outlined"
                              disabled={deployment.status !== 'DEPLOYED'}
                            >
                              Test
                            </ActionButton>
                          </>
                        )}
                        <ActionButton
                          onClick={() => handleOpenDetails(deployment.id)}
                          size="small"
                          variant="contained"
                        >
                          Details
                        </ActionButton>
                        <ActionButton
                          onClick={() => handleOpenLogs(deployment.id)}
                          size="small"
                          variant="outlined"
                          startIcon={<DescriptionIcon />}
                        >
                          Logs
                        </ActionButton>
                        <ActionButton
                          onClick={() => handleStopDeployment(deployment)}
                          size="small"
                          variant="contained"
                          color="error"
                          disabled={stoppingDeploymentId === deployment.id}
                          startIcon={
                            stoppingDeploymentId === deployment.id ? (
                              <CircularProgress size={16} />
                            ) : (
                              <StopIcon />
                            )
                          }
                        >
                          {stoppingDeploymentId === deployment.id ? 'Stopping' : 'Stop'}
                        </ActionButton>
                      </Box>
                    </TableBodyCell>
                    </StyledTableRow>
                  );
                })
            )}
          </TableBody>
        </StyledTable>
      </StyledTableContainer>

      {/* Conditionally render the modal if a deployment ID is selected */}
      {selectedDeploymentId && (
        <ModelDeploymentDetailModal
          deploymentId={selectedDeploymentId}
          isOpen={!!selectedDeploymentId}
          onClose={() => setSelectedDeploymentId(null)}
          onDeploymentStopped={handleRefreshDeployments}
        />
      )}

      {/* Stop Deployment Confirmation Dialog */}
      <Dialog open={showStopConfirm.open} onClose={handleCancelStop}>
        <DialogTitle>Stop Deployment</DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to stop the deployment for{' '}
            <strong>{showStopConfirm.deployment?.m_name}</strong>?
          </DialogContentText>
          <DialogContentText sx={{ mt: 2 }}>
            This will terminate the model serving instance and free up resources.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCancelStop} color="primary">
            Cancel
          </Button>
          <Button
            onClick={handleConfirmStop}
            color="error"
            variant="contained"
            disabled={stoppingDeploymentId === showStopConfirm.deployment?.id}
            startIcon={
              stoppingDeploymentId === showStopConfirm.deployment?.id ? (
                <CircularProgress size={16} />
              ) : null
            }
          >
            {stoppingDeploymentId === showStopConfirm.deployment?.id
              ? 'Stopping...'
              : 'Stop Deployment'}
          </Button>
        </DialogActions>
      </Dialog>

      {/* Test Conversation Modal */}
      <TestConversationModal
        open={!!testDeployment}
        onClose={handleCloseTestModal}
        deployment={testDeployment}
      />

      {/* Logs Modal */}
      <Modal open={!!showLogsDeploymentId} onClose={handleCloseLogs}>
        <LogsModalContainer>
          <LogsModalHeader>
            <Typography variant="h6" sx={{ color: 'primary.main', fontWeight: 600 }}>
              Logs
            </Typography>
            <IconButton onClick={handleCloseLogs} size="small">
              <CloseIcon />
            </IconButton>
          </LogsModalHeader>
          
          {showLogsDeploymentId && (
            <ContainerLogViewer deploymentId={showLogsDeploymentId} />
          )}
        </LogsModalContainer>
      </Modal>
    </Box>
  );
};

ModelDeploymentList.propTypes = {
  model_id: PropTypes.string,
  refreshDeployments: PropTypes.oneOfType([PropTypes.bool, PropTypes.number]),
  selectedDeployments: PropTypes.arrayOf(PropTypes.string),
  onSelectionChange: PropTypes.func,
};

export default ModelDeploymentList;
