import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import axios from 'axios';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Modal,
  IconButton,
  CircularProgress,
  Tabs,
  Tab,
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import StopIcon from '@mui/icons-material/Stop';
import DeleteForeverIcon from '@mui/icons-material/DeleteForever';
import DescriptionIcon from '@mui/icons-material/Description';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';
import TestConversationModal from './TestConversationModal';
import ContainerLogViewer from './ContainerLogViewer';
import {
  format,
  differenceInHours,
  differenceInDays,
  differenceInWeeks,
} from 'date-fns';
import { buildModelRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';

// Styled components
const ModalContainer = styled(Box)(({ theme }) => ({
  position: 'absolute',
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  width: 'auto',
  maxWidth: '95vw',
  minWidth: '600px',
  backgroundColor: theme.palette.background.paper,
  boxShadow: theme.shadows[5],
  padding: theme.spacing(4),
  borderRadius: 12,
  maxHeight: '95vh',
  overflow: 'auto',
  display: 'flex',
  flexDirection: 'column',
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 700,
  marginBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: -4,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 2),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
}));

const ActionButton = styled(Button)(({ theme, color = 'primary' }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.8, 2),
  borderRadius: theme.shape.borderRadius,
  margin: theme.spacing(0.5),
  transition: 'all 0.2s',
  backgroundColor:
    color === 'primary'
      ? theme.palette.primary.main
      : color === 'error'
        ? theme.palette.error.main
        : theme.palette.secondary.main,
  color: theme.palette.common.white,
  '&:hover': {
    backgroundColor:
      color === 'primary'
        ? theme.palette.primary.dark
        : color === 'error'
          ? theme.palette.error.dark
          : theme.palette.secondary.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
}));

const ModalHeader = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'flex-start',
  marginBottom: theme.spacing(2),
}));

const ConfirmModal = styled(Modal)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
}));

const ConfirmModalContent = styled(Box)(({ theme }) => ({
  backgroundColor: theme.palette.background.paper,
  boxShadow: theme.shadows[5],
  padding: theme.spacing(4),
  borderRadius: 12,
  maxWidth: '500px',
  textAlign: 'center',
}));

const ErrorText = styled(Typography)(({ theme }) => ({
  color: theme.palette.error.main,
  marginBottom: theme.spacing(2),
}));

const StyledTabs = styled(Tabs)(({ theme }) => ({
  borderBottom: '1px solid rgba(255, 255, 255, 0.12)',
  marginBottom: theme.spacing(2),
  '& .MuiTab-root': {
    textTransform: 'none',
    fontWeight: 500,
    fontSize: '0.95rem',
    minHeight: '48px',
    color: theme.palette.text.secondary,
    '&.Mui-selected': {
      color: theme.palette.primary.main,
    },
  },
  '& .MuiTabs-indicator': {
    backgroundColor: theme.palette.primary.main,
    height: 3,
  },
}));

const TabPanel = styled(Box)(({ theme }) => ({
  display: 'none',
  '&.active': {
    display: 'block',
  },
}));

/**
 * Modal component to display the details of a model deployment.
 * @param {Object} props - Component props.
 * @param {string} props.deploymentId - The ID of the deployment to display details for.
 * @param {boolean} props.isOpen - Controls the visibility of the modal.
 * @param {Function} props.onClose - Callback function to close the modal.
 * @param {Function} props.onDeploymentStopped - Callback function to trigger refresh of the deployment list.
 */
const ModelDeploymentDetailModal = ({ deploymentId, isOpen, onClose, onDeploymentStopped }) => {
  // Get routing mode from context
  // - 'path': show only path endpoints
  // - 'port': show only port endpoints
  // - 'dual': show BOTH port and path endpoints
  const { routingMode } = useRoutingConfig();

  const [deploymentDetail, setDeploymentDetail] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [stopError, setStopError] = useState(null);
  const [confirmStop, setConfirmStop] = useState(false);
  const [confirmForceRemove, setConfirmForceRemove] = useState(false);
  const [isStoppingDeployment, setIsStoppingDeployment] = useState(false);
  const [showTestModal, setShowTestModal] = useState(false);
  const [activeTab, setActiveTab] = useState(0);
  const [runtimeInfo, setRuntimeInfo] = useState(null);

  useEffect(() => {
    const fetchDeploymentDetail = async () => {
      if (!deploymentId) return;
      setLoading(true);
      try {
        const response = await axios.get(`${BASE_URL}/serving/deployment/${deploymentId}`);
        setDeploymentDetail(response.data);
        setRuntimeInfo(response.data ? buildModelRuntimeInfo(response.data, routingMode) : null);
        setLoading(false);
      } catch (err) {
        setError(err);
        setLoading(false);
      }
    };

    if (isOpen) {
      fetchDeploymentDetail();
    }
  }, [deploymentId, isOpen]);

  useEffect(() => {
    if (deploymentDetail) {
      setRuntimeInfo(buildModelRuntimeInfo(deploymentDetail, routingMode));
    }
  }, [deploymentDetail, routingMode]);

  const pathEndpoints = runtimeInfo?.pathEndpoints || null;
  const portEndpoints = runtimeInfo?.portEndpoints || null;

  // Build endpoint groups based on routing mode
  // - dual: show BOTH port and path (port first as it's the normalized default)
  // - path: show only path endpoints (fail gracefully if unavailable)
  // - port: show only port endpoints (with path fallback if unavailable)
  const endpointGroups = [];
  const isDualMode = routingMode === 'dual';
  const showPort = isDualMode || routingMode !== 'path'; // port, dual, or default
  // Show path in path mode, dual mode, or as fallback when no port available
  const needsPathFallback = showPort && !portEndpoints;
  const showPath = isDualMode || routingMode === 'path' || needsPathFallback;

  // In dual mode, show port first (it's the normalized default)
  if (showPort && portEndpoints) {
    const portLabel = isDualMode
      ? (runtimeInfo?.lbPort ? `Port (${runtimeInfo.lbPort})` : 'Port')
      : 'Endpoints';
    endpointGroups.push({
      label: portLabel,
      endpoints: portEndpoints,
      variant: 'port',
    });
  }
  if (showPath && pathEndpoints) {
    const pathLabel = isDualMode ? 'Path' : 'Endpoints';
    endpointGroups.push({
      label: pathLabel,
      endpoints: pathEndpoints,
      variant: 'path',
    });
  }

  const engineNameNormalized = (deploymentDetail?.engine_name || '').toLowerCase();
  const supportsLegacyCompletions = ['vllm', 'llamacpp', 'llama.cpp'].includes(engineNameNormalized);
  const supportsEmbeddings = engineNameNormalized === 'vllm';

  const handleStopDeployment = async (force = true) => {
    setIsStoppingDeployment(true);
    setConfirmStop(false); // Close the confirmation dialog immediately
    try {
      await axios.delete(`${BASE_URL}/serving/deployment/${deploymentId}`, { params: { force } });
      onClose(); // Close the modal after stopping the deployment
      onDeploymentStopped(); // Trigger refresh of the deployment list
    } catch (err) {
      if (err.response && err.response.status === 404) {
        setStopError('Deployment not found. You may need to force remove it.');
        // Update the deployment status to trigger the "Force Remove" button
        setDeploymentDetail(prevState => {
          const updated = {
            ...prevState,
            status: 'STOP_REQUESTED',
          };
          setRuntimeInfo(buildModelRuntimeInfo(updated, routingMode));
          return updated;
        });
      } else {
        setError(err);
      }
      setIsStoppingDeployment(false);
    }
  };

  const formatDuration = minutes => {
    if (minutes < 60) return `${minutes} minutes`;
    const hours = differenceInHours(
      new Date(),
      new Date().setMinutes(new Date().getMinutes() - minutes)
    );
    if (hours < 24) return `${hours} hours`;
    const days = differenceInDays(
      new Date(),
      new Date().setMinutes(new Date().getMinutes() - minutes)
    );
    if (days < 7) return `${days} days`;
    const weeks = differenceInWeeks(
      new Date(),
      new Date().setMinutes(new Date().getMinutes() - minutes)
    );
    return `${weeks} weeks`;
  };

  if (!isOpen) return null;

  if (loading)
    return (
      <Modal open={isOpen} onClose={onClose}>
        <ModalContainer>
          <Spinner />
        </ModalContainer>
      </Modal>
    );

  if (error)
    return (
      <Modal open={isOpen} onClose={onClose}>
        <ModalContainer>
          <ErrorComponent message={error.message} />
          <ActionButton onClick={onClose} sx={{ mt: 2 }}>
            Close
          </ActionButton>
        </ModalContainer>
      </Modal>
    );

  // Render deployment details
  return (
    <Modal open={isOpen} onClose={onClose}>
      <ModalContainer>
        {!deploymentDetail ? (
          <>
            <ModalHeader>
              <Typography variant="body1" color="text.secondary">
                Deployment detail not found.
              </Typography>
              <IconButton onClick={onClose} aria-label="close" size="small">
                <CloseIcon />
              </IconButton>
            </ModalHeader>
          </>
        ) : (
          <>
            <ModalHeader>
              <SectionTitle variant="h5">Deployment Detail</SectionTitle>
              <IconButton onClick={onClose} aria-label="close" size="small">
                <CloseIcon />
              </IconButton>
            </ModalHeader>

            {/* Tabs Navigation */}
            <StyledTabs value={activeTab} onChange={(e, newValue) => setActiveTab(newValue)}>
              <Tab label="Details" />
              <Tab
                label="Logs"
                icon={<DescriptionIcon />}
                iconPosition="start"
              />
            </StyledTabs>

            {/* Details Tab */}
            <TabPanel className={activeTab === 0 ? 'active' : ''}>
              <StyledTableContainer component={Paper}>
              <StyledTable>
                <TableBody>
                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600, width: '30%' }}>Model Name</TableBodyCell>
                    <TableBodyCell>{deploymentDetail.m_name}</TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Configuration</TableBodyCell>
                    <TableBodyCell>
                      {deploymentDetail.m_config_name || 'Default Config'}
                    </TableBodyCell>
                  </TableRow>

                  {runtimeInfo && endpointGroups.length > 0 && (
                    <>
                      <TableRow>
                        <TableBodyCell sx={{ fontWeight: 600 }}>API Access</TableBodyCell>
                        <TableBodyCell sx={{ wordBreak: 'break-all' }}>
                          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1.5 }}>
                            {endpointGroups.map((group) => (
                              <React.Fragment key={group.label}>
                                <Box
                                  sx={{
                                    border: '1px solid rgba(255,255,255,0.12)',
                                    borderRadius: 1,
                                    p: 1.5,
                                    backgroundColor: 'rgba(255,255,255,0.02)',
                                  }}
                                >
                                  <Typography
                                    variant="subtitle2"
                                    sx={{ color: group.variant === 'port' ? 'secondary.main' : 'primary.main' }}
                                  >
                                    {group.label}
                                  </Typography>
                                  <Typography
                                    component="a"
                                    href={group.endpoints.baseUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    sx={{
                                      color: '#00c07f',
                                      textDecoration: 'none',
                                      fontFamily: 'monospace',
                                      fontSize: '0.85rem',
                                      wordBreak: 'break-all',
                                    }}
                                  >
                                    {group.endpoints.baseUrl}
                                  </Typography>
                                </Box>
                              </React.Fragment>
                            ))}
                          </Box>
                        </TableBodyCell>
                      </TableRow>

                      <TableRow>
                        <TableBodyCell sx={{ fontWeight: 600 }}>OpenAI Compatible</TableBodyCell>
                        <TableBodyCell>
                          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1.5 }}>
                            {endpointGroups.map((group) => (
                              <React.Fragment key={`${group.label}-openai`}>
                                <Box
                                  sx={{
                                    border: '1px solid rgba(255,255,255,0.12)',
                                    borderRadius: 1,
                                    p: 1.5,
                                    backgroundColor: 'rgba(255,255,255,0.02)',
                                  }}
                                >
                                  <Typography
                                    variant="subtitle2"
                                    sx={{ color: group.variant === 'port' ? 'secondary.main' : 'primary.main' }}
                                  >
                                    {group.label}
                                  </Typography>
                                  <Typography
                                    variant="caption"
                                    component="div"
                                    sx={{ fontFamily: 'monospace', color: 'text.secondary' }}
                                  >
                                    POST {group.endpoints.chatUrl}
                                  </Typography>
                                  {supportsLegacyCompletions && (
                                    <Typography
                                      variant="caption"
                                      component="div"
                                      sx={{ fontFamily: 'monospace', color: 'text.secondary' }}
                                    >
                                      POST {group.endpoints.completionsUrl}
                                    </Typography>
                                  )}
                                  <Typography
                                    variant="caption"
                                    component="div"
                                    sx={{ fontFamily: 'monospace', color: 'text.secondary' }}
                                  >
                                    GET {group.endpoints.modelsUrl}
                                  </Typography>
                                  {supportsEmbeddings && (
                                    <Typography
                                      variant="caption"
                                      component="div"
                                      sx={{ fontFamily: 'monospace', color: 'text.secondary' }}
                                    >
                                      POST {group.endpoints.embeddingsUrl}
                                    </Typography>
                                  )}
                                </Box>
                              </React.Fragment>
                            ))}
                          </Box>
                        </TableBodyCell>
                      </TableRow>
                    </>
                  )}

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Running for</TableBodyCell>
                    <TableBodyCell>{formatDuration(deploymentDetail.duration)}</TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Engine</TableBodyCell>
                    <TableBodyCell>{deploymentDetail.engine_name}</TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Status</TableBodyCell>
                    <TableBodyCell>
                      {isStoppingDeployment ? (
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                          <CircularProgress size={16} />
                          <Box>
                            <Typography>Stopping...</Typography>
                            <Typography variant="caption" color="text.secondary">
                              This may take up to a minute
                            </Typography>
                          </Box>
                        </Box>
                      ) : (
                        <Box>
                          <Typography>{deploymentDetail.status}</Typography>
                          {deploymentDetail.status === 'MUST_REDOWNLOAD' && (
                            <Typography variant="caption" color="error.main">
                              One or more of your model files are missing. Please use Download to
                              download again.
                            </Typography>
                          )}
                          {deploymentDetail.status === 'EXAMPLE_STATUS' && (
                            <Typography variant="caption" color="text.secondary">
                              This is a long description for developers to understand how to add
                              more status explanations.
                            </Typography>
                          )}
                        </Box>
                      )}
                    </TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Autoscaling</TableBodyCell>
                    <TableBodyCell>
                      {deploymentDetail.autoscaling ? 'Enabled' : 'Disabled'}
                    </TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Min Copies</TableBodyCell>
                    <TableBodyCell>{deploymentDetail.min_copies}</TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Starting Copies</TableBodyCell>
                    <TableBodyCell>{deploymentDetail.starting_copies}</TableBodyCell>
                  </TableRow>

                  <TableRow>
                    <TableBodyCell sx={{ fontWeight: 600 }}>Max Copies</TableBodyCell>
                    <TableBodyCell>{deploymentDetail.max_copies || 'Unlimited'}</TableBodyCell>
                  </TableRow>

                  {Boolean(deploymentDetail.gpu_allocation) && (
                    <TableRow>
                      <TableBodyCell sx={{ fontWeight: 600 }}>GPU Allocation</TableBodyCell>
                      <TableBodyCell>
                        {(deploymentDetail.vram_allocation / (1000 * 1000 * 1000)).toFixed(2)} GB
                      </TableBodyCell>
                    </TableRow>
                  )}
                </TableBody>
              </StyledTable>
            </StyledTableContainer>

            <SectionTitle variant="h5">Instances</SectionTitle>

            <StyledTableContainer component={Paper}>
              <StyledTable>
                <TableHead>
                  <TableRow>
                    <TableHeaderCell>Host</TableHeaderCell>
                    <TableHeaderCell>Port</TableHeaderCell>
                    <TableHeaderCell>
                      {deploymentDetail.engine_name === 'llamacpp' ||
                      deploymentDetail.engine_name === 'mlx'
                        ? 'Process ID'
                        : 'Container ID'}
                    </TableHeaderCell>
                    <TableHeaderCell>Deployed At</TableHeaderCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {Array.isArray(deploymentDetail.instances) && deploymentDetail.instances.length > 0 ? (
                    deploymentDetail.instances.map(instance => {
                      const listenPortNumber = Number(instance.listen_port);
                      const listenPortDisplay = Number.isFinite(listenPortNumber) && listenPortNumber > 0
                        ? listenPortNumber
                        : '—';

                      return (
                        <TableRow key={instance.id || instance.container_id || instance.host_name}>
                          <TableBodyCell>{instance.host_name || '—'}</TableBodyCell>
                          <TableBodyCell>{listenPortDisplay}</TableBodyCell>
                          <TableBodyCell>{instance.container_id || '—'}</TableBodyCell>
                          <TableBodyCell>
                            {instance.deployed_at
                              ? format(new Date(instance.deployed_at), 'MM/dd/yyyy HH:mm')
                              : '—'}
                          </TableBodyCell>
                        </TableRow>
                      );
                    })
                  ) : (
                    <TableRow>
                      <TableBodyCell colSpan={4} align="center">
                        No instance details are available for this deployment yet.
                      </TableBodyCell>
                    </TableRow>
                  )}
                </TableBody>
              </StyledTable>
            </StyledTableContainer>

            {stopError && <ErrorText>{stopError}</ErrorText>}

            <Box sx={{ display: 'flex', justifyContent: 'center', gap: 2, mt: 2 }}>
              {runtimeInfo && runtimeInfo.baseUrl && (
                <ActionButton
                  color="primary"
                  onClick={() => setShowTestModal(true)}
                  variant="contained"
                  disabled={deploymentDetail.status !== 'DEPLOYED'}
                >
                  Test
                </ActionButton>
              )}
              <ActionButton
                color="error"
                onClick={() => setConfirmStop(true)}
                startIcon={<StopIcon />}
                disabled={isStoppingDeployment}
              >
                {isStoppingDeployment ? 'Stopping...' : 'Stop'}
              </ActionButton>

              {(deploymentDetail.status === 'STOP_REQUESTED' || stopError) && (
                <ActionButton
                  color="error"
                  onClick={() => setConfirmForceRemove(true)}
                  startIcon={<DeleteForeverIcon />}
                  disabled={isStoppingDeployment}
                >
                  Force Remove
                </ActionButton>
              )}
            </Box>
            </TabPanel>

            {/* Logs Tab */}
            <TabPanel className={activeTab === 1 ? 'active' : ''}>
              <ContainerLogViewer deploymentId={deploymentId} />
            </TabPanel>
          </>
        )}

        {/* Confirmation Modals */}
        <ConfirmModal open={confirmStop} onClose={() => setConfirmStop(false)}>
          <ConfirmModalContent>
            <Typography variant="h6" gutterBottom>
              Confirm Stop Deployment
            </Typography>
            <Typography variant="body1" paragraph>
              This will stop this model deployment. It will immediately become unavailable. Are you
              sure?
            </Typography>
            <Box sx={{ display: 'flex', justifyContent: 'center', gap: 2 }}>
              <ActionButton onClick={() => handleStopDeployment(true)} color="error">
                Yes, Stop
              </ActionButton>
              <ActionButton onClick={() => setConfirmStop(false)}>No, Cancel</ActionButton>
            </Box>
          </ConfirmModalContent>
        </ConfirmModal>

        <ConfirmModal open={confirmForceRemove} onClose={() => setConfirmForceRemove(false)}>
          <ConfirmModalContent>
            <Typography variant="h6" gutterBottom>
              Confirm Force Remove
            </Typography>
            <Typography variant="body1" paragraph>
              This will forcibly remove this model deployment from the list. If it&apos;s still running,
              it will be stopped. Are you sure?
            </Typography>
            <Box sx={{ display: 'flex', justifyContent: 'center', gap: 2 }}>
              <ActionButton onClick={() => handleStopDeployment(true)} color="error">
                Yes, Force Remove
              </ActionButton>
              <ActionButton onClick={() => setConfirmForceRemove(false)}>No, Cancel</ActionButton>
            </Box>
          </ConfirmModalContent>
        </ConfirmModal>

        {/* Test Conversation Modal */}
        <TestConversationModal
          open={showTestModal}
          onClose={() => setShowTestModal(false)}
          deployment={deploymentDetail}
        />
      </ModalContainer>
    </Modal>
  );
};

ModelDeploymentDetailModal.propTypes = {
  deploymentId: PropTypes.string,
  isOpen: PropTypes.bool.isRequired,
  onClose: PropTypes.func.isRequired,
  onDeploymentStopped: PropTypes.func,
};

export default ModelDeploymentDetailModal;
