import React, { useState, useCallback } from 'react';
import { styled } from '@mui/material/styles';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Modal,
  Tooltip,
  Chip,
} from '@mui/material';
import AddIcon from '@mui/icons-material/Add';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import SettingsIcon from '@mui/icons-material/Settings';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 2),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
}));

const ActionButton = styled(Button)(({ theme, color = 'primary' }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.8, 2),
  borderRadius: theme.shape.borderRadius,
  margin: theme.spacing(0.5),
  transition: 'all 0.2s',
  backgroundColor: color === 'primary' ? theme.palette.primary.main : theme.palette.secondary.main,
  color: theme.palette.common.white,
  '&:hover': {
    backgroundColor:
      color === 'primary' ? theme.palette.primary.dark : theme.palette.secondary.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
}));

const ViewConfigLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  marginRight: theme.spacing(2),
  display: 'inline-flex',
  alignItems: 'center',
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const ModalContainer = styled(Box)(({ theme }) => ({
  position: 'absolute',
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  width: 'auto',
  maxWidth: '90vw',
  backgroundColor: theme.palette.background.paper,
  boxShadow: theme.shadows[5],
  padding: theme.spacing(4),
  borderRadius: 12,
  maxHeight: '90vh',
  overflow: 'auto',
}));

const PreData = styled('pre')(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.1)',
  padding: theme.spacing(2),
  borderRadius: 8,
  overflow: 'auto',
  fontSize: '0.9rem',
  color: theme.palette.text.secondary,
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const ConfigChip = styled(Chip)(({ theme, isdefault }) => ({
  fontSize: '0.75rem',
  height: '24px',
  backgroundColor: isdefault === 'true' ? theme.palette.success.main : theme.palette.info.main,
  color: 'white',
  fontWeight: 500,
}));

const ModelConfigList = ({
  handleAddConfig,
  triggerRefreshDeployments,
  setParentModelConfigs,
  modelConfigs,
  isModelDownloading = false,
}) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [showVramModal, setShowVramModal] = useState(false);
  const [vramEstimate, setVramEstimate] = useState(null);
  const { model_id } = useParams();

  const deployConfig = useCallback(
    async configId => {
      setLoading(true);
      try {
        // First fetch the config details (as in the original code)
        console.log('Fetching config details for ID:', configId);
        const configDetailResponse = await axios.get(`${BASE_URL}/model_configs/${configId}`);
        const configDetail = configDetailResponse.data;
        console.log('Config detail data:', configDetail);

        // Normalize config data (could be stringified JSON)
        let configData = configDetail?.config;
        if (typeof configData === 'string') {
          try {
            configData = JSON.parse(configData);
          } catch (parseErr) {
            console.warn('Unable to parse config JSON for deployment payload:', parseErr);
          }
        }

        // Create payload for deployment
        const deployPayload = {
          m_id: model_id,
          m_config_id: configId,
        };

        const engineName = configData?.engine_name;
        if (engineName) {
          deployPayload.engine_name = engineName;
        }
        console.log('Deploying model with payload:', deployPayload);

        // Then deploy the model with the config
        const deployResponse = await axios.post(`${BASE_URL}/serving/deploy_model`, deployPayload);
        console.log('Deployment response:', deployResponse.data);

        // Wait a moment to allow the deployment to initialize
        setTimeout(() => {
          console.log('Refreshing deployments after timeout');
          triggerRefreshDeployments(); // Refresh the deployment list after successful deployment
          setLoading(false);
        }, 2000); // 2 second delay
      } catch (err) {
        console.error('Failed to deploy model config:', err);
        if (err.response) {
          console.error('Deployment error response:', err.response.data);
        }
        setError(err);
        setLoading(false);
      }
    },
    [model_id, triggerRefreshDeployments]
  );

  const estimateVram = useCallback(
    async configId => {
      try {
        const response = await axios.post(`${BASE_URL}/serving/estimate_model_vram`, {
          m_id: model_id,
          m_config_id: configId,
        });
        setVramEstimate(JSON.stringify(response.data, null, 2));
        setShowVramModal(true);
      } catch (err) {
        console.error('Failed to estimate VRAM:', err);
        // Handle error (e.g., show error notification)
      }
    },
    [model_id]
  );

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error?.message || 'An error occurred'} />;

  return (
    <Box sx={{ mb: 4 }}>
      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
        <SectionTitle variant="h4">
          Model Configurations
          <Tooltip title="Model configurations define runtime settings for model deployments">
            <InfoIcon />
          </Tooltip>
        </SectionTitle>
      </Box>

      {modelConfigs.length > 0 ? (
        <StyledTableContainer component={Paper}>
          <StyledTable>
            <TableHead>
              <TableRow>
                <TableHeaderCell>Name</TableHeaderCell>
                <TableHeaderCell>Description</TableHeaderCell>
                <TableHeaderCell>Actions</TableHeaderCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {modelConfigs.map(config => {
                // Debug log for each config
                console.log(`Config "${config.name}" in ModelConfigList:`, {
                  id: config.id,
                  name: config.name,
                  config: config.config,
                  system_config: config.system_config,
                  all_keys: Object.keys(config),
                });
                return (
                  <StyledTableRow key={config.id}>
                    <TableBodyCell>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                        {config.name}
                        {config.default && (
                          <ConfigChip label="Default" size="small" isdefault="true" />
                        )}
                      </Box>
                    </TableBodyCell>
                    <TableBodyCell>
                      <Box>
                        {config.description && (
                          <Typography variant="body2" sx={{ mb: 0.5 }}>
                            {config.description}
                          </Typography>
                        )}
                        <Box
                          sx={{
                            display: 'flex',
                            gap: 1,
                            flexWrap: 'wrap',
                            fontSize: '0.75rem', // Reduced from 0.85rem
                            color: 'text.secondary',
                          }}
                        >
                          {config.engine_name && <span>Engine: {config.engine_name}</span>}
                          {/* Display custom config fields */}
                          {config.config && Object.keys(config.config).length > 0 && (
                            <>
                              {Object.entries(config.config)
                                .slice(0, 2)
                                .map(([key, value]) => (
                                  <span key={`config-${key}`}>
                                    {key}:{' '}
                                    {typeof value === 'boolean'
                                      ? value
                                        ? 'Yes'
                                        : 'No'
                                      : String(value)}
                                  </span>
                                ))}
                              {Object.keys(config.config).length > 2 && (
                                <span style={{ fontStyle: 'italic' }}>
                                  +{Object.keys(config.config).length - 2} more
                                </span>
                              )}
                            </>
                          )}
                          {/* Also check system_config */}
                          {config.system_config && Object.keys(config.system_config).length > 0 && (
                            <>
                              {Object.entries(config.system_config)
                                .slice(0, 2)
                                .map(([key, value]) => (
                                  <span key={`sys-${key}`}>
                                    {key}:{' '}
                                    {typeof value === 'boolean'
                                      ? value
                                        ? 'Yes'
                                        : 'No'
                                      : String(value)}
                                  </span>
                                ))}
                              {Object.keys(config.system_config).length > 2 && (
                                <span style={{ fontStyle: 'italic' }}>
                                  +{Object.keys(config.system_config).length - 2} more system
                                </span>
                              )}
                            </>
                          )}
                        </Box>
                      </Box>
                    </TableBodyCell>
                    <TableBodyCell>
                      <Box sx={{ display: 'flex', alignItems: 'center' }}>
                        <ViewConfigLink to={`/model_configs/${config.id}`}>
                          <SettingsIcon fontSize="small" sx={{ mr: 0.5 }} />
                          View Config
                        </ViewConfigLink>

                        <Tooltip
                          title={
                            isModelDownloading
                              ? 'Wait for model files to finish downloading before deploying'
                              : 'Deploy this model configuration'
                          }
                        >
                          <span>
                            <ActionButton
                              onClick={() => deployConfig(config.id)}
                              size="small"
                              startIcon={<RocketLaunchIcon />}
                              disabled={isModelDownloading}
                            >
                              Deploy
                            </ActionButton>
                          </span>
                        </Tooltip>
                      </Box>
                    </TableBodyCell>
                  </StyledTableRow>
                );
              })}
            </TableBody>
          </StyledTable>
        </StyledTableContainer>
      ) : (
        <Typography variant="body1" color="text.secondary" sx={{ my: 2 }}>
          No model configurations found for this model.
        </Typography>
      )}

      <ActionButton onClick={handleAddConfig} startIcon={<AddIcon />}>
        New Config for Model
      </ActionButton>

      <Modal
        open={showVramModal}
        onClose={() => setShowVramModal(false)}
        aria-labelledby="vram-estimate-modal-title"
      >
        <ModalContainer>
          <Typography id="vram-estimate-modal-title" variant="h6" component="h2" gutterBottom>
            VRAM Estimate
          </Typography>
          <PreData>{vramEstimate}</PreData>
          <ActionButton onClick={() => setShowVramModal(false)} sx={{ mt: 2 }}>
            Close
          </ActionButton>
        </ModalContainer>
      </Modal>
    </Box>
  );
};

export default ModelConfigList;
