import React from 'react';
import { StyledTableCell, StyledTextField, StyledTooltipText, StyledButton } from '../../StyledComponents';

export const ConfigFieldRenderer = ({ config, tooltips, section, handleInputChange }) => {
    return Object.entries(config).map(([key, value]) => {
        return (
            <React.Fragment key={key}>
                <tr>
                    <StyledTableCell>{key}</StyledTableCell>
                    <StyledTableCell>
                        {typeof value === 'boolean' || value === null ? (
                            <select
                                name={key}
                                value={value === null ? '' : value.toString()}
                                onChange={(event) => handleInputChange(event, section)}
                            >
                                <option value="" disabled>Select...</option>
                                <option value="true">Yes</option>
                                <option value="false">No</option>
                            </select>
                        ) : (
                            <StyledTextField
                                name={key}
                                value={value}
                                onChange={(event) => handleInputChange(event, section)}
                                placeholder={section === 'general' ? 'Enter value or leave blank for default' : ''}
                            />
                        )}
                    </StyledTableCell>
                </tr>
                {tooltips[key] && (
                    <tr>
                        <StyledTableCell colSpan="2">
                            <StyledTooltipText>{tooltips[key]}</StyledTooltipText>
                        </StyledTableCell>
                    </tr>
                )}
            </React.Fragment>
        );
    });
};

export const CustomConfigFieldRenderer = ({ config, handleInputChange, handleRemoveCustomConfig }) => {
    return Object.entries(config).map(([key, value]) => {
        return (
            <React.Fragment key={key}>
                <tr>
                    <StyledTableCell>{key}</StyledTableCell>
                    <StyledTableCell style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                        <StyledTextField
                            name={key}
                            value={value}
                            onChange={(event) => handleInputChange(event, 'custom')}
                            style={{ flex: 1, marginRight: '10px' }}
                        />
                        <StyledButton 
                            variant="secondary" 
                            onClick={() => handleRemoveCustomConfig(key)}
                            style={{ padding: '5px 10px', minWidth: 'auto' }}
                        >
                            Remove
                        </StyledButton>
                    </StyledTableCell>
                </tr>
            </React.Fragment>
        );
    });
};

export const GgufDropdown = ({ showGgufDropdown, selectedModelFile, setSelectedModelFile, modelFiles, truncateFileName }) => {
    if (!showGgufDropdown) return null;
    
    return (
        <tr>
            <StyledTableCell>Model File (for GGUF)</StyledTableCell>
            <StyledTableCell>
                <select 
                    value={selectedModelFile} 
                    onChange={(e) => setSelectedModelFile(e.target.value)} 
                    disabled={modelFiles.length === 0}
                >
                    {modelFiles.length === 0 ? (
                        <option value="" disabled>No downloaded .gguf files</option>
                    ) : (
                        modelFiles.map(file => (
                            <option key={file.id} value={file.id}>
                                {truncateFileName(file.name)}
                            </option>
                        ))
                    )}
                </select>
                {modelFiles.length === 0 && (
                    <StyledTooltipText>
                        Download a .gguf file for this model to enable selection.
                    </StyledTooltipText>
                )}
            </StyledTableCell>
        </tr>
    );
};

// Configurable filename truncation length
const DEFAULT_FILENAME_MAX_LENGTH = 30;

export const truncateFileName = (fileName, maxLength = DEFAULT_FILENAME_MAX_LENGTH) => {
    // Remove path parts before the last '/'
    let truncated = fileName.split('/').pop();
    
    // Remove leading zeros from both halves of the "000??-of-000??" pattern
    truncated = truncated.replace(/0*(\d+)-of-0*(\d+)/g, '$1-of-$2');
    
    // Remove .gguf extension if present
    truncated = truncated.replace(/\.gguf$/, '');
    
    // If still over maxLength characters, truncate from the front
    if (truncated.length > maxLength) {
        const ellipsis = '(...)';
        const keepLength = maxLength - ellipsis.length;
        truncated = ellipsis + truncated.slice(-keepLength);
    }
    
    return truncated;
};