import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import axios from 'axios';
import { useParams, Link, useNavigate } from 'react-router-dom';
import { 
  Typography, 
  Box, 
  Paper, 
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableRow,
  Button
} from '@mui/material';
import ArrowBackIosIcon from '@mui/icons-material/ArrowBackIos';
import DeleteIcon from '@mui/icons-material/Delete';
import EditIcon from '@mui/icons-material/Edit';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import EditModelConfigForm from './EditModelConfigForm';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  borderRadius: 8,
  padding: theme.spacing(1, 2),
  textTransform: 'none',
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  color: theme.palette.primary.main,
  border: `1px solid ${theme.palette.primary.main}`,
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.2)',
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
}));

const DeleteButton = styled(Button)(({ theme }) => ({
  marginTop: theme.spacing(2),
  borderRadius: 8,
  padding: theme.spacing(1, 2),
  textTransform: 'none',
  backgroundColor: theme.palette.error.main,
  color: theme.palette.common.white,
  '&:hover': {
    backgroundColor: theme.palette.error.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
}));

const EditButton = styled(Button)(({ theme }) => ({
  marginTop: theme.spacing(2),
  marginRight: theme.spacing(2),
  borderRadius: 8,
  padding: theme.spacing(1, 2),
  textTransform: 'none',
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.common.white,
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
}));

const ModelConfigDetail = () => {
  const [modelConfig, setModelConfig] = useState(null);
  const [modelFile, setModelFile] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [showEditModal, setShowEditModal] = useState(false);
  const [, setModelConfigs] = useState([]);
  const { model_config_id } = useParams();
  const navigate = useNavigate();

  useEffect(() => {
    const fetchModelConfigDetail = async () => {
      setLoading(true);
      try {
        const response = await axios.get(`${BASE_URL}/model_configs/${model_config_id}`);
        setModelConfig(response.data);
        if (response.data.m_file_id) {
          const fileResponse = await axios.get(`${BASE_URL}/model_files/${response.data.m_file_id}`);
          setModelFile(fileResponse.data);
        }
        setLoading(false);
      } catch (err) {
        setError(err);
        setLoading(false);
      }
    };

    fetchModelConfigDetail();
  }, [model_config_id]);

  const handleDelete = async () => {
    if (window.confirm('Are you sure you want to delete this configuration?')) {
      setLoading(true);
      try {
        await axios.delete(`${BASE_URL}/model_configs/${model_config_id}`);
        navigate(`/models/${modelConfig?.m_id}`);
      } catch (err) {
        setError(err);
        setLoading(false);
      }
    }
  };

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error.message} />;

  const rawExternalConfig = modelConfig?.config?.external_endpoint;
  let externalConfig = null;
  if (rawExternalConfig) {
    if (typeof rawExternalConfig === 'string') {
      try {
        externalConfig = JSON.parse(rawExternalConfig);
      } catch (parseError) {
        console.warn('Failed to parse external endpoint config:', parseError);
      }
    } else if (typeof rawExternalConfig === 'object') {
      externalConfig = rawExternalConfig;
    }
  }

  const extraBodyDisplay = externalConfig
    ? typeof externalConfig.extra_body === 'string'
      ? externalConfig.extra_body
      : JSON.stringify(externalConfig.extra_body || {}, null, 2)
    : '';

  let configContent;

  if (!modelConfig) {
    configContent = (
      <Typography variant="body1" color="text.secondary">
        Model Config not found
      </Typography>
    );
  } else if (externalConfig) {
    configContent = (
      <>
        <SectionTitle variant="h4">External Endpoint Configuration</SectionTitle>
        <StyledTableContainer component={Paper}>
          <StyledTable>
            <TableBody>
              <TableRow>
                <TableBodyCell sx={{ width: '30%', fontWeight: 600 }}>Service</TableBodyCell>
                <TableBodyCell>{externalConfig.service || 'AWS Bedrock'}</TableBodyCell>
              </TableRow>
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Display Name</TableBodyCell>
                <TableBodyCell>{externalConfig.display_name || modelConfig.name}</TableBodyCell>
              </TableRow>
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Model / Profile ID</TableBodyCell>
                <TableBodyCell>{externalConfig.model_id || externalConfig.inference_profile_id}</TableBodyCell>
              </TableRow>
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Region</TableBodyCell>
                <TableBodyCell>{externalConfig.region || '—'}</TableBodyCell>
              </TableRow>
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Endpoint URL</TableBodyCell>
                <TableBodyCell>{externalConfig.endpoint_url || 'Default Bedrock runtime'}</TableBodyCell>
              </TableRow>
              {externalConfig.credential_secret_urn && (
                <TableRow>
                  <TableBodyCell sx={{ fontWeight: 600 }}>Credential Secret</TableBodyCell>
                  <TableBodyCell>{externalConfig.credential_secret_urn}</TableBodyCell>
                </TableRow>
              )}
              {externalConfig.inference_profile_arn && (
                <TableRow>
                  <TableBodyCell sx={{ fontWeight: 600 }}>Profile ARN</TableBodyCell>
                  <TableBodyCell>{externalConfig.inference_profile_arn}</TableBodyCell>
                </TableRow>
              )}
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Engine</TableBodyCell>
                <TableBodyCell>aws_bedrock</TableBodyCell>
              </TableRow>
            </TableBody>
          </StyledTable>
        </StyledTableContainer>

        <SectionTitle variant="h5">Extra Body</SectionTitle>
        <Paper sx={{ p: 2, backgroundColor: 'rgba(255, 255, 255, 0.03)' }}>
          <Typography component="pre" sx={{ whiteSpace: 'pre-wrap', fontFamily: 'monospace', fontSize: '0.85rem' }}>
            {extraBodyDisplay || '{}'}
          </Typography>
        </Paper>
        {!modelConfig.default && (
          <Box sx={{ display: 'flex', alignItems: 'center', mt: 3 }}>
            <EditButton 
              variant="contained" 
              startIcon={<EditIcon />}
              onClick={() => setShowEditModal(true)}
            >
              Edit Config
            </EditButton>
            <DeleteButton 
              variant="contained" 
              startIcon={<DeleteIcon />}
              onClick={handleDelete}
            >
              Delete Config
            </DeleteButton>
          </Box>
        )}
      </>
    );
  } else {
    configContent = (
      <>
        <SectionTitle variant="h4">Configuration Details</SectionTitle>
        
        <StyledTableContainer component={Paper}>
          <StyledTable>
            <TableBody>
              <TableRow>
                <TableBodyCell sx={{ width: '30%', fontWeight: 600 }}>Name</TableBodyCell>
                <TableBodyCell>{modelConfig.name}</TableBodyCell>
              </TableRow>
              <TableRow>
                <TableBodyCell sx={{ fontWeight: 600 }}>Description</TableBodyCell>
                <TableBodyCell>{modelConfig.description}</TableBodyCell>
              </TableRow>
              {modelFile && (
                <TableRow>
                  <TableBodyCell sx={{ fontWeight: 600 }}>Model File</TableBodyCell>
                  <TableBodyCell>{modelFile.name}</TableBodyCell>
                </TableRow>
              )}
              {modelConfig.config && Object.entries(modelConfig.config).map(([key, value]) => (
                <TableRow key={key}>
                  <TableBodyCell sx={{ fontWeight: 600 }}>{key}</TableBodyCell>
                  <TableBodyCell>
                    {typeof value === 'boolean' ? (value ? 'Yes' : 'No') : value}
                  </TableBodyCell>
                </TableRow>
              ))}
            </TableBody>
          </StyledTable>
        </StyledTableContainer>
        
        {!modelConfig.default && (
          <Box sx={{ display: 'flex', alignItems: 'center' }}>
            <EditButton 
              variant="contained" 
              startIcon={<EditIcon />}
              onClick={() => setShowEditModal(true)}
            >
              Edit Config
            </EditButton>
            <DeleteButton 
              variant="contained" 
              startIcon={<DeleteIcon />}
              onClick={handleDelete}
            >
              Delete Config
            </DeleteButton>
          </Box>
        )}
      </>
    );
  }

  return (
    <ContentContainer>
      <BackButton 
        component={Link} 
        to={`/models/${modelConfig?.m_id}`} 
        startIcon={<ArrowBackIosIcon />}
      >
        Back to Config List
      </BackButton>
      
      {configContent}
      
      {modelConfig && (
        <EditModelConfigForm
          showModal={showEditModal}
          closeModal={() => setShowEditModal(false)}
          configId={model_config_id}
          m_id={modelConfig.m_id}
          setParentModelConfigs={(updater) => {
            if (typeof updater === 'function') {
              setModelConfigs(updater);
              // Refresh the current config data
              const fetchUpdatedConfig = async () => {
                try {
                  const response = await axios.get(`${BASE_URL}/model_configs/${model_config_id}`);
                  setModelConfig(response.data);
                } catch (err) {
                  console.error('Error refreshing config:', err);
                }
              };
              fetchUpdatedConfig();
            }
          }}
        />
      )}
    </ContentContainer>
  );
};

export default ModelConfigDetail;
