import React from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Alert,
  AlertTitle,
  Box,
  Chip,
  Typography,
  Accordion,
  AccordionSummary,
  AccordionDetails,
  List,
  ListItem,
  ListItemText,
} from '@mui/material';
import {
  Warning,
  Error,
  Memory,
  Storage,
  ExpandMore,
} from '@mui/icons-material';

const StyledAlert = styled(Alert)(({ theme, severity }) => ({
  marginBottom: theme.spacing(2),
  borderRadius: 12,
  '& .MuiAlert-icon': {
    fontSize: '1.5rem',
  },
  ...(severity === 'error' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    border: '1px solid rgba(244, 67, 54, 0.3)',
  }),
  ...(severity === 'warning' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    border: '1px solid rgba(255, 152, 0, 0.3)',
  }),
}));

const PatternChip = styled(Chip)(({ theme, pattern }) => ({
  margin: theme.spacing(0.25),
  fontWeight: 500,
  ...(pattern === 'oom_error' && {
    backgroundColor: 'rgba(244, 67, 54, 0.15)',
    color: '#f44336',
  }),
  ...(pattern === 'cuda_error' && {
    backgroundColor: 'rgba(255, 152, 0, 0.15)',
    color: '#ff9800',
  }),
  ...(pattern === 'model_loading_failure' && {
    backgroundColor: 'rgba(156, 39, 176, 0.15)',
    color: '#9c27b0',
  }),
  ...(pattern === 'container_crash' && {
    backgroundColor: 'rgba(33, 33, 33, 0.15)',
    color: '#666',
  }),
}));

const FailureLine = styled(ListItem)(({ theme }) => ({
  backgroundColor: 'rgba(244, 67, 54, 0.05)',
  borderLeft: '3px solid #f44336',
  marginBottom: theme.spacing(0.5),
  borderRadius: theme.spacing(0.5),
  fontFamily: 'monospace',
  fontSize: '0.85rem',
}));

const getPatternIcon = (patternType) => {
  switch (patternType) {
    case 'oom_error':
      return <Memory />;
    case 'cuda_error':
      return <Warning />;
    case 'model_loading_failure':
      return <Storage />;
    case 'container_crash':
      return <Error />;
    default:
      return <Warning />;
  }
};

const getPatternLabel = (patternType) => {
  switch (patternType) {
    case 'oom_error':
      return 'Out of Memory';
    case 'cuda_error':
      return 'CUDA Error';
    case 'model_loading_failure':
      return 'Model Loading Failure';
    case 'container_crash':
      return 'Container Crash';
    default:
      return patternType;
  }
};

/**
 * Component to display detected failure patterns in deployment logs
 * @param {Object} props - Component props
 * @param {Object} props.patterns - Pattern analysis data from API
 * @param {boolean} props.loading - Loading state
 * @param {string} props.error - Error message if any
 */
const LogPatternAlert = ({ patterns, loading, error }) => {
  if (loading || error || !patterns) {
    return null;
  }

  const { patterns_detected, failure_lines } = patterns;
  const detectedPatterns = Object.entries(patterns_detected).filter(([, detected]) => detected);

  if (detectedPatterns.length === 0) {
    return (
      <StyledAlert severity="success" sx={{ mb: 2 }}>
        <AlertTitle>No Issues Detected</AlertTitle>
        Logs appear healthy with no known failure patterns detected.
      </StyledAlert>
    );
  }

  const hasErrorPatterns = detectedPatterns.some(([pattern]) => 
    ['oom_error', 'container_crash'].includes(pattern)
  );
  
  const alertSeverity = hasErrorPatterns ? 'error' : 'warning';

  return (
    <StyledAlert severity={alertSeverity}>
      <AlertTitle>
        {detectedPatterns.length} Issue{detectedPatterns.length > 1 ? 's' : ''} Detected
      </AlertTitle>
      
      <Box sx={{ mb: 2 }}>
        <Typography variant="body2" sx={{ mb: 1 }}>
          Detected failure patterns:
        </Typography>
        <Box>
          {detectedPatterns.map(([pattern]) => (
            <PatternChip
              key={pattern}
              pattern={pattern}
              icon={getPatternIcon(pattern)}
              label={getPatternLabel(pattern)}
              size="small"
            />
          ))}
        </Box>
      </Box>

      {failure_lines && failure_lines.length > 0 && (
        <Accordion 
          elevation={0}
          sx={{ 
            backgroundColor: 'transparent',
            '&:before': { display: 'none' }
          }}
        >
          <AccordionSummary 
            expandIcon={<ExpandMore />}
            sx={{ padding: 0, minHeight: 'auto' }}
          >
            <Typography variant="body2" color="text.secondary">
              View {failure_lines.length} failure line{failure_lines.length > 1 ? 's' : ''}
            </Typography>
          </AccordionSummary>
          <AccordionDetails sx={{ padding: 0, pt: 1 }}>
            <List dense sx={{ width: '100%' }}>
              {failure_lines.map((failureLine, index) => (
                <FailureLine key={index} disablePadding>
                  <ListItemText
                    primary={
                      <Box>
                        <Typography
                          variant="caption"
                          sx={{ 
                            color: 'text.secondary', 
                            display: 'block',
                            mb: 0.5 
                          }}
                        >
                          Line {failureLine.line_number} ({getPatternLabel(failureLine.pattern_type)}):
                        </Typography>
                        <Typography
                          variant="body2"
                          sx={{ 
                            fontFamily: 'monospace',
                            fontSize: '0.8rem',
                            wordBreak: 'break-word',
                            whiteSpace: 'pre-wrap'
                          }}
                        >
                          {failureLine.content}
                        </Typography>
                      </Box>
                    }
                  />
                </FailureLine>
              ))}
            </List>
          </AccordionDetails>
        </Accordion>
      )}
    </StyledAlert>
  );
};

LogPatternAlert.propTypes = {
  patterns: PropTypes.shape({
    patterns_detected: PropTypes.object,
    failure_lines: PropTypes.arrayOf(
      PropTypes.shape({
        line_number: PropTypes.string,
        content: PropTypes.string,
        pattern_type: PropTypes.string,
      })
    ),
  }),
  loading: PropTypes.bool,
  error: PropTypes.string,
};

export default LogPatternAlert;