// EditModelConfigForm.js
import React from 'react';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { StyledModal, WrappedTextDescStyledDiv, StyledButton, StyledTable, StyledTableCell, StyledTextField, StyledModalContent, StyledSubHeader, StyledCollapseButton, StyledParagraph } from '../../StyledComponents';
import { useModelConfigForm } from '../../hooks/useModelConfigForm';
import { ConfigFieldRenderer, CustomConfigFieldRenderer, GgufDropdown, truncateFileName } from './ModelConfigFormComponents';

const EditModelConfigForm = ({ closeModal, showModal, configId, m_id, setParentModelConfigs }) => {
    const {
        // State
        modelConfigData,
        showVLLMFields,
        setShowVLLMFields,
        showLlamaCPPFields,
        setShowLlamaCPPFields,
        showMLXFields,
        setShowMLXFields,
        newConfigKey,
        setNewConfigKey,
        newConfigValue,
        setNewConfigValue,
        modelFiles,
        selectedModelFile,
        setSelectedModelFile,
        hasGgufFilesForSection,
        hasSafetensorsFilesForSection,
        
        // Server info
        serverOs,
        isLoadingServerOs,
        serverOsError,
        error,
        setError,
        
        // Derived states
        useFallbackDisplayOnDarwin,
        canShowVLLM,
        showLlamaCPPSectionForUI,
        showMLXSectionForUI,
        showGgufDropdown,
        isButtonDisabled,
        showMainFormContent,
        
        // Functions
        handleInputChange,
        handleAddCustomConfig,
        handleRemoveCustomConfig,
        validateAndPreparePayload,
    } = useModelConfigForm({ showModal, m_id, configId, isEditMode: true });

    const handleUpdateModelConfig = async () => {
        const modelConfigPayload = validateAndPreparePayload();
        if (!modelConfigPayload) return;

        try {
            const response = await axios.put(`${BASE_URL}/model_configs/${configId}`, modelConfigPayload);
            if (response.status === 200) {
                setParentModelConfigs((prevConfigs) => 
                    prevConfigs.map(config => config.id === configId ? response.data : config)
                );
                closeModal();
            } else {
                setError(`Failed to update model config. Status: ${response.status}`);
            }
        } catch (err) {
            console.error('Update model config error:', err);
            setError(`Failed to update model config: ${err.response?.data?.detail || err.message || 'Unknown error'}`);
        }
    };


    if (isLoadingServerOs) return <StyledModal open={showModal} onClose={closeModal}><StyledModalContent><StyledParagraph>Loading server information...</StyledParagraph></StyledModalContent></StyledModal>;
    if (serverOsError) return <StyledModal open={showModal} onClose={closeModal}><StyledModalContent><StyledParagraph>Error: {serverOsError}</StyledParagraph><StyledButton variant="secondary" onClick={closeModal}>Close</StyledButton></StyledModalContent></StyledModal>;

    return (
        <StyledModal open={showModal} onClose={closeModal}>
            <StyledModalContent>
                {error && (
                    <div style={{ backgroundColor: '#f44336', color: 'white', padding: '10px', marginBottom: '10px', borderRadius: '4px' }}>
                        {error}
                        <button onClick={() => setError(null)} style={{ float: 'right', background: 'transparent', border: 'none', color: 'white', cursor: 'pointer' }}>×</button>
                    </div>
                )}
                {!isLoadingServerOs && !showMainFormContent && serverOs === 'darwin' && (
                    <StyledParagraph>This model does not appear to have .gguf or .safetensors files suitable for configuration on macOS. You can add custom fields if needed, or check the model files.</StyledParagraph>
                )}
                {!isLoadingServerOs && !showMainFormContent && serverOs === 'linux' && (
                     <StyledParagraph>This model does not appear to have files suitable for configuration on Linux (e.g. GGUF for Llama.cpp or compatible formats for vLLM). You can add custom fields if needed, or check the model files.</StyledParagraph>
                )}

                {showMainFormContent ? (
                    <StyledTable>
                        <thead>
                            <tr>
                                <StyledTableCell colSpan="2">
                                    <StyledSubHeader>Edit Model Configuration</StyledSubHeader>
                                    <WrappedTextDescStyledDiv style={{ width: '600px' }}>You must supply Name and Description for this config. The other fields are optional, and we will use defaults for values you do not provide.
                                        Note: some config values overlap! Something like n_ctx that is used in both engines cannot be set twice even if set it in both sections.</WrappedTextDescStyledDiv>
                                </StyledTableCell>
                            </tr>
                        </thead>
                        <tbody>
                            <GgufDropdown 
                                showGgufDropdown={showGgufDropdown}
                                selectedModelFile={selectedModelFile}
                                setSelectedModelFile={setSelectedModelFile}
                                modelFiles={modelFiles}
                                truncateFileName={truncateFileName}
                            />
                            <tr><StyledTableCell colSpan="2"><StyledSubHeader>General</StyledSubHeader></StyledTableCell></tr>
                            <ConfigFieldRenderer 
                                config={modelConfigData.general} 
                                tooltips={modelConfigData.tooltips} 
                                section="general" 
                                handleInputChange={handleInputChange} 
                            />
                            
                            <>
                                {canShowVLLM && (
                                    <>
                                        <tr><StyledTableCell colSpan="2"><StyledSubHeader>vLLM<StyledCollapseButton onClick={() => setShowVLLMFields(!showVLLMFields)}>{showVLLMFields ? '▼' : '▶'}</StyledCollapseButton></StyledSubHeader></StyledTableCell></tr>
                                        {showVLLMFields && (
                                            <ConfigFieldRenderer 
                                                config={modelConfigData.vLLM} 
                                                tooltips={modelConfigData.tooltips} 
                                                section="vLLM" 
                                                handleInputChange={handleInputChange} 
                                            />
                                        )}
                                    </>
                                )}
                                {showLlamaCPPSectionForUI && (
                                    <>
                                        <tr><StyledTableCell colSpan="2"><StyledSubHeader>Llama.cpp
                                            {serverOs === 'darwin' && !useFallbackDisplayOnDarwin && hasGgufFilesForSection && <span style={{ fontSize: '0.8rem', marginLeft: '10px', fontWeight: 'normal' }}>(for .gguf files)</span>}
                                            <StyledCollapseButton onClick={() => setShowLlamaCPPFields(!showLlamaCPPFields)}>{showLlamaCPPFields ? '▼' : '▶'}</StyledCollapseButton>
                                        </StyledSubHeader></StyledTableCell></tr>
                                        {showLlamaCPPFields && (
                                            <ConfigFieldRenderer 
                                                config={modelConfigData.llamaCPP} 
                                                tooltips={modelConfigData.tooltips} 
                                                section="llamaCPP" 
                                                handleInputChange={handleInputChange} 
                                            />
                                        )}
                                    </>
                                )}
                                {showMLXSectionForUI && (
                                    <>
                                        <tr><StyledTableCell colSpan="2"><StyledSubHeader>MLX
                                            {serverOs === 'darwin' && !useFallbackDisplayOnDarwin && hasSafetensorsFilesForSection && <span style={{ fontSize: '0.8rem', marginLeft: '10px', fontWeight: 'normal' }}>(for .safetensors files)</span>}
                                            <StyledCollapseButton onClick={() => setShowMLXFields(!showMLXFields)}>{showMLXFields ? '▼' : '▶'}</StyledCollapseButton>
                                        </StyledSubHeader></StyledTableCell></tr>
                                        {showMLXFields && (
                                            <ConfigFieldRenderer 
                                                config={modelConfigData.mlx} 
                                                tooltips={modelConfigData.tooltips} 
                                                section="mlx" 
                                                handleInputChange={handleInputChange} 
                                            />
                                        )}
                                    </>
                                )}
                                <tr><StyledTableCell colSpan="2"><StyledSubHeader>Custom Fields</StyledSubHeader></StyledTableCell></tr>
                                <CustomConfigFieldRenderer 
                                    config={modelConfigData.custom} 
                                    handleInputChange={handleInputChange} 
                                    handleRemoveCustomConfig={handleRemoveCustomConfig}
                                />
                                <tr>
                                    <StyledTableCell>
                                        <StyledTextField value={newConfigKey} onChange={(e) => setNewConfigKey(e.target.value)} placeholder="Enter new config key" />
                                    </StyledTableCell>
                                    <StyledTableCell style={{ display: 'flex', alignItems: 'center' }}>
                                        <StyledTextField value={newConfigValue} onChange={(e) => setNewConfigValue(e.target.value)} placeholder="Enter new config value" style={{ marginRight: '10px' }} />
                                        <StyledButton variant="secondary" onClick={handleAddCustomConfig}>+ Add Config Field</StyledButton>
                                    </StyledTableCell>
                                </tr>
                            </>
                        </tbody>
                    </StyledTable>
                ) : null }

                <StyledButton variant="secondary" onClick={closeModal}>Close</StyledButton>
                <StyledButton variant="primary" onClick={handleUpdateModelConfig} disabled={isButtonDisabled}>
                    Update Model Config
                </StyledButton>
            </StyledModalContent>
        </StyledModal>
    );
};

export default EditModelConfigForm;