import React, { useContext, useState, useMemo, useEffect } from 'react';
import {
  StyledFloatingWindow,
  StyledDownloadStatusModalContent,
  StyledButton,
  StyledDownloadProgressBar,
  StyledDownloadStatusText,
  StyledDownloadStatusItem,
} from '../../StyledComponents';
import ModelFilesContext from './ModelFilesContext';
import { FaChevronUp, FaChevronDown } from 'react-icons/fa';
import {
  Box,
  Typography,
  Chip,
  IconButton,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  CircularProgress,
  Collapse,
} from '@mui/material';
import CancelIcon from '@mui/icons-material/Cancel';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import ExpandLessIcon from '@mui/icons-material/ExpandLess';
import Toast from '../common/Toast';

const DownloadStatusModal = ({ closeModal }) => {
  const {
    statusModelFiles,
    clearCompletedDownloads,
    cancelDownload,
    cancelAllDownloads,
    toast,
    closeToast,
  } = useContext(ModelFilesContext);
  const [isExpanded, setIsExpanded] = useState(false);
  const [confirmDialog, setConfirmDialog] = useState({ open: false, fileId: null, fileName: '' });
  const [confirmCancelAllDialog, setConfirmCancelAllDialog] = useState(false);
  const [cancellingFiles, setCancellingFiles] = useState(new Set());
  const [cancellingAll, setCancellingAll] = useState(false);
  const [completedFiles, setCompletedFiles] = useState([]);
  const [allSeenFiles, setAllSeenFiles] = useState(new Map());
  const [isActiveExpanded, setIsActiveExpanded] = useState(true);
  const [isCompletedExpanded, setIsCompletedExpanded] = useState(false);
  const [autoCollapseTimer, setAutoCollapseTimer] = useState(null);

  const handleOnClose = (event, reason) => {
    if (reason !== 'backdropClick' && reason !== 'escapeKeyDown') {
      closeModal();
    }
  };

  const toggleExpand = () => {
    setIsExpanded(!isExpanded);
  };

  const handleCancelClick = (fileId, fileName) => {
    setConfirmDialog({
      open: true,
      fileId,
      fileName,
    });
  };

  const handleConfirmCancel = async () => {
    const { fileId, fileName } = confirmDialog;

    // Add to cancelling set to show loading state
    setCancellingFiles(prev => new Set(prev).add(fileId));

    try {
      await cancelDownload(fileId, fileName);
    } catch (error) {
      console.error('Error during cancellation:', error);
    } finally {
      // Remove from cancelling set
      setCancellingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileId);
        return newSet;
      });
    }

    setConfirmDialog({ open: false, fileId: null, fileName: '' });
  };

  const handleCancelDialog = () => {
    setConfirmDialog({ open: false, fileId: null, fileName: '' });
  };

  const handleCancelAllClick = () => {
    const activeDownloads = statusModelFiles.filter(
      status =>
        (status.is_downloading && status.dl_requested_at && !status.storage_location) ||
        (status.dl_requested_at && !status.is_downloading && !status.storage_location)
    );

    if (activeDownloads.length === 0) {
      return; // No active downloads to cancel
    }

    setConfirmCancelAllDialog(true);
  };

  const handleConfirmCancelAll = async () => {
    setCancellingAll(true);

    try {
      await cancelAllDownloads();
    } catch (error) {
      console.error('Error during bulk cancellation:', error);
    } finally {
      setCancellingAll(false);
    }

    setConfirmCancelAllDialog(false);
  };

  const handleCancelAllDialog = () => {
    setConfirmCancelAllDialog(false);
  };

  const isDownloading = status => {
    return (
      status.is_downloading ||
      (status.dl_requested_at && status.download_percentage > 0 && status.download_percentage < 100)
    );
  };

  const isQueued = status => {
    return (
      status.dl_requested_at &&
      !status.is_downloading &&
      (!status.download_percentage || status.download_percentage === 0)
    );
  };

  const isCompleted = status => {
    // A file is completed if it has storage_location and dl_requested_at is cleared
    return status.storage_location && !status.dl_requested_at;
  };

  // Track files that disappear from statusModelFiles (they've completed)
  useEffect(() => {
    // First, update our record of all files we've ever seen
    const updatedSeenFiles = new Map(allSeenFiles);
    statusModelFiles.forEach(file => {
      // Always update with the latest data for each file
      updatedSeenFiles.set(file.id, file);
    });
    setAllSeenFiles(updatedSeenFiles);

    // Create a set of current file IDs for easy lookup
    const currentFileIds = new Set(statusModelFiles.map(f => f.id));

    // Find files that we've seen before but are no longer in the current response
    const disappearedFiles = [];
    const existingCompletedIds = new Set(completedFiles.map(f => f.id));

    updatedSeenFiles.forEach((file, fileId) => {
      // If we've seen this file before, it's not in current response, and it's not already in completed
      if (!currentFileIds.has(fileId) && !existingCompletedIds.has(fileId)) {
        // This file has disappeared - it must have completed
        disappearedFiles.push({
          ...file,
          storage_location: true,
          dl_requested_at: null,
          is_downloading: false,
          download_percentage: 100,
          completedAt: Date.now(),
        });
      }
    });

    // Add disappeared files to completed list
    if (disappearedFiles.length > 0) {
      setCompletedFiles(prev => [...disappearedFiles, ...prev]);

      // Auto-expand completed section when new files complete
      setIsCompletedExpanded(true);

      // Reset auto-collapse timer
      if (autoCollapseTimer) {
        clearTimeout(autoCollapseTimer);
      }
      const timer = setTimeout(() => {
        setIsCompletedExpanded(false);
      }, 10000); // Auto-collapse after 10 seconds
      setAutoCollapseTimer(timer);
    }

    // Also track any files that show as completed in the current status
    const currentlyCompleted = statusModelFiles.filter(isCompleted);
    if (currentlyCompleted.length > 0) {
      setCompletedFiles(prev => {
        const existingIds = new Set(prev.map(f => f.id));
        const newCompleted = currentlyCompleted
          .filter(f => !existingIds.has(f.id))
          .map(f => ({ ...f, completedAt: Date.now() }));
        return [...newCompleted, ...prev];
      });
    }
  }, [statusModelFiles, autoCollapseTimer, completedFiles.length]); // Don't include allSeenFiles to avoid loops

  // Clean up timer on unmount
  useEffect(() => {
    return () => {
      if (autoCollapseTimer) {
        clearTimeout(autoCollapseTimer);
      }
    };
  }, [autoCollapseTimer]);

  // Calculate combined progress statistics
  const stats = useMemo(() => {
    const activeFiles = statusModelFiles.filter(f => !isCompleted(f));
    if (!activeFiles.length && !completedFiles.length)
      return { totalProgress: 0, activeCount: 0, completedCount: 0, queuedCount: 0 };

    let totalProgress = 0;
    let queuedCount = 0;
    let activeCount = 0;

    activeFiles.forEach(status => {
      if (isQueued(status)) {
        queuedCount++;
      } else {
        activeCount++;
        totalProgress += status.download_percentage || 0;
      }
    });

    const totalFiles = activeFiles.length + completedFiles.length;
    const completedProgress = completedFiles.length * 100;

    return {
      totalProgress: totalFiles ? (totalProgress + completedProgress) / totalFiles : 0,
      activeCount,
      completedCount: completedFiles.length,
      queuedCount,
      totalCount: totalFiles,
    };
  }, [statusModelFiles, completedFiles]);

  // Active files are only those currently in statusModelFiles (not completed)
  const activeFiles = useMemo(() => {
    // Only show files that are actually in the current status response
    return statusModelFiles.filter(f => !isCompleted(f));
  }, [statusModelFiles]);

  const handleClearCompleted = () => {
    // Clear completed files from our tracking
    const completedIds = new Set(completedFiles.map(f => f.id));
    const updatedSeenFiles = new Map();
    allSeenFiles.forEach((file, fileId) => {
      if (!completedIds.has(fileId)) {
        updatedSeenFiles.set(fileId, file);
      }
    });
    setAllSeenFiles(updatedSeenFiles);
    setCompletedFiles([]);
    clearCompletedDownloads();
  };

  return (
    <>
      {(statusModelFiles.length > 0 || completedFiles.length > 0) && (
        <StyledFloatingWindow>
          <Box
            sx={{
              display: 'flex',
              justifyContent: 'space-between',
              alignItems: 'center',
              padding: '10px',
              borderBottom: isExpanded ? '1px solid rgba(255, 255, 255, 0.12)' : 'none',
              cursor: 'pointer',
            }}
            onClick={toggleExpand}
          >
            <Typography
              variant="h6"
              component="h3"
              sx={{ m: 0, fontSize: '1rem', fontWeight: 'bold' }}
            >
              Download Status
            </Typography>

            {!isExpanded && activeFiles.length > 0 && (
              <Box sx={{ display: 'flex', alignItems: 'center', mr: 2, flexGrow: 1, ml: 2 }}>
                <StyledDownloadProgressBar
                  variant="determinate"
                  value={stats.totalProgress}
                  sx={{ mx: 2, flexGrow: 1 }}
                />
                <Typography variant="body2" sx={{ ml: 1, whiteSpace: 'nowrap' }}>
                  {Math.round(stats.totalProgress)}%
                </Typography>
              </Box>
            )}

            <Box sx={{ display: 'flex', alignItems: 'center' }}>
              {!isExpanded && (
                <Box sx={{ display: 'flex', mr: 2 }}>
                  {stats.activeCount > 0 && (
                    <Chip
                      label={`${stats.activeCount} active`}
                      size="small"
                      color="primary"
                      sx={{ mr: 0.5 }}
                    />
                  )}
                  {stats.queuedCount > 0 && (
                    <Chip
                      label={`${stats.queuedCount} queued`}
                      size="small"
                      sx={{ mr: 0.5, bgcolor: 'warning.main' }}
                    />
                  )}
                </Box>
              )}
              {isExpanded ? <FaChevronUp /> : <FaChevronDown />}
            </Box>
          </Box>

          {isExpanded && (
            <StyledDownloadStatusModalContent>
              <Box sx={{ width: '100%' }}>
                {/* Active Downloads Section */}
                {activeFiles.length > 0 && (
                  <>
                    <Box
                      sx={{
                        display: 'flex',
                        justifyContent: 'space-between',
                        alignItems: 'center',
                        mb: 1,
                      }}
                    >
                      <Typography
                        variant="subtitle2"
                        sx={{
                          color: 'primary.main',
                          fontWeight: 600,
                          cursor: 'pointer',
                          '&:hover': { opacity: 0.8 },
                        }}
                        onClick={() => setIsActiveExpanded(!isActiveExpanded)}
                      >
                        Active Downloads ({activeFiles.length})
                      </Typography>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                        <StyledButton
                          onClick={handleCancelAllClick}
                          size="small"
                          color="error"
                          variant="text"
                          disabled={cancellingAll || stats.activeCount + stats.queuedCount === 0}
                          startIcon={
                            cancellingAll ? <CircularProgress size={16} /> : <CancelIcon />
                          }
                          sx={{ minWidth: 'auto', py: 0.5, px: 1 }}
                        >
                          {cancellingAll ? 'Cancelling...' : 'Cancel All'}
                        </StyledButton>
                        <IconButton
                          size="small"
                          onClick={() => setIsActiveExpanded(!isActiveExpanded)}
                          sx={{ p: 0.5 }}
                        >
                          {isActiveExpanded ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                        </IconButton>
                      </Box>
                    </Box>

                    <Collapse in={isActiveExpanded} timeout="auto" unmountOnExit>
                      {activeFiles.map((status, index) => (
                        <StyledDownloadStatusItem
                          key={index}
                          sx={{
                            mb: 1.5,
                            p: 1.5,
                            bgcolor: 'background.paper',
                            borderRadius: 1,
                            boxShadow: '0px 2px 4px rgba(0, 0, 0, 0.1)',
                          }}
                        >
                          <Box
                            sx={{
                              display: 'flex',
                              justifyContent: 'space-between',
                              alignItems: 'center',
                              width: '100%',
                            }}
                          >
                            <StyledDownloadStatusText variant="body1" sx={{ fontWeight: 'medium' }}>
                              {status.name || 'Unnamed file'}
                            </StyledDownloadStatusText>

                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              {isCompleted(status) ? (
                                <Chip
                                  label="Complete"
                                  size="small"
                                  color="success"
                                  sx={{ ml: 1 }}
                                />
                              ) : isQueued(status) ? (
                                <>
                                  <Chip
                                    label="Queued"
                                    size="small"
                                    sx={{ ml: 1, bgcolor: 'warning.main' }}
                                  />
                                  <IconButton
                                    size="small"
                                    onClick={e => {
                                      e.stopPropagation();
                                      handleCancelClick(status.id, status.name);
                                    }}
                                    disabled={cancellingFiles.has(status.id)}
                                    sx={{ ml: 1, color: 'error.main' }}
                                    title="Cancel download"
                                  >
                                    {cancellingFiles.has(status.id) ? (
                                      <CircularProgress size={16} />
                                    ) : (
                                      <CancelIcon fontSize="small" />
                                    )}
                                  </IconButton>
                                </>
                              ) : (
                                <>
                                  <Chip
                                    label="Downloading"
                                    size="small"
                                    color="primary"
                                    sx={{ ml: 1 }}
                                  />
                                  <IconButton
                                    size="small"
                                    onClick={e => {
                                      e.stopPropagation();
                                      handleCancelClick(status.id, status.name);
                                    }}
                                    disabled={cancellingFiles.has(status.id)}
                                    sx={{ ml: 1, color: 'error.main' }}
                                    title="Cancel download"
                                  >
                                    {cancellingFiles.has(status.id) ? (
                                      <CircularProgress size={16} />
                                    ) : (
                                      <CancelIcon fontSize="small" />
                                    )}
                                  </IconButton>
                                </>
                              )}
                            </Box>
                          </Box>

                          <StyledDownloadProgressBar
                            variant="determinate"
                            value={status.download_percentage || 0}
                            sx={{
                              height: 8,
                              borderRadius: 4,
                              my: 1,
                            }}
                          />

                          {isCompleted(status) ? (
                            <Typography variant="caption" sx={{ color: 'success.main' }}>
                              {status.download_elapsed
                                ? `Completed in ${status.download_elapsed}`
                                : 'Download complete'}
                            </Typography>
                          ) : isQueued(status) ? (
                            <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                              Waiting in queue...
                            </Typography>
                          ) : (
                            <Box
                              sx={{
                                display: 'flex',
                                justifyContent: 'space-between',
                                width: '100%',
                              }}
                            >
                              <Typography variant="caption" sx={{ color: 'text.secondary' }}>
                                {status.download_elapsed} / {status.download_remaining}
                              </Typography>
                              <Typography
                                variant="caption"
                                sx={{ color: 'text.secondary', fontWeight: 'bold' }}
                              >
                                {status.download_throughput}
                              </Typography>
                            </Box>
                          )}
                        </StyledDownloadStatusItem>
                      ))}
                    </Collapse>
                  </>
                )}

                {/* Completed Downloads Section */}
                {completedFiles.length > 0 && (
                  <>
                    <Box
                      sx={{
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'space-between',
                        mt: activeFiles.length > 0 ? 2 : 0,
                        mb: 1,
                      }}
                    >
                      <Typography
                        variant="subtitle2"
                        sx={{
                          color: 'success.main',
                          fontWeight: 600,
                          cursor: 'pointer',
                          '&:hover': { opacity: 0.8 },
                        }}
                        onClick={() => setIsCompletedExpanded(!isCompletedExpanded)}
                      >
                        Completed Downloads ({completedFiles.length})
                      </Typography>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                        <StyledButton
                          onClick={handleClearCompleted}
                          size="small"
                          variant="text"
                          sx={{ minWidth: 'auto', py: 0.5, px: 1 }}
                        >
                          Clear
                        </StyledButton>
                        <IconButton
                          size="small"
                          onClick={() => setIsCompletedExpanded(!isCompletedExpanded)}
                          sx={{ p: 0.5 }}
                        >
                          {isCompletedExpanded ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                        </IconButton>
                      </Box>
                    </Box>

                    <Collapse in={isCompletedExpanded} timeout="auto" unmountOnExit>
                      {completedFiles.map((file, index) => (
                        <StyledDownloadStatusItem
                          key={file.id || index}
                          sx={{
                            mb: 1,
                            p: 1.5,
                            bgcolor: 'background.paper',
                            borderRadius: 1,
                            boxShadow: '0px 2px 4px rgba(0, 0, 0, 0.1)',
                            opacity: 0.8,
                          }}
                        >
                          <Box
                            sx={{
                              display: 'flex',
                              justifyContent: 'space-between',
                              alignItems: 'center',
                              width: '100%',
                            }}
                          >
                            <StyledDownloadStatusText variant="body1" sx={{ fontWeight: 'medium' }}>
                              {file.name || 'Unnamed file'}
                            </StyledDownloadStatusText>
                          </Box>
                          <Typography variant="caption" sx={{ color: 'text.secondary', mt: 0.5 }}>
                            Completed{' '}
                            {file.completedAt
                              ? new Date(file.completedAt).toLocaleTimeString()
                              : ''}
                          </Typography>
                        </StyledDownloadStatusItem>
                      ))}
                    </Collapse>
                  </>
                )}
              </Box>
            </StyledDownloadStatusModalContent>
          )}
        </StyledFloatingWindow>
      )}

      {/* Confirmation Dialog */}
      <Dialog open={confirmDialog.open} onClose={handleCancelDialog}>
        <DialogTitle>Cancel Download</DialogTitle>
        <DialogContent>
          <Typography>
            Are you sure you want to cancel the download of "{confirmDialog.fileName}"?
          </Typography>
          <Typography variant="body2" sx={{ mt: 1, color: 'text.secondary' }}>
            This action cannot be undone. You'll need to restart the download if you want to
            download this file again.
          </Typography>
        </DialogContent>
        <DialogActions>
          <StyledButton onClick={handleCancelDialog} color="primary">
            Keep Downloading
          </StyledButton>
          <StyledButton onClick={handleConfirmCancel} color="error" variant="contained">
            Cancel Download
          </StyledButton>
        </DialogActions>
      </Dialog>

      {/* Cancel All Confirmation Dialog */}
      <Dialog open={confirmCancelAllDialog} onClose={handleCancelAllDialog}>
        <DialogTitle>Cancel All Downloads</DialogTitle>
        <DialogContent>
          <Typography>
            Are you sure you want to cancel all active downloads (
            {stats.activeCount + stats.queuedCount} files)?
          </Typography>
          <Typography variant="body2" sx={{ mt: 1, color: 'text.secondary' }}>
            This action cannot be undone. You'll need to restart the downloads if you want to
            download these files again.
          </Typography>
        </DialogContent>
        <DialogActions>
          <StyledButton onClick={handleCancelAllDialog} color="primary">
            Keep Downloading
          </StyledButton>
          <StyledButton onClick={handleConfirmCancelAll} color="error" variant="contained">
            Cancel All Downloads
          </StyledButton>
        </DialogActions>
      </Dialog>

      {/* Toast Notifications */}
      <Toast
        open={toast.open}
        message={toast.message}
        severity={toast.severity}
        onClose={closeToast}
      />
    </>
  );
};

export default DownloadStatusModal;
