// DownloadModal.js
//{"result":true,"message":"Download initiated","files":[933137470090870785]}
import React, { useContext, useState } from 'react';
import { styled } from '@mui/material/styles';
import axios from 'axios'; // Ensure axios is imported
import { BASE_URL } from '../../const';
import {
  Modal,
  Box,
  Typography,
  Button,
  Checkbox,
  FormControlLabel,
  IconButton,
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import ModelFilesContext from './ModelFilesContext'; // Import ModelFilesContext

// Styled components
const ModalContent = styled(Box)(({ theme }) => ({
  position: 'absolute',
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  backgroundColor: '#1a1a1a',
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: '8px',
  boxShadow: 24,
  padding: theme.spacing(3),
  width: '90vw',
  maxWidth: '600px',
  maxHeight: '90vh',
  display: 'flex',
  flexDirection: 'column',
}));

const FileListContainer = styled(Box)(({ theme }) => ({
  flex: 1,
  overflowY: 'auto',
  margin: theme.spacing(2, 0),
  padding: theme.spacing(1),
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: '4px',
  backgroundColor: 'rgba(255, 255, 255, 0.02)',
}));

const FileItem = styled(FormControlLabel)(({ theme }) => ({
  display: 'flex',
  marginBottom: theme.spacing(1),
  '& .MuiFormControlLabel-label': {
    flex: 1,
    display: 'flex',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
}));

const StyledButton = styled(Button)(({ theme }) => ({
  margin: theme.spacing(0, 1),
}));

const DownloadModal = ({ model, closeModal, showModal, setModelFiles }) => {
  const [selectedFiles, setSelectedFiles] = useState(model.files.map(file => file.name));
  const { updateStatusModelFiles, setShowDownloadStatus, startPolling, refreshModelList } =
    useContext(ModelFilesContext); // Get necessary functions from context

  const handleDownload = async () => {
    try {
      const response = await axios.post(`${BASE_URL}/models/download/`, {
        model: model.repo_modelId,
        version: model.version || '',
        files_to_download: selectedFiles,
        hub: model.hub,
      });
      // Use updateStatusModelFiles to immediately add the new download items to the global state
      if (response.data.result && Array.isArray(response.data.files)) {
        // Create a map of selected file names for quick lookup
        const selectedFileSet = new Set(selectedFiles);

        // Map the returned file IDs to complete file information
        // Assuming file IDs are returned in the same order as selectedFiles
        const filesForUpdate = response.data.files.map((fileId, index) => {
          // Get the file name from selectedFiles array at the same index
          const fileName = selectedFiles[index];
          // Find the corresponding file info from model.files
          const fileInfo = model.files.find(f => f.name === fileName);

          return {
            id: fileId,
            m_id: model.id,
            name: fileInfo ? fileInfo.name : fileName || 'Unknown file',
            size: fileInfo ? fileInfo.size : 0,
            dl_requested_at: new Date().toISOString(),
            is_downloading: false,
            download_percentage: 0,
          };
        });
        updateStatusModelFiles(filesForUpdate, true); // Pass true to indicate initial data with names
        // Trigger model list refresh to provide delay for backend processing
        refreshModelList();
        startPolling(); // Start polling for download status updates
      }
    } catch (err) {
      console.error('Download error:', err);
      // Handle download error
    }
    closeModal();
  };

  const toggleFileSelection = filename => {
    setSelectedFiles(
      selectedFiles.includes(filename)
        ? selectedFiles.filter(file => file !== filename)
        : [...selectedFiles, filename]
    );
  };

  const formatFileSize = size => {
    if (size > 1e9) return `${(size / 1e9).toFixed(2)} GB`;
    if (size > 1e6) return `${(size / 1e6).toFixed(2)} MB`;
    if (size > 1e3) return `${(size / 1e3).toFixed(2)} KB`;
    return `${size} B`;
  };

  return (
    <Modal open={showModal} onClose={closeModal}>
      <ModalContent>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
          <Typography variant="h5" component="h2" sx={{ color: 'primary.main', fontWeight: 600 }}>
            Select files to download for {model.name || model.repo_modelId}
          </Typography>
          <IconButton aria-label="close" onClick={closeModal} sx={{ color: 'text.secondary' }}>
            <CloseIcon />
          </IconButton>
        </Box>

        {/* Download button at top */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
          <FormControlLabel
            control={
              <Checkbox
                checked={selectedFiles.length === model.files.length}
                onChange={event => {
                  setSelectedFiles(event.target.checked ? model.files.map(file => file.name) : []);
                }}
                sx={{ color: 'primary.main' }}
              />
            }
            label="Select All"
          />
          <StyledButton
            variant="contained"
            onClick={handleDownload}
            disabled={selectedFiles.length === 0}
            sx={{
              backgroundColor: 'primary.main',
              '&:hover': { backgroundColor: 'primary.dark' },
            }}
          >
            Download Selected Files ({selectedFiles.length})
          </StyledButton>
        </Box>

        {/* Scrollable file list */}
        <FileListContainer>
          {model.files.map(file => (
            <FileItem
              key={file.name}
              control={
                <Checkbox
                  checked={selectedFiles.includes(file.name)}
                  onChange={() => toggleFileSelection(file.name)}
                  sx={{ color: 'primary.main' }}
                />
              }
              label={
                <Box sx={{ display: 'flex', justifyContent: 'space-between', width: '100%' }}>
                  <Typography sx={{ wordBreak: 'break-word', mr: 2 }}>{file.name}</Typography>
                  <Typography sx={{ color: 'text.secondary', flexShrink: 0 }}>
                    {formatFileSize(file.size)}
                  </Typography>
                </Box>
              }
            />
          ))}
        </FileListContainer>

        {/* Download button at bottom */}
        <Box sx={{ display: 'flex', justifyContent: 'flex-end', gap: 2, mt: 2 }}>
          <StyledButton
            variant="outlined"
            onClick={closeModal}
            sx={{
              borderColor: 'rgba(255, 255, 255, 0.23)',
              color: 'text.primary',
              '&:hover': {
                borderColor: 'rgba(255, 255, 255, 0.4)',
                backgroundColor: 'rgba(255, 255, 255, 0.05)',
              },
            }}
          >
            Cancel
          </StyledButton>
          <StyledButton
            variant="contained"
            onClick={handleDownload}
            disabled={selectedFiles.length === 0}
            sx={{
              backgroundColor: 'primary.main',
              '&:hover': { backgroundColor: 'primary.dark' },
            }}
          >
            Download Selected Files ({selectedFiles.length})
          </StyledButton>
        </Box>
      </ModalContent>
    </Modal>
  );
};

export default DownloadModal;
