import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import { useNavigate } from 'react-router-dom';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Typography,
  Box,
  Radio,
  RadioGroup,
  FormControlLabel,
  FormControl,
  Chip,
  Divider,
  Alert,
  CircularProgress,
  IconButton,
  Tooltip,
} from '@mui/material';
import AddIcon from '@mui/icons-material/Add';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import SettingsIcon from '@mui/icons-material/Settings';
import EditIcon from '@mui/icons-material/Edit';
import VisibilityIcon from '@mui/icons-material/Visibility';
import axios from 'axios';
import { BASE_URL } from '../../const';
import AddModelConfigForm from './AddModelConfigForm';

const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 12,
    maxWidth: '600px',
    width: '100%',
  },
}));

const ConfigOption = styled(Box)(({ theme }) => ({
  padding: theme.spacing(2),
  border: '1px solid rgba(255, 255, 255, 0.12)',
  borderRadius: 8,
  marginBottom: theme.spacing(1),
  transition: 'all 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
    borderColor: theme.palette.primary.main,
  },
  '&.selected': {
    backgroundColor: 'rgba(0, 192, 127, 0.05)',
    borderColor: theme.palette.primary.main,
  },
}));

const DefaultChip = styled(Chip)(({ theme }) => ({
  backgroundColor: theme.palette.success.main,
  color: 'white',
  fontSize: '0.75rem',
  height: '20px',
  fontWeight: 500,
}));

const EngineChip = styled(Chip)(({ theme }) => ({
  backgroundColor: theme.palette.info.main,
  color: 'white',
  fontSize: '0.75rem',
  height: '20px',
  fontWeight: 500,
}));

const DeployModal = ({ open, onClose, model, onDeploy, onAddConfig }) => {
  const [configs, setConfigs] = useState([]);
  const [selectedConfig, setSelectedConfig] = useState('');
  const [loading, setLoading] = useState(false);
  const [deploying, setDeploying] = useState(false);
  const [error, setError] = useState(null);
  const [showAddConfig, setShowAddConfig] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    if (open && model?.id) {
      fetchConfigs();
    }
  }, [open, model]);

  const fetchConfigs = async () => {
    setLoading(true);
    setError(null);

    try {
      const response = await axios.get(`${BASE_URL}/models/${model.id}/configs`);
      const configData = response.data || [];

      // Debug log to check the structure
      console.log('Fetched configs for model:', model.name);
      configData.forEach(config => {
        console.log(`Config "${config.name}":`, {
          id: config.id,
          default: config.default,
          engine_name: config.engine_name,
          config: config.config,
          system_config: config.system_config,
        });
      });

      // Sort configs: default first, then alphabetically by name
      const sortedConfigs = [...configData].sort((a, b) => {
        // Default configs come first
        if (a.default && !b.default) return -1;
        if (!a.default && b.default) return 1;

        // Then sort alphabetically by name
        const nameA = a.name || 'Unnamed';
        const nameB = b.name || 'Unnamed';
        return nameA.localeCompare(nameB);
      });

      setConfigs(sortedConfigs);

      // Auto-select default config or first config
      const defaultConfig = sortedConfigs.find(config => config.default);
      if (defaultConfig) {
        setSelectedConfig(defaultConfig.id);
      } else if (sortedConfigs.length > 0) {
        setSelectedConfig(sortedConfigs[0].id);
      }
    } catch (err) {
      console.error('Error fetching configs:', err);
      setError('Failed to load deployment configurations');
      setConfigs([]);
    } finally {
      setLoading(false);
    }
  };

  const handleDeploy = async () => {
    if (!selectedConfig) return;

    setDeploying(true);

    try {
      const selectedConfigObj = configs.find(config => config.id === selectedConfig);

      if (onDeploy) {
        await onDeploy(model, selectedConfigObj);
      }

      onClose();
    } catch (err) {
      console.error('Error deploying model:', err);
      setError('Failed to deploy model');
    } finally {
      setDeploying(false);
    }
  };

  const handleAddConfig = () => {
    setShowAddConfig(true);
  };

  const handleConfigAdded = newConfig => {
    // Add the new config to our list
    setConfigs(prevConfigs => {
      const updatedConfigs = [...prevConfigs, newConfig];
      // Sort again to maintain order
      return updatedConfigs.sort((a, b) => {
        if (a.default && !b.default) return -1;
        if (!a.default && b.default) return 1;
        const nameA = a.name || 'Unnamed';
        const nameB = b.name || 'Unnamed';
        return nameA.localeCompare(nameB);
      });
    });

    // Auto-select the new config
    setSelectedConfig(newConfig.id);

    // Close the add config form
    setShowAddConfig(false);
  };

  const handleViewConfig = (configId, event) => {
    event.stopPropagation();
    navigate(`/model_configs/${configId}`);
    onClose();
  };

  const handleEditConfig = (configId, event) => {
    event.stopPropagation();
    navigate(`/model_configs/${configId}`);
    onClose();
  };

  const handleClose = () => {
    if (!deploying) {
      onClose();
    }
  };

  return (
    <>
      <StyledDialog open={open && !showAddConfig} onClose={handleClose} maxWidth="md">
        <DialogTitle>
          <Typography variant="h6" component="div" sx={{ fontWeight: 600 }}>
            Deploy Model: {model?.name}
          </Typography>
          <Typography variant="body2" color="text.secondary">
            Select a deployment configuration
          </Typography>
        </DialogTitle>

        <DialogContent>
          {loading ? (
            <Box sx={{ display: 'flex', justifyContent: 'center', py: 4 }}>
              <CircularProgress />
            </Box>
          ) : error ? (
            <Alert severity="error" sx={{ mb: 2 }}>
              {error}
            </Alert>
          ) : configs.length === 0 ? (
            <Box sx={{ textAlign: 'center', py: 4 }}>
              <Typography variant="body1" color="text.secondary" sx={{ mb: 2 }}>
                No deployment configurations found for this model.
              </Typography>
              <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                Create a configuration to enable deployment.
              </Typography>
              <Button
                variant="contained"
                startIcon={<AddIcon />}
                onClick={handleAddConfig}
                sx={{ mt: 1 }}
              >
                Create First Configuration
              </Button>
            </Box>
          ) : (
            <Box>
              <FormControl component="fieldset" sx={{ width: '100%' }}>
                <RadioGroup
                  value={selectedConfig}
                  onChange={e => setSelectedConfig(e.target.value)}
                >
                  {configs.map(config => (
                    <FormControlLabel
                      key={config.id}
                      value={config.id}
                      control={<Radio sx={{ alignSelf: 'flex-start', mt: 1 }} />}
                      label={
                        <ConfigOption className={selectedConfig === config.id ? 'selected' : ''}>
                          <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                            <Typography
                              variant="subtitle1"
                              sx={{ fontWeight: 600, mr: 1, flex: 1 }}
                            >
                              {config.name || 'Unnamed Configuration'}
                            </Typography>
                            {config.default && <DefaultChip label="Default" size="small" />}
                          </Box>

                          {config.description && (
                            <Typography variant="body2" color="text.secondary" sx={{ mb: 1 }}>
                              {config.description}
                            </Typography>
                          )}

                          <Box
                            sx={{
                              display: 'flex',
                              justifyContent: 'flex-end',
                              alignItems: 'center',
                              mt: 1,
                            }}
                          >
                            <Box sx={{ display: 'flex', gap: 0.5 }}>
                              <Tooltip title="View Config Details">
                                <IconButton
                                  size="small"
                                  onClick={e => handleViewConfig(config.id, e)}
                                  sx={{
                                    color: 'primary.main',
                                    '&:hover': { backgroundColor: 'rgba(0, 192, 127, 0.1)' },
                                  }}
                                >
                                  <VisibilityIcon fontSize="small" />
                                </IconButton>
                              </Tooltip>
                              <Tooltip title="Edit Config">
                                <IconButton
                                  size="small"
                                  onClick={e => handleEditConfig(config.id, e)}
                                  sx={{
                                    color: 'primary.main',
                                    '&:hover': { backgroundColor: 'rgba(0, 192, 127, 0.1)' },
                                  }}
                                >
                                  <EditIcon fontSize="small" />
                                </IconButton>
                              </Tooltip>
                            </Box>
                          </Box>
                        </ConfigOption>
                      }
                      sx={{
                        alignItems: 'flex-start',
                        m: 0,
                        mb: 2,
                        '& .MuiFormControlLabel-label': {
                          width: '100%',
                          ml: 1,
                        },
                      }}
                    />
                  ))}
                </RadioGroup>
              </FormControl>

              <Divider sx={{ my: 3 }} />

              <Box sx={{ display: 'flex', alignItems: 'center' }}>
                <SettingsIcon sx={{ mr: 1, color: 'text.secondary' }} />
                <Typography variant="body2" color="text.secondary" sx={{ mr: 2 }}>
                  Need a different configuration?
                </Typography>
                <Button size="small" startIcon={<AddIcon />} onClick={handleAddConfig}>
                  Add New Config
                </Button>
              </Box>
            </Box>
          )}
        </DialogContent>

        <DialogActions sx={{ px: 3, pb: 3 }}>
          <Button onClick={handleClose} disabled={deploying}>
            Cancel
          </Button>
          <Button
            variant="contained"
            startIcon={deploying ? <CircularProgress size={16} /> : <RocketLaunchIcon />}
            onClick={handleDeploy}
            disabled={!selectedConfig || deploying || configs.length === 0}
          >
            {deploying ? 'Deploying...' : 'Deploy'}
          </Button>
        </DialogActions>
      </StyledDialog>

      {/* Add Model Config Form */}
      {showAddConfig && model && (
        <AddModelConfigForm
          showModal={showAddConfig}
          closeModal={() => setShowAddConfig(false)}
          m_id={model.id}
          setParentModelConfigs={updateFn => {
            if (typeof updateFn === 'function') {
              // Handle the case where it's called with a function (standard setState pattern)
              const currentConfigs = configs;
              const newConfigs = updateFn(currentConfigs);
              if (newConfigs.length > currentConfigs.length) {
                // A new config was added, get the last one
                const newConfig = newConfigs[newConfigs.length - 1];
                handleConfigAdded(newConfig);
              }
            }
          }}
        />
      )}
    </>
  );
};

export default DeployModal;
