import React, { useState, useEffect, useRef } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Box,
  Paper,
  Typography,
  IconButton,
  TextField,
  Switch,
  FormControlLabel,
  Chip,
  CircularProgress,
  Alert,
  Toolbar,
  Tooltip,
} from '@mui/material';
import {
  Refresh,
  Download,
  Search,
  FirstPage,
  LastPage,
  ContentCopy,
} from '@mui/icons-material';
import LogPatternAlert from './LogPatternAlert';
import { getDeploymentLogs, getDeploymentLogPatterns } from '../../services/containerLogsService';

const LogContainer = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(13, 13, 13, 0.95)',
  border: '1px solid rgba(255, 255, 255, 0.12)',
  borderRadius: 12,
  overflow: 'hidden',
  display: 'flex',
  flexDirection: 'column',
  height: '70vh',
  minHeight: '400px',
}));

const LogHeader = styled(Toolbar)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.3)',
  borderBottom: '1px solid rgba(255, 255, 255, 0.08)',
  minHeight: '48px !important',
  padding: theme.spacing(0, 2),
}));

const LogContent = styled(Box)(({ theme }) => ({
  flex: 1,
  overflow: 'auto',
  padding: theme.spacing(1, 2),
  fontFamily: 'Menlo, Monaco, "Courier New", monospace',
  fontSize: '0.8rem',
  lineHeight: 1.4,
  backgroundColor: '#0a0a0a',
  '&::-webkit-scrollbar': {
    width: '8px',
  },
  '&::-webkit-scrollbar-track': {
    background: 'rgba(255, 255, 255, 0.05)',
  },
  '&::-webkit-scrollbar-thumb': {
    background: 'rgba(255, 255, 255, 0.2)',
    borderRadius: '4px',
  },
  '&::-webkit-scrollbar-thumb:hover': {
    background: 'rgba(255, 255, 255, 0.3)',
  },
}));

const LogLine = styled(Box, {
  shouldForwardProp: (prop) => prop !== 'highlighted' && prop !== 'isVisible'
})(({ theme, highlighted, isVisible }) => ({
  display: isVisible ? 'block' : 'none',
  padding: theme.spacing(0.25, 0),
  whiteSpace: 'pre-wrap',
  wordBreak: 'break-word',
  borderRadius: theme.spacing(0.5),
  ...(highlighted && {
    backgroundColor: 'rgba(255, 235, 59, 0.08)',
    padding: theme.spacing(0.5),
    margin: theme.spacing(0.25, 0),
  }),
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
}));

const MetadataChip = styled(Chip)(({ theme }) => ({
  margin: theme.spacing(0, 0.5),
  height: '24px',
  fontSize: '0.75rem',
  '& .MuiChip-label': {
    padding: theme.spacing(0, 1),
  },
}));

const SearchContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
  padding: theme.spacing(1, 2),
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  borderBottom: '1px solid rgba(255, 255, 255, 0.08)',
}));

/**
 * Component to display and interact with container logs
 * @param {Object} props - Component props
 * @param {string} props.deploymentId - The deployment ID
 */
const ContainerLogViewer = ({ deploymentId }) => {
  const [logs, setLogs] = useState(null);
  const [patterns, setPatterns] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [showSearch, setShowSearch] = useState(false);
  const [autoRefresh, setAutoRefresh] = useState(false);
  const [lastRefresh, setLastRefresh] = useState(new Date());
  const [noLogsMessage, setNoLogsMessage] = useState(null);
  
  const logContentRef = useRef(null);
  const refreshIntervalRef = useRef(null);

  // Filter logs based on search term
  const filteredLogs = logs?.logs?.map(line => ({
    content: line,
    filtered: !searchTerm || line.toLowerCase().includes(searchTerm.toLowerCase()),
    highlighted: searchTerm && line.toLowerCase().includes(searchTerm.toLowerCase())
  })) || [];

  const visibleLogCount = filteredLogs.filter(log => log.filtered !== false).length;

  const fetchLogs = async (silent = false) => {
    if (!silent) setLoading(true);
    setError(null);
    setNoLogsMessage(null);

    try {
      const logsResponse = await getDeploymentLogs(deploymentId);
      setLogs(logsResponse);
      if (!logsResponse?.logs || logsResponse.logs.length === 0) {
        setNoLogsMessage('No logs have been captured yet for this deployment. Logs will appear once the service emits output.');
      }

      try {
        const patternsResponse = await getDeploymentLogPatterns(deploymentId);
        setPatterns(patternsResponse);
      } catch (patternErr) {
        if (patternErr.response && (patternErr.response.status === 404 || patternErr.response.status === 204)) {
          setPatterns(null);
        } else {
          console.error('Error analyzing log patterns:', patternErr);
          setError(patternErr.response?.data?.detail || patternErr.message || 'Failed to analyze log patterns');
        }
      }

      setLastRefresh(new Date());
    } catch (err) {
      if (err.response && (err.response.status === 404 || err.response.status === 204)) {
        setLogs({ logs: [], total_lines_seen: 0 });
        setPatterns(null);
        setNoLogsMessage('No logs have been captured yet for this deployment. Logs will appear once the service emits output.');
        setError(null);
        setLastRefresh(new Date());
      } else {
        setError(err.response?.data?.detail || err.message || 'Failed to load logs');
      }
    } finally {
      if (!silent) setLoading(false);
    }
  };

  const handleRefresh = () => {
    fetchLogs();
  };

  const handleDownload = () => {
    if (!logs?.logs) return;
    
    const logText = logs.logs.join('\n');
    const blob = new Blob([logText], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `container-logs-${deploymentId}-${new Date().toISOString().split('T')[0]}.log`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const handleCopyLogs = () => {
    if (!logs?.logs) return;
    
    const logText = logs.logs.join('\n');
    navigator.clipboard.writeText(logText).then(() => {
      // Could add a toast notification here
    });
  };

  const handleToggleSearch = () => {
    const newShowSearch = !showSearch;
    setShowSearch(newShowSearch);
    // Clear search term when hiding search
    if (!newShowSearch) {
      setSearchTerm('');
    }
  };

  const scrollToTop = () => {
    if (logContentRef.current) {
      logContentRef.current.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  const scrollToBottom = () => {
    if (logContentRef.current) {
      logContentRef.current.scrollTo({ 
        top: logContentRef.current.scrollHeight, 
        behavior: 'smooth' 
      });
    }
  };

  // Auto-refresh logic
  useEffect(() => {
    if (autoRefresh) {
      refreshIntervalRef.current = setInterval(() => {
        fetchLogs(true); // Silent refresh
      }, 5000);
    } else {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
        refreshIntervalRef.current = null;
      }
    }

    return () => {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
      }
    };
  }, [autoRefresh, deploymentId]);

  // Initial load
  useEffect(() => {
    fetchLogs();
  }, [deploymentId]);

  // Auto-scroll to bottom on new logs when auto-refresh is on
  useEffect(() => {
    if (autoRefresh && logContentRef.current) {
      scrollToBottom();
    }
  }, [logs, autoRefresh]);

  if (loading && !logs) {
    return (
      <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
        <CircularProgress />
      </Box>
    );
  }

  if (error && !logs) {
    return (
      <Alert severity="error" sx={{ m: 2 }}>
        {error}
      </Alert>
    );
  }

  return (
    <Box sx={{ width: '100%' }}>
      {/* Pattern Analysis */}
      <LogPatternAlert 
        patterns={patterns} 
        loading={loading} 
        error={error} 
      />
      
      <LogContainer>
        {/* Header with controls */}
        <LogHeader>
          <Box sx={{ display: 'flex', alignItems: 'center', flex: 1, gap: 1 }}>
            <Typography variant="h6" sx={{ fontSize: '1rem', fontWeight: 600 }}>
              Logs
            </Typography>
            
            {logs && (
              <>
                <MetadataChip 
                  label={`${logs.engine_type?.toUpperCase() || 'Unknown'}`} 
                  color="primary" 
                  size="small" 
                />
                <MetadataChip 
                  label={`${visibleLogCount}/${logs.total_lines_seen} lines`} 
                  size="small" 
                />
                {logs.compressed && (
                  <MetadataChip 
                    label="Compressed" 
                    color="warning" 
                    size="small" 
                  />
                )}
              </>
            )}
          </Box>
          
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
            <FormControlLabel
              control={
                <Switch
                  checked={autoRefresh}
                  onChange={(e) => setAutoRefresh(e.target.checked)}
                  size="small"
                />
              }
              label="Auto-refresh"
              sx={{ 
                mr: 1,
                '& .MuiFormControlLabel-label': { fontSize: '0.8rem' }
              }}
            />
            
            <Tooltip title="Search logs">
              <IconButton 
                size="small" 
                onClick={handleToggleSearch}
                color={showSearch ? 'primary' : 'default'}
              >
                <Search fontSize="small" />
              </IconButton>
            </Tooltip>
            
            <Tooltip title="Refresh">
              <IconButton size="small" onClick={handleRefresh} disabled={loading}>
                <Refresh fontSize="small" />
              </IconButton>
            </Tooltip>
            
            <Tooltip title="Download logs">
              <IconButton size="small" onClick={handleDownload} disabled={!logs?.logs}>
                <Download fontSize="small" />
              </IconButton>
            </Tooltip>
            
            <Tooltip title="Copy to clipboard">
              <IconButton size="small" onClick={handleCopyLogs} disabled={!logs?.logs}>
                <ContentCopy fontSize="small" />
              </IconButton>
            </Tooltip>
            
            <Tooltip title="Scroll to top">
              <IconButton size="small" onClick={scrollToTop}>
                <FirstPage fontSize="small" />
              </IconButton>
            </Tooltip>
            
            <Tooltip title="Scroll to bottom">
              <IconButton size="small" onClick={scrollToBottom}>
                <LastPage fontSize="small" />
              </IconButton>
            </Tooltip>
          </Box>
        </LogHeader>

        {/* Search bar */}
        {showSearch && (
          <SearchContainer>
            <Search color="action" />
            <TextField
              fullWidth
              size="small"
              placeholder="Search logs..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              sx={{
                '& .MuiOutlinedInput-root': {
                  fontSize: '0.85rem',
                  backgroundColor: 'rgba(255, 255, 255, 0.05)',
                },
              }}
            />
            {searchTerm && (
              <Typography variant="caption" sx={{ minWidth: 'max-content', ml: 1 }}>
                {visibleLogCount} matches
              </Typography>
            )}
          </SearchContainer>
        )}

        {/* Log content */}
        <LogContent ref={logContentRef}>
          {logs?.logs && logs.logs.length > 0 ? (
            filteredLogs.map((logLine, index) => (
              <LogLine
                key={index}
                highlighted={logLine.highlighted}
                isVisible={logLine.filtered}
              >
                <Typography 
                  component="span" 
                  sx={{ 
                    color: logLine.highlighted ? '#fff' : 'text.primary',
                    fontSize: 'inherit',
                    fontFamily: 'inherit',
                    fontWeight: logLine.highlighted ? 500 : 400
                  }}
                >
                  {logLine.content}
                </Typography>
              </LogLine>
            ))
          ) : (
            <Typography color="text.secondary" sx={{ textAlign: 'center', mt: 4 }}>
              {noLogsMessage || 'No logs available'}
            </Typography>
          )}
        </LogContent>

        {/* Footer with metadata */}
        <Box sx={{ 
          p: 1, 
          backgroundColor: 'rgba(0, 0, 0, 0.3)', 
          borderTop: '1px solid rgba(255, 255, 255, 0.08)',
          display: 'flex',
          justifyContent: 'space-between',
          alignItems: 'center'
        }}>
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
            {logs && (
              <>
                <Typography variant="caption" color="text.secondary">
                  Last updated: {lastRefresh.toLocaleTimeString()}
                </Typography>
                {autoRefresh && (
                  <Chip 
                    label="Live" 
                    color="success" 
                    size="small" 
                    sx={{ height: '20px', fontSize: '0.7rem' }}
                  />
                )}
              </>
            )}
          </Box>
          
          <Typography variant="caption" color="text.secondary">
            {logs?.container_id || 'Container ID unavailable'}
          </Typography>
        </Box>
      </LogContainer>
    </Box>
  );
};

ContainerLogViewer.propTypes = {
  deploymentId: PropTypes.string.isRequired,
};

export default ContainerLogViewer;