import CloseIcon from "@mui/icons-material/Close";
import DownloadIcon from "@mui/icons-material/Download";
import ExpandLessIcon from "@mui/icons-material/ExpandLess";
import SearchIcon from "@mui/icons-material/Search";
import WarningAmberIcon from "@mui/icons-material/WarningAmber";
import {
	Alert,
	Box,
	Button,
	Checkbox,
	CircularProgress,
	Collapse,
	Dialog,
	DialogActions,
	DialogContent,
	DialogTitle,
	FormControlLabel,
	IconButton,
	List,
	ListItem,
	ListItemIcon,
	ListItemText,
	Paper,
	Table,
	TableBody,
	TableCell,
	TableContainer,
	TableHead,
	TableRow,
	TextField,
	Typography,
} from "@mui/material";
import { styled } from "@mui/material/styles";
import axios from "axios";
import PropTypes from "prop-types";
import React, {
	useCallback,
	useContext,
	useEffect,
	useRef,
	useState,
} from "react";
import { BASE_URL } from "../../const";
import AddExternalEndpointModal from "./AddExternalEndpointModal";
import ModelFilesContext from "./ModelFilesContext";

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
	"& .MuiDialog-paper": {
		backgroundColor: theme.palette.background.paper,
		borderRadius: 12,
		width: "90vw",
		maxWidth: "1200px",
		height: "90vh",
		maxHeight: "90vh",
		display: "flex",
		flexDirection: "column",
	},
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
	display: "flex",
	justifyContent: "space-between",
	alignItems: "center",
	paddingBottom: theme.spacing(2),
	borderBottom: `1px solid ${theme.palette.divider}`,
}));

const SearchContainer = styled(Box)(({ theme }) => ({
	display: "flex",
	alignItems: "center",
	gap: theme.spacing(2),
	marginBottom: theme.spacing(3),
}));

const SearchInput = styled(TextField)(({ theme }) => ({
	backgroundColor: "rgba(255, 255, 255, 0.03)",
	borderRadius: theme.shape.borderRadius,
	"& .MuiOutlinedInput-root": {
		"& fieldset": {
			borderColor: "rgba(255, 255, 255, 0.1)",
		},
		"&:hover fieldset": {
			borderColor: theme.palette.primary.main,
		},
		"&.Mui-focused fieldset": {
			borderColor: theme.palette.primary.main,
		},
	},
}));

const StyledTableContainer = styled(TableContainer)(() => ({
	backgroundColor: "rgba(255, 255, 255, 0.03)",
	borderRadius: 8,
	flex: 1,
	overflowY: "auto",
	"& .MuiTableCell-root": {
		borderColor: "rgba(255, 255, 255, 0.08)",
	},
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
	backgroundColor: "rgba(0, 0, 0, 0.2)",
	color: theme.palette.primary.main,
	fontWeight: 600,
	fontSize: "0.95rem",
	padding: theme.spacing(1.5, 2),
	position: "sticky",
	top: 0,
	zIndex: 1,
}));

const StyledTableRow = styled(TableRow)(() => ({
	"&:nth-of-type(odd)": {
		backgroundColor: "rgba(255, 255, 255, 0.02)",
	},
	"&:hover": {
		backgroundColor: "rgba(255, 255, 255, 0.05)",
	},
}));

const ActionButton = styled(Button)(({ theme }) => ({
	textTransform: "none",
	fontWeight: 500,
	padding: theme.spacing(0.8, 2),
	borderRadius: theme.shape.borderRadius,
	transition: "all 0.2s",
	"&:hover": {
		transform: "translateY(-2px)",
		boxShadow: "0 4px 8px rgba(0, 0, 0, 0.2)",
	},
}));

const ModelLink = styled("a")(({ theme }) => ({
	color: theme.palette.primary.main,
	textDecoration: "none",
	fontWeight: 500,
	"&:hover": {
		textDecoration: "underline",
	},
}));

const AddModelModal = ({ open, onClose }) => {
	const { updateStatusModelFiles, refreshModelList, startPolling } =
		useContext(ModelFilesContext);
	const [query, setQuery] = useState("");
	const [isExact, setIsExact] = useState(false);
	const [results, setResults] = useState([]);
	const [loading, setLoading] = useState(false);
	const [error, setError] = useState(null);
	const [expandedModel, setExpandedModel] = useState(null);
	const [modelFiles, setCurrentModelFiles] = useState([]);
	const [selectedFiles, setSelectedFiles] = useState([]);
	const [loadingFiles, setLoadingFiles] = useState(false);
	const [hasSearched, setHasSearched] = useState(false);
	const [hubError, setHubError] = useState(null);
	const [showExternalEndpointModal, setShowExternalEndpointModal] =
		useState(false);
	const searchTimerRef = useRef(null);
	const prevIsExactRef = useRef(isExact);

	// Perform the actual search
	const performSearch = useCallback(async (searchQuery, exact) => {
		if (!searchQuery.trim()) {
			setResults([]);
			setHasSearched(false);
			setHubError(null);
			setLoading(false);
			return;
		}

		setLoading(true);
		setError(null);
		setHubError(null);

		try {
			const response = await axios.post(`${BASE_URL}/models/search/`, {
				query: searchQuery,
				exact: exact,
				hubs_to_search: ["*"],
			});
			// Assuming the API returns an object with a 'results' array
			setResults(response.data.results.map((item) => item.model));
			setHasSearched(true);

			// Track hub error for display
			if (response.data.hub_error) {
				setHubError(response.data.hub_error);
			}
		} catch (err) {
			console.error("Search error:", err);
			setError(
				err.response?.data?.message || "An error occurred while searching",
			);
			setHasSearched(true);
		} finally {
			setLoading(false);
		}
	}, []);

	// Handle form submission (immediate search)
	const handleSearch = async (event) => {
		event.preventDefault();
		if (searchTimerRef.current) {
			clearTimeout(searchTimerRef.current);
			searchTimerRef.current = null;
		}
		await performSearch(query, isExact);
	};

	// Handle input changes with debouncing
	const handleQueryChange = useCallback(
		(newQuery) => {
			setQuery(newQuery);

			// Clear previous timer
			if (searchTimerRef.current) {
				clearTimeout(searchTimerRef.current);
				searchTimerRef.current = null;
			}

			// If query is empty, clear results immediately
			if (!newQuery.trim()) {
				setResults([]);
				setHasSearched(false);
				setError(null);
				return;
			}

			// Set new timer for debounced search
			searchTimerRef.current = setTimeout(() => {
				performSearch(newQuery, isExact);
			}, 800); // 800ms debounce for better UX
		},
		[isExact, performSearch],
	);

	// Clean up timer on unmount
	useEffect(() => {
		return () => {
			if (searchTimerRef.current) {
				clearTimeout(searchTimerRef.current);
			}
		};
	}, []);

	// Re-search when isExact changes (if there's a query)
	useEffect(() => {
		if (prevIsExactRef.current !== isExact && query.trim() && hasSearched) {
			// Clear any pending timer first
			if (searchTimerRef.current) {
				clearTimeout(searchTimerRef.current);
				searchTimerRef.current = null;
			}
			performSearch(query, isExact);
		}
		prevIsExactRef.current = isExact;
	}, [isExact, query, hasSearched, performSearch]);

	const handleExpandModel = async (model) => {
		const modelKey = `${model.hub}-${model.repo_modelId}`;

		if (expandedModel === modelKey) {
			// Collapse if already expanded
			setExpandedModel(null);
			setCurrentModelFiles([]);
			setSelectedFiles([]);
			return;
		}

		// Expand and fetch files
		setExpandedModel(modelKey);
		setLoadingFiles(true);
		setError(null);

		try {
			const response = await axios.post(`${BASE_URL}/model_files/search/`, {
				hub: model.hub,
				model: model.repo_modelId,
				version: model.version || "",
			});
			setCurrentModelFiles(response.data);
			// Select all files by default
			setSelectedFiles(response.data.map((file) => file.name));
		} catch (err) {
			console.error("Error fetching files:", err);
			setError("Failed to fetch model files");
			setExpandedModel(null);
		} finally {
			setLoadingFiles(false);
		}
	};

	const handleFileSelection = (fileName) => {
		setSelectedFiles((prev) =>
			prev.includes(fileName)
				? prev.filter((f) => f !== fileName)
				: [...prev, fileName],
		);
	};

	const handleSelectAllFiles = (checked) => {
		setSelectedFiles(checked ? modelFiles.map((file) => file.name) : []);
	};

	const handleDownloadSelected = async () => {
		if (selectedFiles.length === 0) {
			setError("Please select at least one file to download");
			return;
		}

		const model = results.find(
			(m) => `${m.hub}-${m.repo_modelId}` === expandedModel,
		);

		try {
			setLoadingFiles(true);
			const payload = {
				model: model.repo_modelId,
				version: model.version || "",
				files_to_download: selectedFiles,
				hub: model.hub,
			};

			console.log("Download payload:", payload);
			console.log("Model object:", model);

			const response = await axios.post(
				`${BASE_URL}/models/download/`,
				payload,
			);

			if (response.data.result && Array.isArray(response.data.files)) {
				// Map the returned file IDs to complete file information
				const filesForUpdate = response.data.files.map((fileId, index) => {
					const fileName = selectedFiles[index];
					const fileInfo = modelFiles.find((f) => f.name === fileName);

					return {
						id: fileId,
						m_id: response.data.model_id || null, // Model ID might be returned in response
						name: fileInfo ? fileInfo.name : fileName || "Unknown file",
						size: fileInfo ? fileInfo.size : 0,
						dl_requested_at: new Date().toISOString(),
						is_downloading: false,
						download_percentage: 0,
					};
				});
				updateStatusModelFiles(filesForUpdate, true); // Pass true to indicate initial data with names
				// Trigger model list refresh after successful download
				refreshModelList();
				// Start polling for download status
				startPolling();
			}

			// Close modal on successful download
			onClose();
		} catch (err) {
			console.error("Download error:", err);
			console.error("Error response:", err.response?.data);
			console.error("Error status:", err.response?.status);
			const errorMessage =
				err.response?.data?.detail ||
				err.response?.data?.message ||
				err.message ||
				"Failed to start download";
			setError(`Download failed: ${errorMessage}`);
		} finally {
			setLoadingFiles(false);
		}
	};

	const getHubUrl = (model) => {
		switch (model.hub?.toLowerCase()) {
			case "huggingface":
			case "hf":
				return `https://huggingface.co/${model.repo_modelId}`;
			case "github":
				return `https://github.com/${model.repo_modelId}`;
			case "ollama":
				return `https://ollama.com/library/${model.repo_modelId}`;
			default:
				// For unknown hubs, return a generic search URL or the model's source if available
				return (
					model.source_repository ||
					`https://www.google.com/search?q=${encodeURIComponent(model.repo_modelId + " AI model")}`
				);
		}
	};

	const handleClose = () => {
		// Clear any pending search timer
		if (searchTimerRef.current) {
			clearTimeout(searchTimerRef.current);
			searchTimerRef.current = null;
		}

		setQuery("");
		setResults([]);
		setError(null);
		setExpandedModel(null);
		setCurrentModelFiles([]);
		setSelectedFiles([]);
		setLoadingFiles(false);
		setHasSearched(false);
		setHubError(null);
		setShowExternalEndpointModal(false);
		onClose();
	};

	return (
		<>
			<StyledDialog open={open} onClose={handleClose} maxWidth="lg">
				<StyledDialogTitle>
					<Typography variant="h6" sx={{ fontWeight: 600 }}>
						Add Model
					</Typography>
					<Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
						<Button
							variant="outlined"
							size="small"
							onClick={() => setShowExternalEndpointModal(true)}
						>
							Add external inference endpoint
						</Button>
						<IconButton onClick={handleClose} size="small">
							<CloseIcon />
						</IconButton>
					</Box>
				</StyledDialogTitle>

				<DialogContent
					sx={{
						pt: 3,
						flex: 1,
						display: "flex",
						flexDirection: "column",
						overflow: "hidden",
					}}
				>
					<form onSubmit={handleSearch}>
						<SearchContainer>
							<SearchInput
								placeholder="Search for models..."
								value={query}
								onChange={(e) => handleQueryChange(e.target.value)}
								fullWidth
								variant="outlined"
								size="small"
								InputProps={{
									startAdornment: (
										<SearchIcon sx={{ mr: 1, color: "text.secondary" }} />
									),
								}}
							/>

							<FormControlLabel
								control={
									<Checkbox
										checked={isExact}
										onChange={(e) => setIsExact(e.target.checked)}
										color="primary"
									/>
								}
								label="Exact"
							/>

							<ActionButton
								type="submit"
								variant="contained"
								color="primary"
								disabled={loading || !query.trim()}
							>
								Search
							</ActionButton>
						</SearchContainer>
					</form>

					{error && (
						<Alert severity="error" sx={{ mb: 2 }}>
							{error}
						</Alert>
					)}

					{loading && (
						<Box display="flex" justifyContent="center" p={4}>
							<CircularProgress />
						</Box>
					)}

					{!loading && results.length > 0 && (
						<StyledTableContainer component={Paper}>
							<Table stickyHeader>
								<TableHead>
									<TableRow>
										<TableHeaderCell>Model Name</TableHeaderCell>
										<TableHeaderCell>Hub</TableHeaderCell>
										<TableHeaderCell align="center">Action</TableHeaderCell>
									</TableRow>
								</TableHead>
								<TableBody>
									{results.map((model, index) => {
										const modelKey = `${model.hub}-${model.repo_modelId}`;
										const isExpanded = expandedModel === modelKey;

										return (
											<React.Fragment
												key={`${model.hub}-${model.repo_modelId}-${index}`}
											>
												<StyledTableRow>
													<TableCell>
														<ModelLink
															href={getHubUrl(model)}
															target="_blank"
															rel="noopener noreferrer"
															onClick={(e) => {
																// Ensure external links work properly
																const url = getHubUrl(model);
																if (url && url !== "#") {
																	window.open(
																		url,
																		"_blank",
																		"noopener,noreferrer",
																	);
																	e.preventDefault();
																}
															}}
														>
															{model.repo_modelId}
														</ModelLink>
													</TableCell>
													<TableCell>
														<Typography
															variant="body2"
															sx={{ textTransform: "capitalize" }}
														>
															{model.hub}
														</Typography>
													</TableCell>
													<TableCell align="center">
														<ActionButton
															size="small"
															variant={isExpanded ? "outlined" : "contained"}
															color="primary"
															startIcon={
																isExpanded ? (
																	<ExpandLessIcon />
																) : (
																	<DownloadIcon />
																)
															}
															onClick={() => handleExpandModel(model)}
															disabled={loadingFiles && !isExpanded}
														>
															{isExpanded ? "Collapse" : "Select"}
														</ActionButton>
													</TableCell>
												</StyledTableRow>

												{/* Expandable row for file selection */}
												<TableRow>
													<TableCell
														style={{ paddingBottom: 0, paddingTop: 0 }}
														colSpan={3}
													>
														<Collapse
															in={isExpanded}
															timeout="auto"
															unmountOnExit
														>
															<Box sx={{ margin: 2 }}>
																{loadingFiles ? (
																	<Box
																		display="flex"
																		justifyContent="center"
																		p={2}
																	>
																		<CircularProgress size={24} />
																		<Typography sx={{ ml: 2 }}>
																			Loading files...
																		</Typography>
																	</Box>
																) : modelFiles.length > 0 ? (
																	<>
																		<Box
																			sx={{
																				display: "flex",
																				justifyContent: "space-between",
																				alignItems: "center",
																				mb: 2,
																			}}
																		>
																			<Typography variant="h6">
																				Select Files to Download
																			</Typography>
																			<FormControlLabel
																				control={
																					<Checkbox
																						checked={
																							selectedFiles.length ===
																							modelFiles.length
																						}
																						indeterminate={
																							selectedFiles.length > 0 &&
																							selectedFiles.length <
																								modelFiles.length
																						}
																						onChange={(e) =>
																							handleSelectAllFiles(
																								e.target.checked,
																							)
																						}
																					/>
																				}
																				label="Select All"
																			/>
																		</Box>

																		{/* File list with scrollable container - dynamically sized */}
																		<Box
																			sx={{
																				height:
																					modelFiles.length <= 10
																						? "auto"
																						: "calc(50vh - 250px)",
																				minHeight:
																					modelFiles.length <= 5
																						? "auto"
																						: "200px",
																				maxHeight: "calc(50vh - 250px)",
																				overflowY:
																					modelFiles.length <= 10
																						? "hidden"
																						: "auto",
																				border:
																					"1px solid rgba(255, 255, 255, 0.1)",
																				borderRadius: 1,
																				p: 0.5,
																				mb: 2,
																			}}
																		>
																			<List dense sx={{ py: 0 }}>
																				{modelFiles.map((file) => {
																					const fileSize =
																						file.size > 1e9
																							? `${(file.size / 1e9).toFixed(2)} GB`
																							: file.size > 1e6
																								? `${(file.size / 1e6).toFixed(2)} MB`
																								: `${(file.size / 1e3).toFixed(2)} KB`;

																					return (
																						<ListItem
																							key={file.name}
																							dense
																							sx={{
																								py: 0.25,
																								px: 1,
																								"&:hover": {
																									backgroundColor:
																										"rgba(255, 255, 255, 0.05)",
																								},
																							}}
																						>
																							<ListItemIcon
																								sx={{ minWidth: 36 }}
																							>
																								<Checkbox
																									checked={selectedFiles.includes(
																										file.name,
																									)}
																									onChange={() =>
																										handleFileSelection(
																											file.name,
																										)
																									}
																									size="small"
																									sx={{ p: 0.5 }}
																								/>
																							</ListItemIcon>
																							<ListItemText
																								primary={
																									<Box
																										sx={{
																											display: "flex",
																											justifyContent:
																												"space-between",
																											alignItems: "center",
																										}}
																									>
																										<Typography
																											variant="body2"
																											sx={{
																												fontSize: "0.875rem",
																												wordBreak: "break-word",
																												mr: 2,
																											}}
																										>
																											{file.name}
																										</Typography>
																										<Typography
																											variant="body2"
																											sx={{
																												fontSize: "0.875rem",
																												color: "text.secondary",
																												flexShrink: 0,
																											}}
																										>
																											{fileSize}
																										</Typography>
																									</Box>
																								}
																								sx={{ my: 0 }}
																							/>
																						</ListItem>
																					);
																				})}
																			</List>
																		</Box>

																		{/* Bottom action buttons */}
																		<Box
																			sx={{
																				display: "flex",
																				gap: 2,
																				justifyContent: "flex-end",
																			}}
																		>
																			<ActionButton
																				variant="outlined"
																				onClick={() => handleExpandModel(model)}
																			>
																				Cancel
																			</ActionButton>
																			<ActionButton
																				variant="contained"
																				color="primary"
																				onClick={handleDownloadSelected}
																				disabled={
																					selectedFiles.length === 0 ||
																					loadingFiles
																				}
																				startIcon={
																					loadingFiles ? (
																						<CircularProgress size={16} />
																					) : (
																						<DownloadIcon />
																					)
																				}
																			>
																				{loadingFiles
																					? "Starting Download..."
																					: `Download Selected Files (${selectedFiles.length})`}
																			</ActionButton>
																		</Box>
																	</>
																) : null}
															</Box>
														</Collapse>
													</TableCell>
												</TableRow>
											</React.Fragment>
										);
									})}
								</TableBody>
							</Table>
						</StyledTableContainer>
					)}

					{/* Show hub error warning banner */}
					{hubError && !loading && (
						<Alert
							severity="warning"
							icon={<WarningAmberIcon />}
							sx={{ mb: 2 }}
							onClose={() => setHubError(null)}
						>
							<strong>HuggingFace Hub temporarily unavailable:</strong>{" "}
							{hubError}
							{results.length > 0
								? " Showing local results only."
								: " Please try again later or search for models you've already downloaded."}
						</Alert>
					)}

					{!loading &&
						results.length === 0 &&
						query &&
						hasSearched &&
						!hubError && (
							<Typography
								variant="body1"
								color="text.secondary"
								align="center"
								sx={{ py: 4 }}
							>
								No models found for &quot;{query}&quot;
							</Typography>
						)}

					{!loading && !hasSearched && query && (
						<Typography
							variant="body1"
							color="text.secondary"
							align="center"
							sx={{ py: 4 }}
						>
							Type to search for models...
						</Typography>
					)}
				</DialogContent>

				<DialogActions sx={{ p: 2, borderTop: 1, borderColor: "divider" }}>
					<Button onClick={handleClose} color="inherit">
						Cancel
					</Button>
				</DialogActions>
			</StyledDialog>
			<AddExternalEndpointModal
				open={showExternalEndpointModal}
				onClose={() => setShowExternalEndpointModal(false)}
				onCreated={() => {
					setShowExternalEndpointModal(false);
					handleClose();
				}}
				refreshModelList={refreshModelList}
			/>
		</>
	);
};

AddModelModal.propTypes = {
	open: PropTypes.bool.isRequired,
	onClose: PropTypes.func.isRequired,
};

export default AddModelModal;
