// addModelConfigForm.js
import React from 'react';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Typography,
  Box,
  TextField,
  Alert,
  CircularProgress,
  Divider,
  Collapse,
  IconButton,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Paper,
  Tooltip,
  Chip,
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import ExpandLessIcon from '@mui/icons-material/ExpandLess';
import AddIcon from '@mui/icons-material/Add';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { useModelConfigForm } from '../../hooks/useModelConfigForm';
import {
  ConfigFieldRenderer,
  CustomConfigFieldRenderer,
  GgufDropdown,
  truncateFileName,
} from './ModelConfigFormComponents';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    backgroundColor: theme.palette.background.paper,
    borderRadius: 12,
    maxWidth: '800px',
    width: '90vw',
    maxHeight: '90vh',
  },
}));

const SectionHeader = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between',
  padding: theme.spacing(2, 0),
  cursor: 'pointer',
  '&:hover': {
    '& .section-title': {
      color: theme.palette.primary.main,
    },
  },
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  transition: 'color 0.2s',
}));

const ConfigField = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(2),
}));

const StyledTextField = styled(TextField)(({ theme }) => ({
  '& .MuiOutlinedInput-root': {
    '&:hover fieldset': {
      borderColor: theme.palette.primary.main,
    },
  },
}));

const CustomConfigRow = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  alignItems: 'flex-start',
  marginBottom: theme.spacing(2),
}));

// Helper component to render config fields in modern style
const ModernConfigField = ({ fieldKey, value, tooltip, section, handleInputChange }) => {
  const isBoolean = typeof value === 'boolean' || value === null;

  return (
    <ConfigField>
      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 0.5 }}>
        <Typography variant="body2" sx={{ fontWeight: 500 }}>
          {fieldKey}
        </Typography>
        {tooltip && (
          <Tooltip title={tooltip} placement="right">
            <InfoOutlinedIcon sx={{ fontSize: 16, color: 'text.secondary' }} />
          </Tooltip>
        )}
      </Box>
      {isBoolean ? (
        <FormControl fullWidth size="small">
          <Select
            name={fieldKey}
            value={value === null ? '' : value.toString()}
            onChange={event => handleInputChange(event, section)}
          >
            <MenuItem value="">Default</MenuItem>
            <MenuItem value="true">True</MenuItem>
            <MenuItem value="false">False</MenuItem>
          </Select>
        </FormControl>
      ) : (
        <StyledTextField
          fullWidth
          size="small"
          name={fieldKey}
          value={value || ''}
          onChange={event => handleInputChange(event, section)}
          placeholder="Enter value"
        />
      )}
    </ConfigField>
  );
};

const AddModelConfigForm = ({ closeModal, showModal, m_id, setParentModelConfigs }) => {
  const {
    // State
    modelConfigData,
    showVLLMFields,
    setShowVLLMFields,
    showLlamaCPPFields,
    setShowLlamaCPPFields,
    showMLXFields,
    setShowMLXFields,
    newConfigKey,
    setNewConfigKey,
    newConfigValue,
    setNewConfigValue,
    modelFiles,
    selectedModelFile,
    setSelectedModelFile,
    hasGgufFilesForSection,
    hasSafetensorsFilesForSection,

    // Server info
    serverOs,
    isLoadingServerOs,
    serverOsError,
    error,
    setError,

    // Derived states
    useFallbackDisplayOnDarwin,
    canShowVLLM,
    showLlamaCPPSectionForUI,
    showMLXSectionForUI,
    showGgufDropdown,
    isButtonDisabled,
    showMainFormContent,

    // Functions
    handleInputChange,
    handleAddCustomConfig,
    validateAndPreparePayload,
  } = useModelConfigForm({ showModal, m_id, isEditMode: false });

  const handleAddModelConfig = async () => {
    const modelConfigPayload = validateAndPreparePayload();
    if (!modelConfigPayload) return;

    try {
      const response = await axios.post(`${BASE_URL}/model_configs/`, modelConfigPayload);
      if (response.status === 200) {
        setParentModelConfigs(prevConfigs => [...prevConfigs, response.data]);
        closeModal();
      } else {
        setError(`Failed to add model config. Status: ${response.status}`);
      }
    } catch (err) {
      console.error('Add model config error:', err);
      setError(
        `Failed to add model config: ${err.response?.data?.detail || err.message || 'Unknown error'}`
      );
    }
  };

  if (isLoadingServerOs) {
    return (
      <StyledDialog open={showModal} onClose={closeModal}>
        <DialogContent>
          <Box sx={{ display: 'flex', justifyContent: 'center', alignItems: 'center', py: 4 }}>
            <CircularProgress />
            <Typography sx={{ ml: 2 }}>Loading server information...</Typography>
          </Box>
        </DialogContent>
      </StyledDialog>
    );
  }

  if (serverOsError) {
    return (
      <StyledDialog open={showModal} onClose={closeModal}>
        <DialogContent>
          <Alert severity="error" sx={{ mb: 2 }}>
            {serverOsError}
          </Alert>
        </DialogContent>
        <DialogActions>
          <Button onClick={closeModal}>Close</Button>
        </DialogActions>
      </StyledDialog>
    );
  }

  return (
    <StyledDialog open={showModal} onClose={closeModal}>
      <DialogTitle>
        <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
          <Typography variant="h6" component="div" sx={{ fontWeight: 600 }}>
            Add New Model Configuration
          </Typography>
          <IconButton
            aria-label="close"
            onClick={closeModal}
            sx={{
              color: theme => theme.palette.grey[500],
            }}
          >
            <CloseIcon />
          </IconButton>
        </Box>
        <Typography variant="body2" color="text.secondary" sx={{ mt: 1 }}>
          Configure deployment settings for your model. Name and Description are required.
        </Typography>
      </DialogTitle>

      <DialogContent dividers>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }} onClose={() => setError(null)}>
            {error}
          </Alert>
        )}

        {!isLoadingServerOs && !showMainFormContent && serverOs === 'darwin' && (
          <Alert severity="info" sx={{ mb: 2 }}>
            This model does not appear to have .gguf or .safetensors files suitable for
            configuration on macOS. You can add custom fields if needed, or check the model files.
          </Alert>
        )}

        {!isLoadingServerOs && !showMainFormContent && serverOs === 'linux' && (
          <Alert severity="info" sx={{ mb: 2 }}>
            This model does not appear to have files suitable for configuration on Linux (e.g. GGUF
            for Llama.cpp or compatible formats for vLLM). You can add custom fields if needed, or
            check the model files.
          </Alert>
        )}

        {showMainFormContent ? (
          <Box>
            {/* File Selection Dropdown */}
            {showGgufDropdown && (
              <Box sx={{ mb: 3 }}>
                <FormControl fullWidth>
                  <InputLabel>Model File</InputLabel>
                  <Select
                    value={selectedModelFile?.id || ''}
                    onChange={e => {
                      const file = modelFiles.find(f => f.id === e.target.value);
                      setSelectedModelFile(file);
                    }}
                    label="Model File"
                  >
                    {modelFiles.map(file => (
                      <MenuItem key={file.id} value={file.id}>
                        {truncateFileName(file.name)}
                      </MenuItem>
                    ))}
                  </Select>
                </FormControl>
              </Box>
            )}

            {/* General Section */}
            <Paper elevation={0} sx={{ p: 2, mb: 2, backgroundColor: 'rgba(255, 255, 255, 0.02)' }}>
              <Typography variant="h6" sx={{ mb: 2, color: 'primary.main' }}>
                General Settings
              </Typography>
              <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
                {Object.entries(modelConfigData.general).map(([key, value]) => (
                  <ModernConfigField
                    key={key}
                    fieldKey={key}
                    value={value}
                    tooltip={modelConfigData.tooltips?.[key]}
                    section="general"
                    handleInputChange={handleInputChange}
                  />
                ))}
              </Box>
            </Paper>

            {/* Engine-specific sections */}
            {canShowVLLM && (
              <Paper
                elevation={0}
                sx={{ p: 2, mb: 2, backgroundColor: 'rgba(255, 255, 255, 0.02)' }}
              >
                <SectionHeader onClick={() => setShowVLLMFields(!showVLLMFields)}>
                  <SectionTitle className="section-title">vLLM Configuration</SectionTitle>
                  <IconButton size="small">
                    {showVLLMFields ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                  </IconButton>
                </SectionHeader>
                <Collapse in={showVLLMFields}>
                  <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2, mt: 2 }}>
                    {Object.entries(modelConfigData.vLLM).map(([key, value]) => (
                      <ModernConfigField
                        key={key}
                        fieldKey={key}
                        value={value}
                        tooltip={modelConfigData.tooltips?.[key]}
                        section="vLLM"
                        handleInputChange={handleInputChange}
                      />
                    ))}
                  </Box>
                </Collapse>
              </Paper>
            )}
            {showLlamaCPPSectionForUI && (
              <Paper
                elevation={0}
                sx={{ p: 2, mb: 2, backgroundColor: 'rgba(255, 255, 255, 0.02)' }}
              >
                <SectionHeader onClick={() => setShowLlamaCPPFields(!showLlamaCPPFields)}>
                  <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                    <SectionTitle className="section-title">Llama.cpp Configuration</SectionTitle>
                    {serverOs === 'darwin' &&
                      !useFallbackDisplayOnDarwin &&
                      hasGgufFilesForSection && (
                        <Chip label="for .gguf files" size="small" variant="outlined" />
                      )}
                  </Box>
                  <IconButton size="small">
                    {showLlamaCPPFields ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                  </IconButton>
                </SectionHeader>
                <Collapse in={showLlamaCPPFields}>
                  <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2, mt: 2 }}>
                    {Object.entries(modelConfigData.llamaCPP).map(([key, value]) => (
                      <ModernConfigField
                        key={key}
                        fieldKey={key}
                        value={value}
                        tooltip={modelConfigData.tooltips?.[key]}
                        section="llamaCPP"
                        handleInputChange={handleInputChange}
                      />
                    ))}
                  </Box>
                </Collapse>
              </Paper>
            )}

            {showMLXSectionForUI && (
              <Paper
                elevation={0}
                sx={{ p: 2, mb: 2, backgroundColor: 'rgba(255, 255, 255, 0.02)' }}
              >
                <SectionHeader onClick={() => setShowMLXFields(!showMLXFields)}>
                  <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                    <SectionTitle className="section-title">MLX Configuration</SectionTitle>
                    {serverOs === 'darwin' &&
                      !useFallbackDisplayOnDarwin &&
                      hasSafetensorsFilesForSection && (
                        <Chip label="for .safetensors files" size="small" variant="outlined" />
                      )}
                  </Box>
                  <IconButton size="small">
                    {showMLXFields ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                  </IconButton>
                </SectionHeader>
                <Collapse in={showMLXFields}>
                  <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2, mt: 2 }}>
                    {Object.entries(modelConfigData.mlx).map(([key, value]) => (
                      <ModernConfigField
                        key={key}
                        fieldKey={key}
                        value={value}
                        tooltip={modelConfigData.tooltips?.[key]}
                        section="mlx"
                        handleInputChange={handleInputChange}
                      />
                    ))}
                  </Box>
                </Collapse>
              </Paper>
            )}
            {/* Custom Fields Section */}
            <Paper elevation={0} sx={{ p: 2, mb: 2, backgroundColor: 'rgba(255, 255, 255, 0.02)' }}>
              <Typography variant="h6" sx={{ mb: 2, color: 'primary.main' }}>
                Custom Fields
              </Typography>

              {/* Existing custom fields */}
              {Object.entries(modelConfigData.custom).length > 0 && (
                <Box sx={{ mb: 2 }}>
                  {Object.entries(modelConfigData.custom).map(([key, value]) => (
                    <CustomConfigRow key={key}>
                      <StyledTextField size="small" value={key} disabled sx={{ flex: 1 }} />
                      <StyledTextField
                        size="small"
                        name={key}
                        value={value}
                        onChange={event => handleInputChange(event, 'custom')}
                        sx={{ flex: 1 }}
                      />
                      <IconButton
                        size="small"
                        onClick={() => {
                          const newCustom = { ...modelConfigData.custom };
                          delete newCustom[key];
                          // Update custom config
                          handleInputChange(
                            { target: { name: 'custom', value: newCustom } },
                            'custom'
                          );
                        }}
                        color="error"
                      >
                        <CloseIcon />
                      </IconButton>
                    </CustomConfigRow>
                  ))}
                </Box>
              )}

              {/* Add new custom field */}
              <CustomConfigRow>
                <StyledTextField
                  size="small"
                  value={newConfigKey}
                  onChange={e => setNewConfigKey(e.target.value)}
                  placeholder="Config key"
                  sx={{ flex: 1 }}
                />
                <StyledTextField
                  size="small"
                  value={newConfigValue}
                  onChange={e => setNewConfigValue(e.target.value)}
                  placeholder="Config value"
                  sx={{ flex: 1 }}
                />
                <Button
                  variant="outlined"
                  startIcon={<AddIcon />}
                  onClick={handleAddCustomConfig}
                  disabled={!newConfigKey || !newConfigValue}
                  size="small"
                >
                  Add
                </Button>
              </CustomConfigRow>
            </Paper>
          </Box>
        ) : null}
      </DialogContent>

      <DialogActions sx={{ px: 3, pb: 3 }}>
        <Button onClick={closeModal} color="secondary">
          Cancel
        </Button>
        <Button
          variant="contained"
          onClick={handleAddModelConfig}
          disabled={isButtonDisabled}
          startIcon={<AddIcon />}
        >
          Add Model Config
        </Button>
      </DialogActions>
    </StyledDialog>
  );
};

export default AddModelConfigForm;
